# Copyright (c) 2001 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.9 $"

from Thuban import _

from types import StringTypes

import Projection
BaseProjection = Projection.Projection
del Projection

PROJ_UNITS_METERS  = 1
PROJ_UNITS_DEGREES = 2

class Projection(BaseProjection):

    """A proj4 projection object that remembers the parameters"""

    def __init__(self, params, name = None):
        """Initialize the Projection with a list of 
        'parameter=value' strings and an optional name. If 'name' is
        None, the name will be set to 'Unknown' in the local language.
        """

        BaseProjection.__init__(self, params)

        if name is None:
            self.name = _("Unknown")
        elif isinstance(name, StringTypes):
            self.name = name

        self.params = params

    def ForwardBBox(self, bbox):
        """Return the bounding box of the corners of the bounding box bbox
        """
        # This is not really the correct way to determine the bbox of a
        # projected shape, but for now it works well enough
        llx, lly, urx, ury = bbox
        xs = []; ys = []
        x, y = self.Forward(llx, lly)
        xs.append(x); ys.append(y)
        x, y = self.Forward(llx, ury)
        xs.append(x); ys.append(y)
        x, y = self.Forward(urx, ury)
        xs.append(x); ys.append(y)
        x, y = self.Forward(urx, lly)
        xs.append(x); ys.append(y)
        
        return min(xs), min(ys), max(xs), max(ys)

    def GetName(self):
        """Return the name of the projection."""
        return self.name

    def GetParameter(self, param):
        """Return the projection value for the given parameter.

        If 'param' exists as a valid parameter then the returned
        value is a string with that value. If the parameter doesn't
        exist an empty string is returned.
        """

        for pair in self.params:
            p, v = pair.split("=")
            if p == param:
                return v

        return ""

    def GetAllParameters(self):
        """Return list of 'parameter=value' strings"""
        return self.params

    def GetProjectedUnits(self):
        if self.GetParameter("proj") == "latlong":
            return PROJ_UNITS_DEGREES
        else:
            return PROJ_UNITS_METERS

    def __repr__(self):
        return self.name + ": " + repr(self.params)

class ProjFile:

    def __init__(self, filename):
        """Intialize the ProjFile.

        filename -- name of the file that this ProjFile represents.
        """

        self.__projs = []

        self.SetFilename(filename)
    
    def Add(self, proj):
        """Add the given projection to the end of the file. If 'proj'
        already exists, it is replaced in the same location as the
        original projection.
        """

        try:
            #
            # see if the projection already exists.
            # this only works if Projection doesn't override __eq__
            #
            self.__projs[self.__projs.index(proj)] = proj
        except ValueError:
            self.__projs.append(proj)

    def Remove(self, proj):
        """Remove the object proj from the projection file.

        Raises a ValueError is proj is not found.
        """

        self.__projs.remove(proj)

    def Replace(self, oldproj, newproj):
        """Replace the object 'oldproj' with 'newproj'.
        
        Raises ValueError if oldproj is not in the file.
        """

        #
        # see if the projection already exists.
        # this only works if Projection doesn't override __eq__
        #
        self.__projs[self.__projs.index(oldproj)] = newproj

    def GetFilename(self):
        """Return the filename where the ProjFile was read or will be
        written to.
        """

        return self.__filename

    def SetFilename(self, filename):
        """Set the filename where the ProjFile will be written to."""
        self.__filename = filename

    def GetProjections(self):
        """Return a list of the projections in the order they were read
        from the file or will be written.

        This is not a deep copy list, so any modifications made to the
        Projection objects will be written to the file.
        """

        return self.__projs

