# Copyright (c) 2001, 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.46 $"

from math import log, ceil
import warnings

from Thuban import _
import shapelib, shptree

from messages import LAYER_PROJECTION_CHANGED, LAYER_VISIBILITY_CHANGED, \
     LAYER_CHANGED, LAYER_SHAPESTORE_REPLACED

import classification

from color import Color
from base import TitledObject, Modifiable

import resource


class Shape:

    """Represent one shape"""

    def __init__(self, points):
        self.points = points
        #self.compute_bbox()
        self.bbox = None

    def compute_bbox(self):
        if self.bbox is not None:
            return self.bbox

        xs = []
        ys = []
        for x, y in self.points:
            xs.append(x)
            ys.append(y)
        self.llx = min(xs)
        self.lly = min(ys)
        self.urx = max(xs)
        self.ury = max(ys)

        self.bbox = (self.llx, self.lly, self.urx, self.ury)

        return self.bbox

    def Points(self):
        return self.points



# Shape type constants
SHAPETYPE_POLYGON = "polygon"
SHAPETYPE_ARC = "arc"
SHAPETYPE_POINT = "point"

# mapping from shapelib shapetype constants to our constants
shapelib_shapetypes = {shapelib.SHPT_POLYGON: SHAPETYPE_POLYGON,
                       shapelib.SHPT_ARC: SHAPETYPE_ARC,
                       shapelib.SHPT_POINT: SHAPETYPE_POINT}

shapetype_names = {SHAPETYPE_POINT: "Point",
                   SHAPETYPE_ARC: "Arc",
                   SHAPETYPE_POLYGON: "Polygon"}

class BaseLayer(TitledObject, Modifiable):

    """Base class for the layers."""

    def __init__(self, title, visible = True, projection = None):
        """Initialize the layer.

        title -- the title
        visible -- boolean. If true the layer is visible.
        """
        TitledObject.__init__(self, title)
        Modifiable.__init__(self)
        self.visible = visible
        self.projection = projection

    def Visible(self):
        """Return true if layer is visible"""
        return self.visible

    def SetVisible(self, visible):
        """Set the layer's visibility."""
        self.visible = visible
        self.issue(LAYER_VISIBILITY_CHANGED, self)

    def HasClassification(self):
        """Determine if this layer support classifications."""
        return False

    def HasShapes(self):
        """Determine if this layer supports shapes."""
        return False

    def GetProjection(self):
        """Return the layer's projection."""
        return self.projection

    def SetProjection(self, projection):
        """Set the layer's projection"""
        self.projection = projection
        self.changed(LAYER_PROJECTION_CHANGED, self)

class Layer(BaseLayer):

    """Represent the information of one geodata file (currently a shapefile)

    All children of the layer have the same type.

    A layer has fill and stroke colors. Colors should be instances of
    Color. They can also be None, indicating no fill or no stroke.

    The layer objects send the following events, all of which have the
    layer object as parameter:

        TITLE_CHANGED -- The title has changed.
        LAYER_PROJECTION_CHANGED -- the projection has changed.
    """

    def __init__(self, title, data, projection = None,
                 fill = Color.Transparent, 
                 stroke = Color.Black, 
                 lineWidth = 1, 
                 visible = True):
        """Initialize the layer.

        title -- the title
        data -- datastore object for the shape data shown by the layer
        projection -- the projection object. Its Inverse method is
               assumed to map the layer's coordinates to lat/long
               coordinates
        fill -- the fill color or Color.Transparent if the shapes are 
                not filled
        stroke -- the stroke color or Color.Transparent if the shapes 
                are not stroked
        visible -- boolean. If true the layer is visible.

        colors are expected to be instances of Color class
        """
        BaseLayer.__init__(self, title, 
                                 visible = visible,
                                 projection = projection)

        #
        # this is really important so that when the classification class
        # tries to set its parent layer the variable will exist
        #
        self.__classification = None
        self.__setClassLock = False

        self.SetShapeStore(data)

        self.SetClassification(None)

        self.__classification.SetDefaultLineColor(stroke)
        self.__classification.SetDefaultLineWidth(lineWidth)
        self.__classification.SetDefaultFill(fill)
        self.__classification.SetLayer(self)

        self.UnsetModified()

    def __getattr__(self, attr):
        """Access to some attributes for backwards compatibility

        The attributes implemented here are now held by the shapestore
        if at all. For backwards compatibility pretend that they are
        still there but issue a DeprecationWarning when they are
        accessed.
        """
        if attr in ("table", "shapetable"):
            value = self.store.Table()
        elif attr == "shapefile":
            value = self.store.Shapefile()
        elif attr == "filename":
            value = self.store.FileName()
        else:
            raise AttributeError, attr
        warnings.warn("The Layer attribute %r is deprecated."
                      " It's value can be accessed through the shapestore"
                      % attr, DeprecationWarning, stacklevel = 2)
        return value

    def SetShapeStore(self, store):
        self.store = store
        shapefile = self.store.Shapefile()

        numshapes, shapetype, mins, maxs = shapefile.info()
        self.numshapes = numshapes
        self.shapetype = shapelib_shapetypes[shapetype]

        # if there are shapes, set the bbox accordingly. Otherwise
        # set it to None.
        if self.numshapes:
            self.bbox = mins[:2] + maxs[:2]
        else:
            self.bbox = None

        # estimate a good depth for the quad tree. Each depth
        # multiplies the number of nodes by four, therefore we
        # basically take the base 4 logarithm of the number of
        # shapes.
        if self.numshapes < 4:
            maxdepth = 1
        else:
            maxdepth = int(ceil(log(self.numshapes / 4.0) / log(4)))

        self.shapetree = shptree.SHPTree(shapefile.cobject(), 2,
                                         maxdepth)
        # Set the classification to None if there is a classification
        # and the new shapestore doesn't have a table with a suitable
        # column, i.e one with the same name and type as before
        # FIXME: Maybe we should keep it the same if the type is
        # compatible enough such as FIELDTYPE_DOUBLE and FIELDTYPE_INT
        if self.__classification is not None:
            fieldname = self.__classification.GetField()
            fieldtype = self.__classification.GetFieldType()
            table = self.store.Table()
            if (fieldname is not None
                and (not table.HasColumn(fieldname)
                     or table.Column(fieldname).type != fieldtype)):
                self.SetClassification(None)
        self.changed(LAYER_SHAPESTORE_REPLACED, self)

    def ShapeStore(self):
        return self.store

    def Destroy(self):
        BaseLayer.Destroy(self)
        self.SetClassification(None)

    def BoundingBox(self):
        """Return the layer's bounding box in the intrinsic coordinate system.

        If the layer has no shapes, return None.
        """
        return self.bbox

    def LatLongBoundingBox(self):
        """Return the layer's bounding box in lat/long coordinates.

        Return None, if the layer doesn't contain any shapes.
        """
        bbox = self.BoundingBox()
        if bbox is not None:
            llx, lly, urx, ury = bbox
            if self.projection is not None:
                llx, lly = self.projection.Inverse(llx, lly)
                urx, ury = self.projection.Inverse(urx, ury)
            return llx, lly, urx, ury
        else:
            return None

    def ShapesBoundingBox(self, shapes):
        """Return a bounding box in lat/long coordinates for the given
        list of shape ids.

        If shapes is None or empty, return None.
        """

        if shapes is None or len(shapes) == 0: return None

        llx = []
        lly = []
        urx = []
        ury = []

        if self.projection is not None:
            inverse = lambda x, y: self.projection.Inverse(x, y)
        else:
            inverse = lambda x, y: (x, y)

        for id in shapes:
            left, bottom, right, top = self.Shape(id).compute_bbox()

            left, bottom = inverse(left, bottom)
            right, top   = inverse(right, top)

            llx.append(left)
            lly.append(bottom)
            urx.append(right)
            ury.append(top)

        return (min(llx), min(lly), max(urx), max(ury))

    def GetFieldType(self, fieldName):
        table = self.store.Table()
        if table.HasColumn(fieldName):
            return table.Column(fieldName).type
        return None

    def HasShapes(self):
        return True

    def NumShapes(self):
        """Return the number of shapes in the layer"""
        return self.numshapes

    def ShapeType(self):
        """Return the type of the shapes in the layer.
        This is either SHAPETYPE_POINT, SHAPETYPE_ARC or SHAPETYPE_POLYGON.
        """
        return self.shapetype

    def Shape(self, index):
        """Return the shape with index index"""
        shape = self.store.Shapefile().read_object(index)

        if self.shapetype == SHAPETYPE_POINT:
            points = shape.vertices()
        else:
            #for poly in shape.vertices():
            poly = shape.vertices()[0]
            points = []
            for x, y in poly:
                points.append((x, y))

        return Shape(points)

    def ShapesInRegion(self, box):
        """Return the ids of the shapes that overlap the box.

        Box is a tuple (left, bottom, right, top) in unprojected coordinates.
        """
        left, bottom, right, top = box

        if self.projection is not None:
            left,  bottom = self.projection.Forward(left, bottom)
            right, top    = self.projection.Forward(right, top)

        return self.shapetree.find_shapes((left, bottom), (right, top))

    def HasClassification(self):
        return True

    def GetClassification(self):
        return self.__classification

    def SetClassification(self, clazz):
        """Set the classification to 'clazz'

        If 'clazz' is None a default classification is created
        """

        # prevent infinite recursion when calling SetLayer()
        if self.__setClassLock: return

        self.__setClassLock = True

        if clazz is None:
            if self.__classification is not None:
                self.__classification.SetLayer(None)
            self.__classification = classification.Classification()
        else:
            self.__classification = clazz
            try:
                self.__classification.SetLayer(self)
            except ValueError:
                self.__setClassLock = False
                raise ValueError

        self.changed(LAYER_CHANGED, self)

        self.__setClassLock = False

    def ClassChanged(self):
        """Called from the classification object when it has changed."""
        self.changed(LAYER_CHANGED, self)
 
    def TreeInfo(self):
        items = []

        if hasattr(self, 'filename'):
            items.append(_("Filename: %s") % self.filename)

        if self.Visible():
            items.append(_("Shown"))
        else:
            items.append(_("Hidden"))
        items.append(_("Shapes: %d") % self.NumShapes())

        bbox = self.LatLongBoundingBox()
        if bbox is not None:
            items.append(_("Extent (lat-lon): (%g, %g, %g, %g)") % bbox)
        else:
            items.append(_("Extent (lat-lon):"))
        items.append(_("Shapetype: %s") % shapetype_names[self.ShapeType()])

        if self.projection and len(self.projection.params) > 0:
            items.append((_("Projection"),
                        [str(param) for param in self.projection.params]))

        items.append(self.__classification)

        return (_("Layer '%s'") % self.Title(), items)


if resource.has_gdal_support():
    import gdal
    from gdalconst import GA_ReadOnly

class RasterLayer(BaseLayer):

    def __init__(self, title, filename, projection = None, visible = True):
        """Initialize the Raster Layer.

        title -- title for the layer.

        filename -- file name of the source image.

        projection -- Projection object describing the projection which
                      the source image is in.

        visible -- True is the layer should initially be visible.

        Throws IOError if the filename is invalid or points to a file that
        is not in a format GDAL can use.
        """

        BaseLayer.__init__(self, title, visible = visible)

        self.projection = projection
        self.filename = filename

        self.bbox = -1

        if resource.has_gdal_support():
            #
            # temporarily open the file so that GDAL can test if it's valid.
            #
            dataset = gdal.Open(self.filename, GA_ReadOnly)

            if dataset is None:
                raise IOError()

        self.UnsetModified()

    def BoundingBox(self):
        """Return the layer's bounding box in the intrinsic coordinate system.

        If the layer has no shapes, return None.
        """
        if not resource.has_gdal_support():
            return None

        if self.bbox == -1:
            dataset = gdal.Open(self.filename, GA_ReadOnly)
            if dataset is None:
                self.bbox = None
            else:
                geotransform = dataset.GetGeoTransform()
                if geotransform is None:
                    return None

                x = 0
                y = dataset.RasterYSize
                left = geotransform[0] +        \
                       geotransform[1] * x +    \
                       geotransform[2] * y

                bottom = geotransform[3] +      \
                         geotransform[4] * x +  \
                         geotransform[5] * y

                x = dataset.RasterXSize
                y = 0
                right = geotransform[0] +       \
                        geotransform[1] * x +   \
                        geotransform[2] * y

                top = geotransform[3] +         \
                      geotransform[4] * x +     \
                      geotransform[5] * y

                self.bbox = (left, bottom, right, top)

        return self.bbox

    def LatLongBoundingBox(self):
        bbox = self.BoundingBox()
        if bbox is None:
            return None

        llx, lly, urx, ury = bbox
        if self.projection is not None:
            llx, lly = self.projection.Inverse(llx, lly)
            urx, ury = self.projection.Inverse(urx, ury)

        return llx, lly, urx, ury

    def GetImageFilename(self):
        return self.filename

    def TreeInfo(self):
        items = []

        if self.Visible():
            items.append(_("Shown"))
        else:
            items.append(_("Hidden"))
        items.append(_("Shapes: %d") % self.NumShapes())

        bbox = self.LatLongBoundingBox()
        if bbox is not None:
            items.append(_("Extent (lat-lon): (%g, %g, %g, %g)") % bbox)
        else:
            items.append(_("Extent (lat-lon):"))

        if self.projection and len(self.projection.params) > 0:
            items.append((_("Projection"),
                        [str(param) for param in self.projection.params]))

        return (_("Layer '%s'") % self.Title(), items)

