# Copyright (C) 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""Data source abstractions"""

__version__ = "$Revision: 1.5 $"
# $Source: /home/thuban/jail/thubanrepository/thuban/Thuban/Model/data.py,v $
# $Id: data.py,v 1.5 2003/06/20 14:14:42 jonathan Exp $

import os
import warnings
import weakref

import shapelib
import table
import transientdb

from Thuban import _


class ShapeTable(transientdb.AutoTransientTable):

    """A Table that depends on a ShapefileStore

    Intended use is by the ShapefileStore for the table associated with
    the shapefiles.
    """

    def __init__(self, store, db, table):
        """Initialize the ShapeTable.

        Parameters:
            store -- the ShapefileStore the table is to depend on
            db -- The transient database to use
            table -- the table
        """
        transientdb.AutoTransientTable.__init__(self, db, table)
        self.store = weakref.ref(store)

    def Dependencies(self):
        """Return a tuple containing the shapestore"""
        return (self.store(),)


class ShapefileStore:

    """Combine a shapefile and the corresponding DBF file into one object"""

    def __init__(self, session, filename):
        # Make the filename absolute. The filename will be
        # interpreted relative to that anyway, but when saving a
        # session we need to compare absolute paths and it's usually
        # safer to always work with absolute paths.
        self.filename = os.path.abspath(filename)

        self.shapefile = shapelib.ShapeFile(self.filename)
        self.dbftable = table.DBFTable(filename)
        self.table = ShapeTable(self, session.TransientDB(), self.dbftable)

    def Table(self):
        """Return the table containing the attribute data"""
        return self.table

    def Shapefile(self):
        """Return the shapefile object"""
        return self.shapefile

    def FileName(self):
        """Return the filename used to open the shapefile"""
        return self.filename

    def FileType(self):
        """Return the filetype. This is always the string 'shapefile'"""
        return "shapefile"

    def Dependencies(self):
        """Return the empty tuple.

        The ShapefileStore doesn't depend on anything else.
        """
        return ()

    def OrigShapeStore(self):
        """Return None.

        The ShapefileStore was not derived from another shapestore.
        """
        return None


class DerivedShapeStore:

    """A ShapeStore derived from other shapestores or tables"""

    def __init__(self, shapestore, table):
        """Initialize the derived shapestore.

        The arguments are a shapestore for the shapedata and a table for
        the tabular data.

        Raises ValueError if the number of shapes in the shapestore
        is different from the number of rows in the table.
        """

        numShapes = shapestore.Shapefile().info()[0]
        if numShapes != table.NumRows():
            raise ValueError(_("Table not compatible with shapestore."))

        self.shapestore = shapestore
        self.table = table

    def Table(self):
        """Return the table"""
        return self.table

    def Shapefile(self):
        """Return the shapefile of the underlying shapestore"""
        return self.shapestore.Shapefile()

    def Dependencies(self):
        """Return a tuple containing the shapestore and the table"""
        return (self.shapestore, self.table)

    def OrigShapeStore(self):
        """
        Return the original shapestore the derived store was instantiated with
        """
        return self.shapestore


class SimpleStore:

    """Combine a shapefile and a table object"""

    def __init__(self, shapefile, table):
        warnings.warn("The SimpleStore is deprecated."
                      " Use DerivedShapeStore instead",
                      DeprecationWarning, stacklevel = 2)
        self.shapefile = shapefile
        self.table = table
        self.filename = None

    def Table(self):
        return self.table

    def Shapefile(self):
        return self.shapefile
