# Copyright (c) 2001, 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
# Jonathan Coles <jonathan@intevation.de>
# Frank Koormann <frank.koormann@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.32 $"

import cStringIO

from Thuban import _

from wxPython.wx import wxMemoryDC, wxEmptyBitmap, \
    wxPoint, wxRect, wxPen, wxBrush, wxFont, \
    wxTRANSPARENT_PEN, wxTRANSPARENT_BRUSH, \
    wxBLACK_PEN, wxRED_PEN, wxBLACK, \
    wxSOLID, wxCROSS_HATCH, wxSWISS, wxNORMAL, \
    wxBitmap, wxImageFromBitmap, wxBitmapFromImage, \
    wxImageFromStream, wxBITMAP_TYPE_BMP

from wxproj import draw_polygon_shape, draw_polygon_init

from Thuban.UI.common import Color2wxColour
from Thuban.UI.classifier import ClassDataPreviewer
from Thuban.UI.scalebar import ScaleBar

from Thuban.Model.layer import Layer, RasterLayer, \
     SHAPETYPE_POLYGON, SHAPETYPE_ARC, SHAPETYPE_POINT
from Thuban.Model.label import ALIGN_CENTER, ALIGN_TOP, ALIGN_BOTTOM, \
     ALIGN_LEFT, ALIGN_RIGHT, ALIGN_BASELINE

from Thuban.Model.classification import Classification
from Thuban.Model.color import Color
import Thuban.Model.resource

if Thuban.Model.resource.has_gdal_support():
    from gdalwarp import ProjectRasterFile

class MapRenderer:

    """Class to render a map onto a wxDC"""

    honor_visibility = 1

    def __init__(self, dc, scale, offset, resolution = 72.0,
                 honor_visibility = None):
        """Inititalize the renderer.

        dc -- the wxPython DC to render on
        scale, offset -- the scale factor and translation to convert
                between projected coordinates and the DC coordinates

        resolution -- the assumed resolution of the DC. Used to convert
                absolute lengths like font sizes to DC coordinates

        honor_visibility -- boolean. If true, honor the visibility flag
                of the layers, otherwise draw all layers. If None, use
                the renderer's default.
        """
        # resolution in pixel/inch

        self.dc = dc
        self.scale = scale
        self.offset = offset
        if honor_visibility is not None:
            self.honor_visibility = honor_visibility
        # store the resolution in pixel/point because it's more useful
        # later.
        self.resolution = resolution / 72.0

    def render_map(self, map):
        self.map = map
        seenRaster = True

        if self.scale == 0:
            return

        #
        # This is only a good optimization if there is only one
        # raster layer and the image covers the entire window (as
        # it currently does). We note if there is a raster layer
        # and only begin drawing layers once we have drawn it. 
        # That way we avoid drawing layers that won't be seen.
        #
        if Thuban.Model.resource.has_gdal_support():
            for layer in map.Layers():
                if isinstance(layer, RasterLayer) and layer.Visible():
                    seenRaster = False
                    break

        for layer in map.Layers():
            # if honor_visibility is true, only draw visible layers,
            # otherwise draw all layers
            if not self.honor_visibility or layer.Visible():
                if isinstance(layer, Layer) and seenRaster:
                    self.draw_shape_layer(layer)
                elif isinstance(layer, RasterLayer) \
                    and Thuban.Model.resource.has_gdal_support():
                    self.draw_raster_layer(layer)
                    seenRaster = True

        self.draw_label_layer(map.LabelLayer())

    def draw_shape_layer(self, layer):
        scale = self.scale
        offx, offy = self.offset

        map_proj = self.map.projection
        layer_proj = layer.projection

        shapetype = layer.ShapeType()

        brush = wxTRANSPARENT_BRUSH
        pen   = wxTRANSPARENT_PEN

        old_prop = None
        old_group = None
        lc = layer.GetClassification()
        field = lc.GetField()
        defaultGroup = lc.GetDefaultGroup()


        if shapetype != SHAPETYPE_POINT:
            polygon_render_param = self.polygon_render_param(layer)

        if shapetype == SHAPETYPE_POINT:
            draw_func = lambda i: \
                   self.draw_point_shape(layer, i)
        else:
            draw_func = lambda i: \
                   self.draw_polygon_shape(polygon_render_param, i, pen, brush)

        table = layer.ShapeStore().Table()
        for i in self.layer_ids(layer):

            if field is None:
                group = defaultGroup
            else:
                record = table.ReadRowAsDict(i)
                assert record is not None
                group = lc.FindGroup(record[field])


            if not group.IsVisible():
                continue


            # don't recreate new objects if they are the same as before
            if group is not old_group:
                old_group = group

                prop = group.GetProperties()

                if prop != old_prop:
                    old_prop = prop

                    if shapetype == SHAPETYPE_ARC:
                        fill = Color.Transparent
                    else:
                        fill = prop.GetFill()


                    if fill is Color.Transparent:
                        brush = wxTRANSPARENT_BRUSH
                    else:
                        color = Color2wxColour(fill)
                        brush = wxBrush(color, wxSOLID)

                    stroke = prop.GetLineColor()
                    stroke_width = prop.GetLineWidth()
                    if stroke is Color.Transparent:
                        pen = wxTRANSPARENT_PEN
                    else:
                        color = Color2wxColour(stroke)
                        pen = wxPen(color, stroke_width, wxSOLID)

                    if shapetype == SHAPETYPE_POINT:
                        self.dc.SetBrush(brush)
                        self.dc.SetPen(pen)

            draw_func(i)

    def draw_raster_layer(self, layer):
        data = None
        offx, offy = self.offset
        width, height = self.dc.GetSizeTuple()

        inProj = ""
        proj = layer.GetProjection()
        if proj is not None:
            for p in proj.GetAllParameters():
                inProj += "+" + p + " "

        outProj = ""
        proj = self.map.GetProjection()
        if proj is not None:
            for p in proj.GetAllParameters():
                outProj += "+" + p + " "

        xmin = (0 - offx) / self.scale
        ymin = (offy - height) / self.scale
        xmax = (width - offx) / self.scale
        ymax = (offy - 0) / self.scale

        try:
            data = ProjectRasterFile(
                layer.GetImageFilename(),
                inProj,
                outProj,
                (xmin, ymin, xmax, ymax), 
                "", (width, height))
        except IOError, (strerr):
            print strerr
        except (AttributeError, ValueError):
            pass
        else:
            if data is not None:
                stream = cStringIO.StringIO(data)
                image = wxImageFromStream(stream, wxBITMAP_TYPE_BMP)
                bitmap = wxBitmapFromImage(image)
                self.dc.BeginDrawing()
                self.dc.DrawBitmap(bitmap, 0, 0)
                self.dc.EndDrawing()

    def layer_ids(self, layer):
        """Return the shape ids of the given layer that have to be drawn.

        The default implementation simply returns all ids in the layer.
        Override in derived classes to be more precise.
        """
        return range(layer.NumShapes())

    def polygon_render_param(self, layer):
        """Return the low-lever render parameter for the layer"""
        offx, offy = self.offset
        return draw_polygon_init(layer.ShapeStore().Shapefile(), self.dc,
                                 self.map.projection,
                                 layer.projection,
                                 self.scale, -self.scale,
                                 offx, offy)

    def draw_polygon_shape(self, draw_polygon_info, index, pen, brush):
        draw_polygon_shape(draw_polygon_info, index, pen, brush)

    def projected_points(self, layer, index):
        proj = self.map.GetProjection()
        if proj is not None:
            forward = proj.Forward
        else:
            forward = None
        proj = layer.GetProjection()
        if proj is not None:
            inverse = proj.Inverse
        else:
            inverse = None
        shape = layer.Shape(index)
        points = []
        scale = self.scale
        offx, offy = self.offset
        for x, y in shape.Points():
            if inverse:
                x, y = inverse(x, y)
            if forward:
                x, y = forward(x, y)
            points.append(wxPoint(x * scale + offx,
                                  -y * scale + offy))
        return points

    def draw_arc_shape(self, layer, index):
        points = self.projected_points(layer, index)
        self.dc.DrawLines(points)

    def draw_point_shape(self, layer, index):
        pp = self.projected_points(layer, index)

        if len(pp) == 0: return # ignore Null Shapes which have no points

        p = pp[0]
        radius = self.resolution * 5
        self.dc.DrawEllipse(p.x - radius, p.y - radius, 2*radius, 2*radius)

    def draw_label_layer(self, layer):
        scale = self.scale
        offx, offy = self.offset

        font = wxFont(self.resolution * 10, wxSWISS, wxNORMAL, wxNORMAL)
        self.dc.SetFont(font)

        map_proj = self.map.projection
        if map_proj is not None:
            forward = map_proj.Forward
        else:
            forward = None

        for label in layer.Labels():
            x = label.x
            y = label.y
            text = label.text
            if forward:
                x, y = forward(x, y)
            x = x * scale + offx
            y = -y * scale + offy
            width, height = self.dc.GetTextExtent(text)
            if label.halign == ALIGN_LEFT:
                # nothing to be done
                pass
            elif label.halign == ALIGN_RIGHT:
                x = x - width
            elif label.halign == ALIGN_CENTER:
                x = x - width/2
            if label.valign == ALIGN_TOP:
                # nothing to be done
                pass
            elif label.valign == ALIGN_BOTTOM:
                y = y - height
            elif label.valign == ALIGN_CENTER:
                y = y - height/2
            self.dc.DrawText(text, x, y)


class ScreenRenderer(MapRenderer):

    # On the screen we want to see only visible layers by default
    honor_visibility = 1
    
    def RenderMap(self, map, region, selected_layer, selected_shapes):
        """Render the map.

        Only the given region (a tuple in window coordinates as returned
        by a wxrect's asTuple method) needs to be redrawn. Highlight the
        shapes given by the ids in selected_shapes in the
        selected_layer.
        """
        self.update_region = region
        self.selected_layer = selected_layer
        self.selected_shapes = selected_shapes
        self.render_map(map)

    def draw_shape_layer(self, layer):
        MapRenderer.draw_shape_layer(self, layer)
        if layer is self.selected_layer and self.selected_shapes:
            pen = wxPen(wxBLACK, 3, wxSOLID)
            brush = wxBrush(wxBLACK, wxCROSS_HATCH)

            shapetype = layer.ShapeType()
            if shapetype == SHAPETYPE_POLYGON:
                offx, offy = self.offset
                renderparam = self.polygon_render_param(layer)
                func = self.draw_polygon_shape
                args = (pen, brush)
            elif shapetype == SHAPETYPE_ARC:
                renderparam = self.polygon_render_param(layer)
                func = self.draw_polygon_shape
                args = (pen, None)
            elif shapetype == SHAPETYPE_POINT:
                renderparam = layer
                self.dc.SetBrush(brush)
                self.dc.SetPen(pen)
                func = self.draw_point_shape
                args = ()
            else:
                raise TypeError(_("Unhandled shape type %s") % shapetype)

            for index in self.selected_shapes:
                func(renderparam, index, *args)


    def layer_ids(self, layer):
        """Return the shapeids covered by the region that has to be redrawn

        Call the layer's ShapesInRegion method to determine the ids so
        that it can use the quadtree.
        """
        # FIXME: the quad-tree should be built from the projected
        # coordinates not the lat-long ones because it's not trivial to
        # determine an appropriate rectangle in lat-long for a given
        # rectangle in projected coordinates which we have to start from
        # here.
        proj = self.map.projection
        if proj is not None:
            inverse = proj.Inverse
        else:
            inverse = None

        scale = self.scale
        offx, offy = self.offset
        xs = []
        ys = []
        x, y, width, height = self.update_region
        for winx, winy in ((x, y), (x + width, y),
                           (x + width, y + height), (x, y + height)):
            px = (winx - offx) / scale
            py = -(winy - offy) / scale
            if inverse:
                px, py = inverse(px, py)
            xs.append(px)
            ys.append(py)
        left = min(xs)
        right = max(xs)
        top = max(ys)
        bottom = min(ys)

        return layer.ShapesInRegion((left, bottom, right, top))


class ExportRenderer(ScreenRenderer):

    honor_visibility = 1
    
    def RenderMap(self, map, region, mapregion, 
                  selected_layer, selected_shapes ):
        """Render the map.

        The rendering device has been specified during initialisation.
        The device border distance was set in Thuban.UI.view.OutputTranform().
        
        RenderMap renders a frame set (one page frame, one around
        legend/scalebar and one around the map), the map, the legend and the 
        scalebar on the given DC. The map is rendered with the region displayed
        in the canvas view, centered on the area available for map display.
        """
        
        self.update_region = region
        self.selected_layer = selected_layer
        self.selected_shapes = selected_shapes

        # Get some dimensions 
        llx, lly, urx, ury = region
        self.mapregion = mapregion
        mminx, mminy, mmaxx, mmaxy = self.mapregion

        # Manipulate the offset to position the map 
        offx, offy = self.offset
        # 1. Shift to corner of map drawing area
        offx = offx + mminx 
        offy = offy + mminy

        # 2. Center the map on the map drawing area:
        # region identifies the region on the canvas view: 
        # center of map drawing area - half the size of region: rendering origin
        self.shiftx = (mmaxx - mminx)*0.5 - (urx - llx)*0.5
        self.shifty = (mmaxy - mminy)*0.5 - (ury - lly)*0.5

        self.offset = (offx+self.shiftx, offy+self.shifty)

        # Draw the map
        self.dc.BeginDrawing()
        self.dc.DestroyClippingRegion()
        self.dc.SetClippingRegion(mminx+self.shiftx, mminy+self.shifty, 
                                  urx, ury)
        self.render_map(map)
        self.dc.EndDrawing()

        # Draw the rest (frames, legend, scalebar)
        self.dc.BeginDrawing()
        self.dc.DestroyClippingRegion()

        # Force the font for Legend drawing
        font = wxFont(self.resolution * 10, wxSWISS, wxNORMAL, wxNORMAL)
        self.dc.SetFont(font)

        self.render_frame(region)
        self.render_legend(map)
        self.render_scalebar(map)
        self.dc.EndDrawing()

    def render_frame(self, region):
        """Render the frames for map and legend/scalebar."""

        dc = self.dc
        dc.SetPen(wxBLACK_PEN)
        dc.SetBrush(wxTRANSPARENT_BRUSH)

        # Dimension stuff
        width, height = dc.GetSizeTuple()
        mminx, mminy, mmaxx, mmaxy = self.mapregion

        # Page Frame
        dc.DrawRectangle(15,15,width-30, (mmaxy-mminy)+10)
        
        # Map Frame
        llx, lly, urx, ury = region
        dc.DrawRectangle(mminx + self.shiftx, mminy + self.shifty, urx, ury)
        
        # Legend Frame
        dc.DrawRectangle(mmaxx+10,mminy,(width-20) - (mmaxx+10), mmaxy-mminy)

        dc.DestroyClippingRegion()
        dc.SetClippingRegion(mmaxx+10,mminy,
                             (width-20) - (mmaxx+10), mmaxy-mminy)

    def render_legend(self, map):
        """Render the legend on the Map."""

        previewer = ClassDataPreviewer()
        dc = self.dc
        dc.SetPen(wxBLACK_PEN)
        dc.SetBrush(wxTRANSPARENT_BRUSH)

        # Dimension stuff
        width, height = dc.GetSizeTuple()
        mminx, mminy, mmaxx, mmaxy = self.mapregion
        textwidth, textheight = dc.GetTextExtent("0")
        iconwidth  = textheight
        iconheight = textheight
        stepy = textheight+3
        dx = 10
        posx = mmaxx + 10 + 5   # 10 pix distance mapframe/legend frame, 
                                # 5 pix inside legend frame
        posy = mminy + 5        # 5 pix inside legend frame
        
        # Render the legend
        dc.SetTextForeground(wxBLACK)
        if map.HasLayers():
            for l in map.Layers():
                if l.Visible():
                    # Render title
                    dc.DrawText(l.Title(), posx, posy)
                    posy+=stepy
                    # Render classification
                    clazz = l.GetClassification()
                    shapeType = l.ShapeType()
                    for g in clazz:
                        if g.IsVisible():
                            previewer.Draw(dc, 
                                wxRect(posx+dx, posy, iconwidth, iconheight), 
                                g.GetProperties(), shapeType)
                            dc.DrawText(g.GetDisplayText(), 
                                        posx+2*dx+iconwidth, posy)
                            posy+=stepy
        
    def render_scalebar(self, map):
        """Render the scalebar."""

        scalebar = ScaleBar(map)

        # Dimension stuff
        width, height = self.dc.GetSizeTuple()
        mminx, mminy, mmaxx, mmaxy = self.mapregion
       
        # Render the scalebar
        scalebar.DrawScaleBar(self.scale, self.dc, 
                             (mmaxx+10+5, mmaxy-25),
                             ((width-15-5) - (mmaxx+10+5),20)
                            )
        # 10 pix between map and legend frame, 5 pix inside legend frame
        # 25 pix from the legend frame bottom line
        # Width: 15 pix from DC border, 5 pix inside frame, 10, 5 as above
        # Height: 20

class PrinterRenderer(ExportRenderer):

    # Printing as well as Export / Screen display only the visible layer.
    honor_visibility = 1

