# Copyright (c) 2003 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Functions to generate Classifications
"""

__version__ = "$Revision: 1.9 $"
# $Source: /home/thuban/jail/thubanrepository/thuban/Thuban/Model/classgen.py,v $
# $Id: classgen.py,v 1.9 2003/06/12 12:39:54 jonathan Exp $

import operator

from color import Color
from range import Range
from classification import Classification, ClassGroupSingleton, \
    ClassGroupRange, ClassGroupProperties

def generate_singletons(_list, numGroups, ramp):
    """Generate a new classification consisting solely of singletons.

    The resulting classification will consist of at most 'numGroups'
    groups whose group properties ramp between 'prop1' and 'prop2'. There
    could be fewer groups if '_list' contains fewer that 'numGroups' items.

    _list -- any object that implements the iterator interface

    numGroups -- how many groups to generate. This can not be
                 determined while the classification is being
                 generated because the stepping values must
                 be precalculated to ramp between prop1 and prop2.

    ramp -- an object which implements the CustomRamp interface
    """

    clazz = Classification()
    if numGroups == 0: return clazz

    ramp.SetNumGroups(numGroups)

    for value, prop in zip(_list, ramp):
        clazz.AppendGroup(ClassGroupSingleton(value, prop))

    return clazz

def generate_uniform_distribution(min, max, numGroups, ramp, intStep = False):
    """Generate a classification with numGroups range groups
    each with the same interval.

    intStep -- force the calculated stepping to an integer.
               Useful if the values are integers but the
               number of groups specified doesn't evenly
               divide (max - min).
    """

    clazz = Classification()
    if numGroups == 0: return clazz

    ramp.SetNumGroups(numGroups)

    cur_min = min

    i = 1
    end = "["
    for prop in ramp:

        if intStep:
            cur_max = min + int(round((i * (max - min + 1)) / float(numGroups)))
        else:
            cur_max = min + (i * (max - min)) / float(numGroups)

        if i == numGroups:
            cur_max = max
            end = "]"

        if cur_min == cur_max:
            range = Range(("[", cur_min, cur_max, "]"))
        else:
            range = Range(("[", cur_min, cur_max, end))

        clazz.AppendGroup(ClassGroupRange(range, None, prop))

        cur_min = cur_max
        i += 1

    return clazz


def generate_quantiles(_list, percents, ramp, _range):
    """Generates a Classification which has groups of ranges that
    represent quantiles of _list at the percentages given in percents.
    Only the values that fall within _range are considered. 

    Returns a tuple (adjusted, Classification) where adjusted is
    True if the Classification does not exactly represent the given
    range, or if the Classification is empty.

    _list -- a sort list of values

    percents -- a sorted list of floats in the range 0.0-1.0 which
                represent the upper bound of each quantile. the
                union of all percentiles should be the entire 
                range from 0.0-1.0

    ramp -- an object which implements the CustomRamp interface

    _range -- a Range object

    Raises a Value Error if 'percents' has fewer than two items, or
    does not cover the entire range.
    """

    clazz = Classification()
    quantiles = calculate_quantiles(_list, percents, _range)
    adjusted = True

    if quantiles is not None:

        numGroups = len(quantiles[3])

        if numGroups != 0:

            adjusted = quantiles[0]

            ramp.SetNumGroups(numGroups)

            start, min, endMax, right = _range.GetRange()

            oldp = 0
            i = 1
            end = "]"

            for (q, p), prop in zip(quantiles[3], ramp): 
                if i == numGroups:
                    max = endMax
                    end = right
                else:
                    max = _list[q]

                group = ClassGroupRange(Range((start, min, max, end)),
                                        None, prop)
    
                group.SetLabel("%s%% - %s%%" % (round(oldp*100, 2), 
                                                round(p*100, 2)))
                oldp = p
                start = "]"
                min = max
                clazz.AppendGroup(group)
                i += 1

    return (adjusted, clazz)

def GenQuantiles0(_list, percents, ramp, _range):
    """Same as GenQuantiles, but the first class won't be added to
    the classification.

    Returns a tuple (adjusted, Classification, upper_class0) where
    upper_class0 is the highest value inside the first class.

    _list -- a sort list of values

    percents -- a sorted list of floats in the range 0.0-1.0 which
                represent the upper bound of each quantile. the
                union of all percentiles should be the entire 
                range from 0.0-1.0

    ramp -- an object which implements the CustomRamp interface

    _range -- a Range object

    Raises a Value Error if 'percents' has fewer than two items, or
    does not cover the entire range.
    """

    clazz = Classification()
    quantiles = calculate_quantiles(_list, percents, _range)
    adjusted = True

    if quantiles is not None:

        numGroups = len(quantiles[3]) - 1

        if numGroups > 0:
            adjusted = quantiles[0]

            ramp.SetNumGroups(numGroups)

            start, min, endMax, right = _range.GetRange()

            class0 = quantiles[3][0]
            min = _list[class0[0]]
            oldp = class0[1]
            i = 1
            end = "]"

            for (q, p), prop in zip(quantiles[3][1:], ramp): 
                if i == numGroups:
                    max = endMax
                    end = right
                else:
                    max = _list[q]

                group = ClassGroupRange(Range((start, min, max, end)),
                                        None, prop)
    
                group.SetLabel("%s%% - %s%%" % (round(oldp*100, 2), 
                                                round(p*100, 2)))
                oldp = p
                start = "]"
                min = max
                clazz.AppendGroup(group)
                i += 1

    return (adjusted, clazz, _list[class0[0]])


def calculate_quantiles(_list, percents, _range):
    """Calculate quantiles for the given _list of percents from the
    sorted list of values that are in range.
                                                                            
    This may not actually generate len(percents) quantiles if
    many of the values that fall on quantile borders are the same.

    Returns a tuple of the form: 
        (adjusted, minIndex, maxIndex, [quantile_list])

    where adjusted is True if the the quantile percentages differ from
    those supplied, minIndex is the index into _list where the 
    minimum value used is located, maxIndex is the index into _list
    where the maximum value used is located, and quantile_list is a 
    list of tuples of the form: (list_index, quantile_percentage)

    Returns None, if no quantiles could be generated based on the
    given range or input list.

    _list -- a sort list of values

    percents -- a sorted list of floats in the range 0.0-1.0 which
                represent the upper bound of each quantile. the
                union of all percentiles should be the entire 
                range from 0.0-1.0

    _range -- a Range object

    Raises a Value Error if 'percents' has fewer than two items, or
    does not cover the entire range.
    """

    quantiles = []
    adjusted = False

    if len(percents) <= 1:
        raise ValueError("percents parameter must have more than one item")

    if percents[len(percents) - 1] != 1.0:
        raise ValueError("percents does not cover the entire range")

    #
    # find what part of the _list range covers
    #
    minIndex = -1
    maxIndex = -2
    for i in xrange(0, len(_list), 1):
        if operator.contains(_range, _list[i]):
            minIndex = i
            break

    for i in xrange(len(_list)-1, -1, -1):
        if operator.contains(_range, _list[i]):
            maxIndex = i
            break

    numValues = maxIndex - minIndex + 1

    if numValues > 0:

        #
        # build a list of unique indices into list of where each
        # quantile *should* be. set adjusted if the resulting
        # indices are different
        #
        quantiles = {}
        for p in percents:
            index = min(minIndex + int(p*numValues)-1, maxIndex)

            adjusted = adjusted \
                or quantiles.has_key(index) \
                or ((index - minIndex + 1) / float(numValues)) != p

            quantiles[index] = 0

        quantiles = quantiles.keys()
        quantiles.sort()

        #
        # the current quantile index must be strictly greater than
        # the lowerBound
        #
        lowerBound = minIndex - 1

        for qindex in xrange(len(quantiles)):
            if lowerBound >= maxIndex:
                # discard higher quantiles
                quantiles = quantiles[:qindex]
                break

            # lowerBound + 1 is always a valid index

            #
            # bump up the current quantile index to be a usable index
            # if it currently falls below the lowerBound
            #
            if quantiles[qindex] <= lowerBound:
                quantiles[qindex] = lowerBound + 1

            listIndex = quantiles[qindex]
            value = _list[listIndex]

            #
            # look for similar values around the quantile index
            #
            lindex = listIndex - 1
            while lindex > lowerBound and value == _list[lindex]:
                lindex -= 1
            lcount = (listIndex - 1) - lindex

            rindex = listIndex + 1
            while rindex < maxIndex + 1 and value == _list[rindex]:
                rindex += 1
            rcount = (listIndex + 1) - rindex

            #
            # adjust the current quantile index based on how many 
            # numbers in the _list are the same as the current value
            #
            newIndex = listIndex
            if lcount == rcount:
                if lcount != 0:
                    #
                    # there are an equal number of numbers to the left
                    # and right, try going to the left first unless
                    # doing so creates an empty quantile.
                    #
                    if lindex != lowerBound:
                        newIndex = lindex
                    else:
                        newIndex = rindex - 1

            elif lcount < rcount:
                # there are fewer items to the left, so 
                # try going to the left first unless
                # doing so creates an empty quantile.
                if lindex != lowerBound:
                    newIndex = lindex
                else:
                    newIndex = rindex - 1

            elif rcount < lcount:
                # there are fewer items to the right, so go to the right
                newIndex = rindex - 1

            adjusted = adjusted or newIndex != listIndex

            quantiles[qindex] = newIndex
            lowerBound = quantiles[qindex]

    if len(quantiles) == 0:
        return None
    else:
        return (adjusted, minIndex, maxIndex,
                [(q, (q - minIndex+1) / float(numValues)) \
                 for q in quantiles])

CLR  = 0
STEP = 1
class CustomRamp:

    def __init__(self, prop1, prop2):
        self.prop1 = prop1
        self.prop2 = prop2

        self.count = 0

    def __iter__(self):
        return self

    def GetRamp(self):
        return self

    def SetNumGroups(self, num):

        if num <= 0:
            return False

        self.count = int(num)
        num = float(num)

        prop1 = self.prop1
        prop2 = self.prop2

        clr = prop1.GetLineColor()
        lineColor2 = prop2.GetLineColor()
        
        self.noLine = clr is not Color.Transparent \
                        and lineColor2 is not Color.Transparent


        self.lineInfo = self.__GetColorInfo(prop1.GetLineColor(), 
                                            prop2.GetLineColor(),
                                            num)

        self.fillInfo = self.__GetColorInfo(prop1.GetFill(), 
                                            prop2.GetFill(),
                                            num)

        self.lineWidth = prop1.GetLineWidth()
        self.lineWidthStep = (prop2.GetLineWidth() - self.lineWidth) / num

        return True

    def next(self):
        if self.count == 0:
            raise StopIteration

        prop = ClassGroupProperties()

        if self.lineInfo is None:
            prop.SetLineColor(Color.Transparent)
        else:
            prop.SetLineColor(Color(self.lineInfo[CLR][0] / 255,
                                    self.lineInfo[CLR][1] / 255, 
                                    self.lineInfo[CLR][2] / 255))

            self.lineInfo[CLR][0] += self.lineInfo[STEP][0]
            self.lineInfo[CLR][1] += self.lineInfo[STEP][1]
            self.lineInfo[CLR][2] += self.lineInfo[STEP][2]

        if self.fillInfo is None:
            prop.SetFill(Color.Transparent)
        else:
            prop.SetFill(Color(self.fillInfo[CLR][0] / 255, 
                            self.fillInfo[CLR][1] / 255, 
                            self.fillInfo[CLR][2] / 255))

            self.fillInfo[CLR][0] += self.fillInfo[STEP][0]
            self.fillInfo[CLR][1] += self.fillInfo[STEP][1]
            self.fillInfo[CLR][2] += self.fillInfo[STEP][2]


        prop.SetLineWidth(int(self.lineWidth))
        self.lineWidth        += self.lineWidthStep

        self.count -= 1

        return prop

    def __GetColorInfo(self, color1, color2, numGroups):

        if color1 is Color.Transparent and color2 is Color.Transparent:
            #
            # returning early
            #
            return None
        elif color1 is not Color.Transparent and color2 is Color.Transparent:
            color = [color1.red   * 255, 
                     color1.green * 255, 
                     color1.blue  * 255]
            step = (0, 0, 0)
        elif color1 is Color.Transparent and color2 is not Color.Transparent:
            color = [color2.red   * 255, 
                     color2.green * 255, 
                     color2.blue  * 255]
            step = (0, 0, 0)
        else:
            color = [color1.red   * 255, 
                     color1.green * 255, 
                     color1.blue  * 255]
            step = ((color2.red   * 255 - color1.red   * 255) / numGroups,
                    (color2.green * 255 - color1.green * 255) / numGroups,
                    (color2.blue  * 255 - color1.blue  * 255) / numGroups)


        return (color, step)

class MonochromaticRamp(CustomRamp):
    def __init__(self, start, end):
        sp = ClassGroupProperties()
        sp.SetLineColor(start)
        sp.SetFill(start)

        ep = ClassGroupProperties()
        ep.SetLineColor(end)
        ep.SetFill(end)

        CustomRamp.__init__(self, sp, ep)

class GreyRamp(MonochromaticRamp):
    def __init__(self):
        MonochromaticRamp.__init__(self, Color(1, 1, 1), Color(0, 0, 0))

class RedRamp(MonochromaticRamp):
    def __init__(self):
        MonochromaticRamp.__init__(self, Color(1, 1, 1), Color(.8, 0, 0))

class GreenRamp(MonochromaticRamp):
    def __init__(self):
        MonochromaticRamp.__init__(self, Color(1, 1, 1), Color(0, .8, 0))

class BlueRamp(MonochromaticRamp):
    def __init__(self):
        MonochromaticRamp.__init__(self, Color(1, 1, 1), Color(0, 0, .8))

class GreenToRedRamp(MonochromaticRamp):
    def __init__(self):
        MonochromaticRamp.__init__(self, Color(0, .8, 0), Color(1, 0, 0))

class HotToColdRamp:

    def __iter__(self):
        return self
        
    def GetRamp(self):
        return self

    def SetNumGroups(self, num):
        if num < 0:
            return False

        self.num = float(num)
        self.index = 0

        return True

    def next(self):
        if self.index == self.num:
            raise StopIteration

        clr = [1.0, 1.0, 1.0]

        if self.index < (.25 * self.num):
            clr[0] = 0
            clr[1] = 4 * self.index / self.num
        elif self.index < (.5 * self.num):
            clr[0] = 0
            clr[2] = 1 + 4 * (.25 * self.num - self.index) / self.num
        elif self.index < (.75 * self.num):
            clr[0] = 4 * (self.index - .5 * self.num) / self.num
            clr[2] = 0
        else:
            clr[1] = 1 + 4 * (.75 * self.num - self.index) / self.num
            clr[2] = 0

        self.index += 1

        prop = ClassGroupProperties()
        prop.SetLineColor(Color(clr[0], clr[1], clr[2]))
        prop.SetFill(Color(clr[0], clr[1], clr[2]))

        return prop
