# Copyright (c) 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test the Table class
"""

__version__ = "$Revision: 1.1 $"
# $Source: /thubanrepository/thuban/test/test_table.py,v $
# $Id: test_table.py,v 1.1 2002/08/30 09:44:12 bh Exp $

import os
import unittest

import support
support.initthuban()

from Thuban.Model.table import Table
import dbflib

class TableTest(unittest.TestCase, support.FileTestMixin):

    def test_read(self):
        """Reading from DBF tables"""
        eq = self.assertEquals
        table = Table(os.path.join("..", "Data", "iceland", "political.dbf"))
        eq(table.record_count(), 156)
        eq(table.field_count(), 8)

        # Check one each of the possible field types
        eq(table.field_info(0), ('double', 'AREA', 12, 3))
        eq(table.field_info(3), ('int', 'PONET_ID', 11, 0))
        eq(table.field_info(6), ('string', 'POPYCOUN', 2, 0))

        # Read an `interesting` record
        eq(table.read_record(144),
           {'POPYCOUN': 'IC', 'POPYADMIN': '', 'PONET_': 146, 'AREA': 19.462,
            'POPYTYPE': 1, 'PERIMETER': 88.518000000000001, 'POPYREG': '1',
            'PONET_ID': 145})

        table.Destroy()

    def test_write(self):
        """Writing to DBF tables"""
        eq = self.assertEquals

        # First create a DBF file
        dbffilename = self.temp_file_name("table_write.dbf")
        dbf = dbflib.create(dbffilename)
        dbf.add_field("NAME", dbflib.FTString, 20, 0)
        dbf.add_field("INT", dbflib.FTInteger, 10, 0)
        dbf.add_field("FLOAT", dbflib.FTDouble, 10, 4)
        dbf.write_record(0, {'NAME': "Weatherwax", "INT":1,
                             "FLOAT":3.1415926535})
        dbf.close()

        # Create the table
        table = Table(dbffilename)
        record = table.read_record(0)
        # The FLOAT value is different from above because of rounding
        eq(record, {'NAME': "Weatherwax", "INT":1, "FLOAT":3.1415999999999999})

        # change only one field
        table.write_record(0, {"NAME": "Ogg"})
        # check whether it has been written immediately
        dbf = dbflib.DBFFile(dbffilename)
        control = dbf.read_record(0)
        eq(control, {'NAME': "Ogg", "INT":1, "FLOAT":3.1415999999999999})
        dbf.close()

        # check whether the table itself returns the new value
        eq(table.read_record(0),
           {'NAME': "Ogg", "INT":1, "FLOAT":3.1415999999999999})

        # Check whether we can specify the record as a tuple
        table.write_record(0, ("Garlick", 2, 1.5))
        eq(table.read_record(0), {"NAME": "Garlick", "INT": 2, "FLOAT": 1.5})

        table.Destroy()


if __name__ == "__main__":
    unittest.main()
