# Copyright (c) 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test saving a thuban session as XML
"""

__version__ = "$Revision: 1.4 $"
# $Source: /thubanrepository/thuban/test/test_save.py,v $
# $Id: test_save.py,v 1.4 2003/03/12 19:58:00 jonathan Exp $

import os
import unittest
from StringIO import StringIO

import xml.sax
import xml.sax.handler
from xml.sax import make_parser, ErrorHandler, SAXNotRecognizedException

import support
support.initthuban()

from Thuban.Model.save import save_session
from Thuban.Model.session import Session
from Thuban.Model.map import Map
from Thuban.Model.layer import Layer
from Thuban.Model.proj import Projection


class SaxEventLister(xml.sax.handler.ContentHandler):

    def __init__(self):
        self.eventlist = []

    def startElementNS(self, name, qname, attrs):
        items = attrs.items()
        items.sort()
        self.eventlist.append(("start", name, qname, items))

    def endElementNS(self, name, qname):
        self.eventlist.append(("end", name, qname))


def sax_eventlist(data):
    """Return a list of SAX event generated for the XML data.
    """
    handler = SaxEventLister()
    parser = make_parser()
    parser.setContentHandler(handler)
    parser.setErrorHandler(ErrorHandler())
    parser.setFeature(xml.sax.handler.feature_namespaces, 1)

    #
    # see comment at the end of Thuban/Model/load.py
    #
    try:
        parser.setFeature(xml.sax.handler.feature_validation, 0)
        parser.setFeature(xml.sax.handler.feature_external_ges, 0)
        parser.setFeature(xml.sax.handler.feature_external_pes, 0)
    except SAXNotRecognizedException:
        pass

    inpsrc = xml.sax.InputSource()
    inpsrc.setByteStream(StringIO(data))
    parser.parse(inpsrc)

    return handler.eventlist

class SaveSessionTest(unittest.TestCase, support.FileTestMixin):

    def compare_xml(self, xml1, xml2):
        self.assertEquals(sax_eventlist(xml1), sax_eventlist(xml2))

    def testEmptySession(self):
        """Save an empty session"""
        session = Session("empty session")
        filename = self.temp_file_name("save_emptysession.thuban")
        save_session(session, filename)
        session.Destroy()

        file = open(filename)
        written_contents = file.read()
        file.close()
        self.compare_xml(written_contents,
                         '<?xml version="1.0" encoding="UTF-8"?>\n'
                         '<!DOCTYPE session SYSTEM "thuban.dtd">\n'
                         '<session title="empty session">\n</session>\n')

    def testSingleLayer(self):
        """Save a session with a single map with a single layer"""
        # deliberately put an apersand in the title :)
        session = Session("single map&layer")
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        map = Map("Test Map", projection = proj)
        session.AddMap(map)
        # use shapefile from the example data
        shpfile = os.path.join(os.path.dirname(__file__),
                               os.pardir, "Data", "iceland", "political.shp")
        layer = Layer("My Layer", shpfile)
        map.AddLayer(layer)

        filename = self.temp_file_name("save_singlemap.thuban")
        save_session(session, filename)
        session.Destroy()

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_contents = '''<?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE session SYSTEM "thuban.dtd">
        <session title="single map&amp;layer">
            <map title="Test Map">
                <projection>
                    <parameter value="zone=26"/>
                    <parameter value="proj=utm"/>
                    <parameter value="ellps=clrk66"/>
                </projection>
                <layer title="My Layer" filename="%s"
                fill="None" stroke="#000000" stroke_width="1"/>
            </map>
        </session>''' % os.path.join("..", "..", "Data", "iceland",
                                     "political.shp")
        #print written_contents
        #print "********************************************"
        #print expected_contents
        self.compare_xml(written_contents, expected_contents)



if __name__ == "__main__":
    # Fake the __file__ global because it's needed by a test
    import sys
    __file__ = sys.argv[0]
    unittest.main()
