# Copyright (c) 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test loading a thuban session from a file
"""

__version__ = "$Revision: 1.3 $"
# $Source: /thubanrepository/thuban/test/test_load.py,v $
# $Id: test_load.py,v 1.3 2003/03/06 16:47:02 jonathan Exp $

import os
import unittest

import support
support.initthuban()

from Thuban.Model.load import load_session
from Thuban.Model.session import Session
from Thuban.Model.map import Map
from Thuban.Model.layer import Layer
from Thuban.Model.proj import Projection
from Thuban.Model.color import Color

def filenames_equal(name1, name2):
    """Return true if the filenames name1 and name2 are equal.

    On systems where it is available, simply use os.path.samefile,
    otherwise return whether the normalized versions of the filenames
    according to os.path.normpath are equal.
    """
    if hasattr(os.path, "samefile"):
        return os.path.samefile(name1, name2)
    return os.path.normpath(name1) == os.path.normpath(name2)


contents_single_map = '''\
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE session SYSTEM "thuban.dtd">
<session title="single map&amp;layer">
	<map title="Test Map">
		<projection>
			<parameter value="zone=26"/>
			<parameter value="proj=utm"/>
			<parameter value="ellps=clrk66"/>
		</projection>
		<layer title="My Layer" stroke_width="1" fill="None"
                    filename="../../Data/iceland/political.shp"
                    stroke="#000000"/>
	</map>
</session>
'''


class LoadSessionTest(unittest.TestCase, support.FileTestMixin):

    def setUp(self):
        """Create the test files"""
        file = open(self.temp_file_name("load_singlelayer.thuban"), "w")
        file.write(contents_single_map)
        file.close()
        self.session = None

    def tearDown(self):
        if self.session is not None:
            self.session.Destroy()

    def testSingleLayer(self):
        """Load a session with a single map with a single layer"""
        eq = self.assertEquals
        session = load_session(self.temp_file_name("load_singlelayer.thuban"))
        self.session = session

        # Check the title
        eq(session.Title(), "single map&layer")

        # the session has one map.
        maps = session.Maps()
        eq(len(maps), 1)

        # Check the map's attributes
        map = maps[0]
        eq(map.Title(), "Test Map")

        # the map has a single layer
        layers = map.Layers()
        eq(len(layers), 1)

        # Check the layer attributes
        layer = layers[0]
        eq(layer.Title(), "My Layer")
        self.failUnless(filenames_equal(layer.filename,
                                        os.path.join(self.temp_dir(),
                                                     os.pardir, os.pardir,
                                                     "Data", "iceland",
                                                     "political.shp")))
        eq(layer.GetClassification().GetDefaultFill(), Color.None)
        eq(layer.GetClassification().GetDefaultLineColor().hex(), "#000000")


if __name__ == "__main__":
    unittest.main()
