# Copyright (c) 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test the Layer class
"""

__version__ = "$Revision: 1.4 $"
# $Source: /thubanrepository/thuban/test/test_layer.py,v $
# $Id: test_layer.py,v 1.4 2003/02/19 16:50:39 jonathan Exp $

import os
import unittest

import support
support.initthuban()

import shapelib
import dbflib

from Thuban.Model.layer import Layer, SHAPETYPE_POLYGON, SHAPETYPE_ARC, \
     SHAPETYPE_POINT
from Thuban.Model.messages import LAYER_LEGEND_CHANGED, \
     LAYER_VISIBILITY_CHANGED
from Thuban.Model.color import Color

class TestLayer(unittest.TestCase, support.FileTestMixin,
                support.FloatComparisonMixin):

    """Test cases for different layer (shape) types"""

    def assertFloatTuplesEqual(self, test, value):
        """Assert equality of two lists of tuples of float"""
        for i in range(len(test)):
            self.assertFloatSeqEqual(test[i], value[i])

    def test_arc_layer(self):
        """Test Layer with arc shapes"""
        layer = Layer("Test Layer",
                      os.path.join("..", "Data", "iceland", "roads-line.shp"))
        self.assertEquals(layer.Title(), "Test Layer")
        self.assertEquals(layer.ShapeType(), SHAPETYPE_ARC)
        self.assertEquals(layer.NumShapes(), 839)
        shape = layer.Shape(32)
        self.assertFloatTuplesEqual(shape.Points(),
                                    [(-15.082174301147461, 66.27738189697265),
                                     (-15.026350021362305, 66.27339172363281)])
        self.assertFloatSeqEqual(layer.BoundingBox(),
                                 [-24.450359344482422, 63.426830291748047,
                                  -13.55668830871582, 66.520111083984375])
        self.assertEquals(layer.ShapesInRegion((-24.0, 64.0, -23.75, 64.25)),
                          [613, 726, 838])

    def test_polygon_layer(self):
        """Test Layer with polygon shapes"""
        layer = Layer("Test Layer",
                      os.path.join("..", "Data", "iceland", "political.shp"))
        self.assertEquals(layer.Title(), "Test Layer")
        self.assertEquals(layer.ShapeType(), SHAPETYPE_POLYGON)
        self.assertEquals(layer.NumShapes(), 156)
        shape = layer.Shape(4)
        self.assertFloatTuplesEqual(shape.Points(),
                                    [(-22.406391143798828, 64.714111328125),
                                     (-22.41621208190918, 64.71600341796875),
                                     (-22.406051635742188, 64.719200134277344),
                                     (-22.406391143798828, 64.714111328125)])
        self.assertFloatSeqEqual(layer.BoundingBox(),
                                 [-24.546524047851562, 63.286754608154297,
                                  -13.495815277099609, 66.563774108886719])
        self.assertEquals(layer.ShapesInRegion((-24.0, 64.0, -23.9, 64.1)),
                          [91, 92, 144, 146, 148, 150, 152, 153])

    def test_point_layer(self):
        """Test Layer with point shapes"""
        layer = Layer("Test Layer",
                      os.path.join("..", "Data", "iceland",
                                   "cultural_landmark-point.shp"))
        self.assertEquals(layer.Title(), "Test Layer")
        self.assertEquals(layer.ShapeType(), SHAPETYPE_POINT)
        self.assertEquals(layer.NumShapes(), 34)
        shape = layer.Shape(0)
        self.assertFloatTuplesEqual(shape.Points(),
                                    [(-22.711074829101562, 66.36572265625)])
        self.assertFloatSeqEqual(layer.BoundingBox(),
                                 [-23.806047439575195, 63.405960083007812,
                                  -15.12291431427002, 66.36572265625])
        self.assertEquals(layer.ShapesInRegion((-24.0, 64.0, -23.80, 64.1)),
                          [0, 1, 2, 3, 4, 5, 27, 28, 29, 30, 31])

    def test_empty_layer(self):
        """Test Layer with empty shape file"""
        # create an empty shape file
        shapefilename = self.temp_file_name("layer_empty.shp")
        shp = shapelib.create(shapefilename, shapelib.SHPT_POLYGON)
        shp.close()
        # create an empty DBF file too because Thuban can't cope yet
        # with missing DBF file.
        dbffilename = self.temp_file_name("layer_empty.dbf")
        dbf = dbflib.create(dbffilename)
        dbf.add_field("NAME", dbflib.FTString, 20, 0)

        # Now try to open it.
        layer = Layer("Empty Layer", shapefilename)
        self.assertEquals(layer.BoundingBox(), None)
        self.assertEquals(layer.LatLongBoundingBox(), None)
        self.assertEquals(layer.NumShapes(), 0)


class TestLayerLegend(unittest.TestCase, support.SubscriberMixin):

    """Test cases for Layer method that modify the layer.
    """

    def setUp(self):
        """Clear the list of received messages and create a layer

        The layer is bound to self.layer.
        """
        self.clear_messages()
        self.layer = Layer("Test Layer",
                           os.path.join("..", "Data", "iceland",
                                        "political.shp"))
        self.layer.Subscribe(LAYER_LEGEND_CHANGED, self.subscribe_with_params,
                             LAYER_LEGEND_CHANGED)
        self.layer.Subscribe(LAYER_VISIBILITY_CHANGED,
                             self.subscribe_with_params,
                             LAYER_VISIBILITY_CHANGED)

    def tearDown(self):
        """Clear the list of received messages and explictly destroy self.layer
        """
        self.layer.Destroy()
        self.clear_messages()

    def test_initial_settings(self):
        """Test Layer's initial legend attributes"""
        # test default settings
        self.failIf(self.layer.WasModified())
        #self.assertEquals(self.layer.fill, None)
        #self.assertEquals(self.layer.stroke.hex(), "#000000")
        #self.assertEquals(self.layer.stroke_width, 1)
        self.assertEquals(self.layer.Visible(), 1)
        # no messages should have been produced
        self.check_messages([])

    def test_visibility(self):
        """Test Layer visibility"""
        self.layer.SetVisible(0)
        self.assertEquals(self.layer.Visible(), 0)
        self.check_messages([(self.layer, LAYER_VISIBILITY_CHANGED)])

        # currently, modifying the visibility doesn't count as changing
        # the layer.
        self.failIf(self.layer.WasModified())

    
#
# the tree info now contains Color objects which are difficult to test
#
#   def test_tree_info(self):
#       """Test Layer.TreeInfo"""
#       self.assertEquals(self.layer.TreeInfo(),
#                         ("Layer 'Test Layer'",
#                          ['Shown',
#                           'Shapes: 156',
#                           ('Extent (lat-lon):'
#                            ' (-24.5465, 63.2868, -13.4958, 66.5638)'),
#                           'Shapetype: Polygon',
#                           'Fill: None',
#                           'Outline: (0.000, 0.000, 0.000)']))


if __name__ == "__main__":
    unittest.main()
