# Copyright (c) 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Support classes and function for the test suite
"""

__version__ = "$Revision: 1.3 $"
# $Source: /thubanrepository/thuban/test/support.py,v $
# $Id: support.py,v 1.3 2002/10/18 17:28:46 bh Exp $

import os, sys
import unittest

def thuban_dir():
    """Return the directory containing the Thuban package"""
    thisdir = os.path.dirname(__file__)
    return os.path.join(thisdir, os.pardir)


def add_thuban_dir_to_path():
    """Insert the Thuban directory at the beginning of the python path.

    If it's already part of the path, remove later occurrences.
    """
    dir = thuban_dir()
    while 1:
        try:
            sys.path.remove(dir)
        except ValueError:
            break
    sys.path.insert(0, dir)


_initthuban_done = 0
def initthuban():
    """Initialize the interpreter for using Thuban modules
    """
    global _initthuban_done
    if not _initthuban_done:
        add_thuban_dir_to_path()
        import thubaninit
        _initthuban_done = 1

def run_suite(suite):
    """Run the test suite suite and return the result"""
    runner = unittest.TextTestRunner(verbosity = 2)
    return runner.run(suite)


def create_temp_dir():
    """Create a temporary directory and return its name.

    The temporary directory is always called temp and is created in the
    directory where support module is located.

    If the temp directory already exists, just return the name.
    """
    name = os.path.abspath(os.path.join(os.path.dirname(__file__), "temp"))

    # if the directory already exists, we're done
    if os.path.isdir(name):
        return name

    # create the directory
    os.mkdir(name)
    return name


class FileTestMixin:

    """Mixin class for tests that use files in the temporary directory
    """

    def temp_file_name(self, basename):
        """Return the full name of the file named basename in the temp. dir"""
        return os.path.join(create_temp_dir(), basename)

    def temp_dir(self):
        """Return the name of the directory for the temporary files"""
        return create_temp_dir()


class FloatComparisonMixin:

    """
    Mixin class for tests comparing floating point numbers.

    This class provides a few methods for testing floating point
    operations.
    """

    fp_epsilon = 1e-6
    
    def assertFloatEqual(self, test, value):
        """Assert equality of test and value with some tolerance.

        Assert that the absolute difference between test and value is
        less than self.fp_epsilon.
        """
        self.assert_(self.fp_epsilon > abs(test - value),
                     "abs(%g - %g) >= %g" % (test, value, self.fp_epsilon))

    def assertFloatSeqEqual(self, test, value, epsilon = None):
        """Assert equality of the sequences test and value with some tolerance.

        Assert that the absolute difference between each corresponding
        value in test and value is less than the optional parameter
        epsilon. If epsilon is not given use self.fp_epsilon.
        """
        if epsilon is None:
            epsilon = self.fp_epsilon
        for i in range(len(test)):
            self.assert_(epsilon > abs(test[i] - value[i]),
                         "abs(%g - %g) >= %g" % (test[i], value[i], epsilon))


class SubscriberMixin:

    """Mixin class for tests for messages sent through the Connector

    The SubscriberMixin has some methods that can be used as subscribers
    of events that when called append information about the message into
    a list of messages received.

    A derived class should call the clear_messages() method in both its
    setUp and tearDown methods to clear the list of messages received.
    """

    def clear_messages(self):
        """Clear the list of received messages.

        Call this at least in the tests setUp and tearDown methods. It's
        important to do it in tearDown too because otherwise there may
        be cyclic references.
        """
        self.received_messages = []

    def subscribe_no_params(self):
        """Method for subscriptions without parameters.

        Add an empty tuple to the list of received messages.
        """
        self.received_messages.append(())

    def subscribe_with_params(self, *args):
        """Method for subscriptions with parameters.

        Append the tuple will all arguments to this function (except for
        the self argument) to the list of received messages.
        """
        self.received_messages.append(args)

    def check_messages(self, messages):
        """Check whether the messages received match the list messages"""
        self.assertEquals(messages, self.received_messages)
