# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.7 $"

from wxPython.wx import *
from wxPython.grid import *

from Thuban.Lib.connector import Publisher
from Thuban.Model.table import FIELDTYPE_INT, FIELDTYPE_DOUBLE, \
     FIELDTYPE_STRING
import view
from dialogs import NonModalDialog
from messages import SELECTED_SHAPE

wx_value_type_map = {FIELDTYPE_INT: wxGRID_VALUE_NUMBER,
                     FIELDTYPE_DOUBLE: wxGRID_VALUE_FLOAT,
                     FIELDTYPE_STRING: wxGRID_VALUE_STRING}

ROW_SELECTED = "ROW_SELECTED"


class DataTable(wxPyGridTableBase):

    """Wrapper around a Thuban table object suitable for a wxGrid"""

    def __init__(self, table = None):
        wxPyGridTableBase.__init__(self)
        self.num_cols = 0
        self.num_rows = 0
        self.columns = []
        self.table = None
        self.SetTable(table)

    def SetTable(self, table):
        self.table = table
        self.num_cols = table.field_count()
        self.num_rows = table.record_count()

        self.columns = []
        for i in range(self.num_cols):
            type, name, len, decc = table.field_info(i)
            self.columns.append((name, wx_value_type_map[type], len, decc))

    #
    # required methods for the wxPyGridTableBase interface
    #

    def GetNumberRows(self):
        return self.num_rows

    def GetNumberCols(self):
        return self.num_cols

    def IsEmptyCell(self, row, col):
        return 0

    # Get/Set values in the table.  The Python version of these
    # methods can handle any data-type, (as long as the Editor and
    # Renderer understands the type too,) not just strings as in the
    # C++ version.
    def GetValue(self, row, col):
        record = self.table.read_record(row)
        return record[self.columns[col][0]]

    def SetValue(self, row, col, value):
        pass

    #
    # Some optional methods
    #

    # Called when the grid needs to display labels
    def GetColLabelValue(self, col):
        return self.columns[col][0]

    # Called to determine the kind of editor/renderer to use by
    # default, doesn't necessarily have to be the same type used
    # nativly by the editor/renderer if they know how to convert.
    def GetTypeName(self, row, col):
        return self.columns[col][1]

    # Called to determine how the data can be fetched and stored by the
    # editor and renderer.  This allows you to enforce some type-safety
    # in the grid.
    def CanGetValueAs(self, row, col, typeName):
        # perhaps we should allow conversion int->double?
        return self.GetTypeName(row, col) == typeName

    def CanSetValueAs(self, row, col, typeName):
        return self.CanGetValueAs(row, col, typeName)


class TableGrid(wxGrid, Publisher):

    """A grid view for a Thuban table"""

    def __init__(self, parent, table = None):
        wxGrid.__init__(self, parent, -1)

        self.table = DataTable(table)

        # The second parameter means that the grid is to take ownership
        # of the table and will destroy it when done. Otherwise you
        # would need to keep a reference to it and call its Destroy
        # method later.
        self.SetTable(self.table, true)

        #self.SetMargins(0,0)

        # AutoSizeColumns would allow us to make the grid have optimal
        # column widths automatically but it would cause a traversal of
        # the entire table which for large .dbf files can take a very
        # long time.
        #self.AutoSizeColumns(false)

        self.SetSelectionMode(wxGrid.wxGridSelectRows)

        EVT_GRID_RANGE_SELECT(self, self.OnRangeSelect)
        EVT_GRID_SELECT_CELL(self, self.OnSelectCell)

    def SetTableObject(self, table):
        self.table.SetTable(table)

    def OnRangeSelect(self, event):
        if event.Selecting():
            self.issue(ROW_SELECTED, event.GetTopLeftCoords().GetRow())

    def OnSelectCell(self, event):
        self.issue(ROW_SELECTED, event.GetRow())


class LayerTableGrid(TableGrid):

    """Table grid for the layer tables.

    The LayerTableGrid is basically the same as a TableGrid but it's
    selection is usually coupled to the selected object in the map.
    """

    def select_shape(self, layer, shape):
        """Select the row corresponding to the specified shape and layer

        If layer is not the layer the table is associated with do
        nothing. If shape or layer is None also do nothing.
        """
        if layer is not None and layer.table is self.table.table \
           and shape is not None:
            self.SelectRow(shape)
            self.SetGridCursor(shape, 0)
            self.MakeCellVisible(shape, 0)


class TableFrame(NonModalDialog):

    """Frame that displays a Thuban table in a grid view"""

    def __init__(self, parent, interactor, name, title, table):
        NonModalDialog.__init__(self, parent, interactor, name, title)
        self.table = table
        self.grid = self.make_grid(self.table)

    def make_grid(self, table):
        """Return the table grid to use in the frame.

        The default implementation returns a TableGrid instance.
        Override in derived classes to use different grid classes.
        """
        return TableGrid(self, table)


class LayerTableFrame(TableFrame):

    """Frame that displays a layer table in a grid view

    A LayerTableFrame is TableFrame whose selection is connected to the
    selected object in a map.
    """

    def __init__(self, parent, interactor, name, title, layer, table):
        TableFrame.__init__(self, parent, interactor, name, title, table)
        self.layer = layer
        self.grid.Subscribe(ROW_SELECTED, self.row_selected)
        self.interactor.Subscribe(SELECTED_SHAPE, self.select_shape)

    def make_grid(self, table):
        """Override the derived method to return a LayerTableGrid.
        """
        return LayerTableGrid(self, table)

    def OnClose(self, event):
        self.interactor.Unsubscribe(SELECTED_SHAPE, self.select_shape)
        TableFrame.OnClose(self, event)

    def select_shape(self, layer, shape):
        self.grid.select_shape(layer, shape)

    def row_selected(self, row):
        if self.layer is not None:
            self.interactor.SelectLayerAndShape(self.layer, row)
