# Copyright (c) 2001, 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.13 $"

from wxPython.wx import wxPoint, wxPen, wxBrush, wxFont, \
     wxTRANSPARENT_PEN, wxTRANSPARENT_BRUSH, \
     wxBLACK, wxSOLID, wxCROSS_HATCH, wxSWISS, wxNORMAL

from wxproj import draw_polygon_shape

from Thuban import _
from Thuban.UI.common import *

from Thuban.Model.layer import SHAPETYPE_POLYGON, SHAPETYPE_ARC, \
     SHAPETYPE_POINT
from Thuban.Model.label import ALIGN_CENTER, ALIGN_TOP, ALIGN_BOTTOM, \
     ALIGN_LEFT, ALIGN_RIGHT, ALIGN_BASELINE

from Thuban.Model.classification import Classification
from Thuban.Model.color import Color


class MapRenderer:

    """Class to render a map onto a wxDC"""

    honor_visibility = 1

    def __init__(self, dc, scale, offset, resolution = 72.0,
                 honor_visibility = None):
        """Inititalize the renderer.

        dc -- the wxPython DC to render on
        scale, offset -- the scale factor and translation to convert
                between projected coordinates and the DC coordinates

        resolution -- the assumed resolution of the DC. Used to convert
                absolute lengths like font sizes to DC coordinates

        honor_visibility -- boolean. If true, honor the visibility flag
                of the layers, otherwise draw all layers. If None, use
                the renderer's default.
        """
        # resolution in pixel/inch
        self.dc = dc
        self.scale = scale
        self.offset = offset
        if honor_visibility is not None:
            self.honor_visibility = honor_visibility
        # store the resolution in pixel/point because it's more useful
        # later.
        self.resolution = resolution / 72.0

    def render_map(self, map):
        self.map = map
        for layer in map.Layers():
            # if honor_visibility is true, only draw visible layers,
            # otherwise draw all layers
            if not self.honor_visibility or layer.Visible():
                self.draw_shape_layer(layer)
        self.draw_label_layer(map.LabelLayer())

    def draw_shape_layer(self, layer):
        scale = self.scale
        offx, offy = self.offset

        map_proj = self.map.projection
        layer_proj = layer.projection

        shapetype = layer.ShapeType()

        brush = wxTRANSPARENT_BRUSH
        pen   = wxTRANSPARENT_PEN

        old_prop = None
        lc = layer.GetClassification()
        field = lc.GetField()

        defaultProps = lc.GetDefaultGroup().GetProperties()

        for i in self.layer_ids(layer):
            value = None

            if field is not None:
                try:
                    record = layer.table.read_record(i)
                    if record is not None:
                        value = record[field]
                except:
                    pass

                #
                # if the above statements fail 'value' should
                # be null, at which point this call will
                # at least retreive the NullData
                #
                prop = lc.GetProperties(value)
            else:
                prop = defaultProps

            # don't recreate new objects if they are the same as before
            if prop != old_prop: 
                old_prop = prop

                if shapetype == SHAPETYPE_ARC:
                    fill = Color.None
                else:
                    fill = prop.GetFill()
    
                if fill is Color.None:
                    brush = wxTRANSPARENT_BRUSH
                else:
                    color = Color2wxColour(fill)
                    brush = wxBrush(color, wxSOLID)
    
                stroke = prop.GetLineColor()
                stroke_width = prop.GetLineWidth()
                if stroke is Color.None:
                    pen = wxTRANSPARENT_PEN
                else:
                    color = Color2wxColour(stroke)
                    pen = wxPen(color, stroke_width, wxSOLID)
    
            if shapetype == SHAPETYPE_POINT:
                self.dc.SetBrush(brush)
                self.dc.SetPen(pen)
                self.draw_point_shape(layer, i)
            else:
                self.draw_polygon_shape(layer, i, pen, brush)

    def layer_ids(self, layer):
        """Return the shape ids of the given layer that have to be drawn.
        
        The default implementation simply returns all ids in the layer.
        Override in derived classes to be more precise.
        """
        return range(layer.NumShapes())

    def draw_polygon_shape(self, layer, index, pen, brush):
        offx, offy = self.offset        
        draw_polygon_shape(layer.shapefile.cobject(), index,
                           self.dc, pen, brush,
                           self.map.projection, layer.projection,
                           self.scale, -self.scale, offx, offy)

    def projected_points(self, layer, index):
        proj = self.map.projection
        if proj is not None:
            forward = proj.Forward
        else:
            forward = None
        proj = layer.projection
        if proj is not None:
            inverse = proj.Inverse
        else:
            inverse = None
        shape = layer.Shape(index)
        points = []
        scale = self.scale
        offx, offy = self.offset
        for x, y in shape.Points():
            if inverse:
                x, y = inverse(x, y)
            if forward:
                x, y = forward(x, y)
            points.append(wxPoint(x * scale + offx,
                                  -y * scale + offy))
        return points

    def draw_arc_shape(self, layer, index):
        points = self.projected_points(layer, index)
        self.dc.DrawLines(points)

    def draw_point_shape(self, layer, index):
        p = self.projected_points(layer, index)[0]
        radius = self.resolution * 5
        self.dc.DrawEllipse(p.x - radius, p.y - radius, 2*radius, 2*radius)

    def draw_label_layer(self, layer):
        scale = self.scale
        offx, offy = self.offset

        font = wxFont(self.resolution * 10, wxSWISS, wxNORMAL, wxNORMAL)
        self.dc.SetFont(font)

        map_proj = self.map.projection
        if map_proj is not None:
            forward = map_proj.Forward
        else:
            forward = None

        for label in layer.Labels():
            x = label.x
            y = label.y
            text = label.text
            if forward:
                x, y = forward(x, y)
            x = x * scale + offx
            y = -y * scale + offy
            width, height = self.dc.GetTextExtent(text)
            if label.halign == ALIGN_LEFT:
                # nothing to be done
                pass
            elif label.halign == ALIGN_RIGHT:
                x = x - width
            elif label.halign == ALIGN_CENTER:
                x = x - width/2
            if label.valign == ALIGN_TOP:
                # nothing to be done
                pass
            elif label.valign == ALIGN_BOTTOM:
                y = y - height
            elif label.valign == ALIGN_CENTER:
                y = y - height/2
            self.dc.DrawText(text, x, y)


class ScreenRenderer(MapRenderer):

    # On the screen we want to see only visible layers by default
    honor_visibility = 1
    
    def RenderMap(self, map, region, selected_layer, selected_shape):
        """Render the map.

        Only the given region (a tuple in window coordinates as returned
        by a wxrect's asTuple method) needs to be redrawn. Highlight the
        shape with id selected_shape in the selected_layer.
        """
        self.update_region = region
        self.selected_layer = selected_layer
        self.selected_shape = selected_shape
        self.render_map(map)

    def draw_shape_layer(self, layer):
        MapRenderer.draw_shape_layer(self, layer)
        if layer is self.selected_layer and self.selected_shape is not None:
            pen = wxPen(wxBLACK, 3, wxSOLID)
            brush = wxBrush(wxBLACK, wxCROSS_HATCH)
            
            shapetype = layer.ShapeType()
            index = self.selected_shape
            if shapetype == SHAPETYPE_POLYGON:
                self.draw_polygon_shape(layer, index, pen, brush)
            elif shapetype == SHAPETYPE_ARC:
                self.draw_polygon_shape(layer, index, pen, None)
            else:
                self.dc.SetBrush(brush)
                self.dc.SetPen(pen)
                if shapetype == SHAPETYPE_POINT:
                    self.draw_point_shape(layer, index)
                else:
                    raise TypeError(_("Unhandled shape type %s") % shapetype)

    def layer_ids(self, layer):
        """Return the shapeids covered by the region that has to be redrawn

        Call the layer's ShapesInRegion method to determine the ids so
        that it can use the quadtree.
        """
        # FIXME: the quad-tree should be built from the projected
        # coordinates not the lat-long ones because it's not trivial to
        # determine an appropriate rectangle in lat-long for a given
        # rectangle in projected coordinates which we have to start from
        # here.
        proj = self.map.projection
        if proj is not None:
            inverse = proj.Inverse
        else:
            inverse = None

        scale = self.scale
        offx, offy = self.offset
        xs = []
        ys = []
        x, y, width, height = self.update_region
        for winx, winy in ((x, y), (x + width, y),
                           (x + width, y + height), (x, y + height)):
            px = (winx - offx) / scale
            py = -(winy - offy) / scale
            if inverse:
                px, py = inverse(px, py)
            xs.append(px)
            ys.append(py)
        left = min(xs)
        right = max(xs)
        top = max(ys)
        bottom = min(ys)

        return layer.ShapesInRegion((left, bottom, right, top))


class PrinterRender(MapRenderer):

    # When printing we want to see all layers
    honor_visibility = 0

    RenderMap = MapRenderer.render_map
    
