# Copyright (C) 2001, 2002 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
The main window
"""

__version__ = "$Revision: 1.51 $"

__ThubanVersion__ = "0.2" #"$THUBAN_0_2$"
#__BuildDate__ = "$Date: 2003/03/11 17:28:39 $"

import os

from wxPython.wx import *

import Thuban
from Thuban import _
from Thuban.Model.session import create_empty_session
from Thuban.Model.layer import Layer
from Thuban.Model.color import Color
from Thuban.Model.proj import Projection

import view
import tree
import proj4dialog
import tableview, identifyview
import classifier
from menu import Menu

from context import Context
from command import registry, Command, ToolCommand
from messages import SELECTED_SHAPE, VIEW_POSITION


# the directory where the toolbar icons are stored
bitmapdir = os.path.join(Thuban.__path__[0], os.pardir, "Resources", "Bitmaps")
bitmapext = ".xpm"


class MainWindow(wxFrame):

    def __init__(self, parent, ID, title, application, interactor,
                 initial_message = None, size = wxSize(-1, -1)):
        wxFrame.__init__(self, parent, ID, title, wxDefaultPosition, size)

        self.application = application
        self.interactor = interactor

        self.CreateStatusBar()
        if initial_message:
            self.SetStatusText(initial_message)

        self.identify_view = None

        self.init_ids()

        # creat the menubar from the main_menu description
        self.SetMenuBar(self.build_menu_bar(main_menu))

        # Similarly, create the toolbar from main_toolbar
        toolbar = self.build_toolbar(main_toolbar)
        # call Realize to make sure that the tools appear.
        toolbar.Realize()

        # Create the map canvas
        canvas = view.MapCanvas(self, -1, interactor)
        canvas.Subscribe(VIEW_POSITION, self.view_position_changed)
        self.canvas = canvas

        self.init_dialogs()

        interactor.Subscribe(SELECTED_SHAPE, self.identify_view_on_demand)

        EVT_CLOSE(self, self.OnClose)

    def init_ids(self):
        """Initialize the ids"""
        self.current_id = 6000
        self.id_to_name = {}
        self.name_to_id = {}
        self.events_bound = {}

    def get_id(self, name):
        """Return the wxWindows id for the command named name.

        Create a new one if there isn't one yet"""
        ID = self.name_to_id.get(name)
        if ID is None:
            ID = self.current_id
            self.current_id = self.current_id + 1
            self.name_to_id[name] = ID
            self.id_to_name[ID] = name
        return ID

    def bind_command_events(self, command, ID):
        """Bind the necessary events for the given command and ID"""
        if not self.events_bound.has_key(ID):
            # the events haven't been bound yet
            EVT_MENU(self, ID, self.invoke_command)
            if command.IsDynamic():
                EVT_UPDATE_UI(self, ID, self.update_command_ui)

    def build_menu_bar(self, menudesc):
        """Build and return the menu bar from the menu description"""
        menu_bar = wxMenuBar()

        for item in menudesc.items:
            # here the items must all be Menu instances themselves
            menu_bar.Append(self.build_menu(item), item.title)

        return menu_bar

    def build_menu(self, menudesc):
        """Return a wxMenu built from the menu description menudesc"""
        wxmenu = wxMenu()
        last = None
        for item in menudesc.items:
            if item is None:
                # a separator. Only add one if the last item was not a
                # separator
                if last is not None:
                    wxmenu.AppendSeparator()
            elif isinstance(item, Menu):
                # a submenu
                wxmenu.AppendMenu(wxNewId(), item.title, self.build_menu(item))
            else:
                # must the name the name of a command
                self.add_menu_command(wxmenu, item)
            last = item
        return wxmenu

    def build_toolbar(self, toolbardesc):
        """Build and return the main toolbar window from a toolbar description

        The parameter should be an instance of the Menu class but it
        should not contain submenus.
        """
        toolbar = self.CreateToolBar(wxTB_3DBUTTONS)

        # set the size of the tools' bitmaps. Not needed on wxGTK, but
        # on Windows, although it doesn't work very well there. It seems
        # that only 16x16 icons are really supported on windows.
        # We probably shouldn't hardwire the bitmap size here.
        toolbar.SetToolBitmapSize(wxSize(24, 24))

        for item in toolbardesc.items:
            if item is None:
                toolbar.AddSeparator()
            else:
                # assume it's a string.
                self.add_toolbar_command(toolbar, item)

        return toolbar

    def add_menu_command(self, menu, name):
        """Add the command with name name to the menu menu.

        If name is None, add a separator.
        """
        if name is None:
            menu.AppendSeparator()
        else:
            command = registry.Command(name)
            if command is not None:
                ID = self.get_id(name)
                menu.Append(ID, command.Title(), command.HelpText(),
                            command.IsCheckCommand())
                self.bind_command_events(command, ID)
            else:
                print _("Unknown command %s") % name

    def add_toolbar_command(self, toolbar, name):
        """Add the command with name name to the toolbar toolbar.

        If name is None, add a separator.
        """
        # Assume that all toolbar commands are also menu commmands so
        # that we don't have to add the event handlers here
        if name is None:
            toolbar.AddSeparator()
        else:
            command = registry.Command(name)
            if command is not None:
                ID = self.get_id(name)
                filename = os.path.join(bitmapdir, command.Icon()) + bitmapext
                bitmap = wxBitmap(filename, wxBITMAP_TYPE_XPM)
                toolbar.AddTool(ID, bitmap,
                                shortHelpString = command.HelpText(),
                                isToggle = command.IsCheckCommand())
                self.bind_command_events(command, ID)
            else:
                print _("Unknown command %s") % name

    def Context(self):
        """Return the context object for a command invoked from this window
        """
        return Context(self.application, self.application.Session(), self)

    def invoke_command(self, event):
        name = self.id_to_name.get(event.GetId())
        if name is not None:
            command = registry.Command(name)
            command.Execute(self.Context())
        else:
            print _("Unknown command ID %d") % event.GetId()

    def update_command_ui(self, event):
        #print "update_command_ui", self.id_to_name[event.GetId()]
        context = self.Context()
        command = registry.Command(self.id_to_name[event.GetId()])
        if command is not None:
            sensitive = command.Sensitive(context)
            event.Enable(sensitive)
            if command.IsTool() and not sensitive and command.Checked(context):
                # When a checked tool command is disabled deselect all
                # tools. Otherwise the tool would remain active but it
                # might lead to errors if the tools stays active. This
                # problem occurred in GREAT-ER and this fixes it, but
                # it's not clear to me whether this is really the best
                # way to do it (BH, 20021206).
                self.canvas.SelectTool(None)
            event.SetText(command.DynText(context))
            if command.IsCheckCommand():
                    event.Check(command.Checked(context))

    def RunMessageBox(self, title, text, flags = wxOK | wxICON_INFORMATION):
        """Run a modal message box with the given text, title and flags
        and return the result"""
        dlg = wxMessageDialog(self, text, title, flags)
        dlg.CenterOnParent()
        result = dlg.ShowModal()
        dlg.Destroy()
        return result

    def init_dialogs(self):
        """Initialize the dialog handling"""
        # The mainwindow maintains a dict mapping names to open
        # non-modal dialogs. The dialogs are put into this dict when
        # they're created and removed when they're closed
        self.dialogs = {}

    def add_dialog(self, name, dialog):
        if self.dialogs.has_key(name):
            raise RuntimeError(_("The Dialog named %s is already open") % name)
        self.dialogs[name] = dialog

    def dialog_open(self, name):
        return self.dialogs.has_key(name)

    def remove_dialog(self, name):
        del self.dialogs[name]

    def get_open_dialog(self, name):
        return self.dialogs.get(name)

    def view_position_changed(self):
        pos = self.canvas.CurrentPosition()
        if pos is not None:
            text = "(%10.10g, %10.10g)" % pos
        else:
            text = ""
        self.set_position_text(text)

    def set_position_text(self, text):
        """Set the statusbar text showing the current position.

        By default the text is shown in field 0 of the status bar.
        Override this method in derived classes to put it into a
        different field of the statusbar.
        """
        self.SetStatusText(text)

    def save_modified_session(self, can_veto = 1):
        """If the current session has been modified, ask the user
        whether to save it and do so if requested. Return the outcome of
        the dialog (either wxID_OK, wxID_CANCEL or wxID_NO). If the
        dialog wasn't run return wxID_NO.

        If the can_veto parameter is true (default) the dialog includes
        a cancel button, otherwise not.
        """
        if self.application.session.WasModified():
            flags = wxYES_NO | wxICON_QUESTION
            if can_veto:
                flags = flags | wxCANCEL
            result = self.RunMessageBox(_("Exit"),
                                        _("The session has been modified."
                                         " Do you want to save it?"),
                                        flags)
            if result == wxID_YES:
                self.SaveSession()
        else:
            result = wxID_NO
        return result

    def prepare_new_session(self):
        for d in self.dialogs.values():
            if not isinstance(d, tree.SessionTreeView):
                d.Close()

    def NewSession(self):
        self.save_modified_session()
        self.prepare_new_session()
        self.application.SetSession(create_empty_session())

    def OpenSession(self):
        self.save_modified_session()
        dlg = wxFileDialog(self, _("Open Session"), ".", "", "*.thuban", wxOPEN)
        if dlg.ShowModal() == wxID_OK:
            self.prepare_new_session()
            self.application.OpenSession(dlg.GetPath())
        dlg.Destroy()

    def SaveSession(self):
        if self.application.session.filename == None:
            self.SaveSessionAs()
        else:
            self.application.SaveSession()

    def SaveSessionAs(self):
        dlg = wxFileDialog(self, _("Save Session As"), ".", "",
                           "*.thuban", wxOPEN)
        if dlg.ShowModal() == wxID_OK:
            self.application.session.SetFilename(dlg.GetPath())
            self.application.SaveSession()
        dlg.Destroy()

    def Exit(self):
        self.Close(false)

    def OnClose(self, event):
        result = self.save_modified_session(can_veto = event.CanVeto())
        if result == wxID_CANCEL:
            event.Veto()
        else:
            # FIXME: it would be better to tie the unsubscription to
            # wx's destroy event, but that isn't implemented for wxGTK
            # yet.
            self.canvas.Unsubscribe(VIEW_POSITION, self.view_position_changed)
            self.Destroy()

    def SetMap(self, map):
        self.canvas.SetMap(map)

    def Map(self):
        """Return the map displayed by this mainwindow"""
        return self.canvas.Map()

    def ShowSessionTree(self):
        name = "session_tree"
        dialog = self.get_open_dialog(name)
        if dialog is None:
            dialog = tree.SessionTreeView(self, self.application, name)
            self.add_dialog(name, dialog)
            dialog.Show(True)
        else:
            # FIXME: bring dialog to front here
            pass


    def About(self):
        self.RunMessageBox(_("About"),
                           _("Thuban v%s\n"
                            #"Build Date: %s\n"
                            "\n"
                            "Thuban is a program for\n"
                            "exploring geographic data.\n"
                            "Copyright (C) 2001-2003 Intevation GmbH.\n"
                            "Thuban is licensed under the GNU GPL" 
                           % __ThubanVersion__), #__BuildDate__)),
                           wxOK | wxICON_INFORMATION)

    def AddLayer(self):
        dlg = wxFileDialog(self, _("Select a data file"), ".", "", "*.*",
                           wxOPEN)
        if dlg.ShowModal() == wxID_OK:
            filename = dlg.GetPath()
            title = os.path.splitext(os.path.basename(filename))[0]
            layer = Layer(title, filename)
            map = self.canvas.Map()
            has_layers = map.HasLayers()
            try:
                map.AddLayer(layer)
            except IOError:
                # the layer couldn't be opened
                self.RunMessageBox(_("Add Layer"),
                                   _("Can't open the file '%s'.") % filename)
            else:
                if not has_layers:
                    # if we're adding a layer to an empty map, for the
                    # new map to the window
                    self.canvas.FitMapToWindow()
        dlg.Destroy()

    def RemoveLayer(self):
        layer = self.current_layer()
        if layer is not None:
            self.canvas.Map().RemoveLayer(layer)

    def CanRemoveLayer(self):
        """Return true if the currently selected layer can be deleted.

        If no layer is selected return false.

        The return value of this method determines whether the remove
        layer command is sensitive in menu.
        """
        layer = self.current_layer()
        if layer is not None:
            return self.canvas.Map().CanRemoveLayer(layer)
        return 0

    def RaiseLayer(self):
        layer = self.current_layer()
        if layer is not None:
            self.canvas.Map().RaiseLayer(layer)

    def LowerLayer(self):
        layer = self.current_layer()
        if layer is not None:
            self.canvas.Map().LowerLayer(layer)

    def current_layer(self):
        """Return the currently selected layer.

        If no layer is selected, return None
        """
        return self.interactor.SelectedLayer()

    def has_selected_layer(self):
        """Return true if a layer is currently selected"""
        return self.interactor.HasSelectedLayer()

    def choose_color(self):
        """Run the color selection dialog and return the selected color.

        If the user cancels, return None.
        """
        dlg = wxColourDialog(self)
        color = None
        if dlg.ShowModal() == wxID_OK:
            data = dlg.GetColourData()
            wxc = data.GetColour()
            color = Color(wxc.Red() / 255.0,
                          wxc.Green() / 255.0,
                          wxc.Blue() / 255.0)
        dlg.Destroy()
        return color

    def LayerFillColor(self):
        layer = self.current_layer()
        if layer is not None:
            color = self.choose_color()
            if color is not None:
                layer.GetClassification().SetDefaultFill(color)

    def LayerTransparentFill(self):
        layer = self.current_layer()
        if layer is not None:
            layer.GetClassification().SetDefaultFill(Color.None)

    def LayerOutlineColor(self):
        layer = self.current_layer()
        if layer is not None:
            color = self.choose_color()
            if color is not None:
                layer.GetClassification().SetDefaultLineColor(color)

    def LayerNoOutline(self):
        layer = self.current_layer()
        if layer is not None:
            layer.GetClassification().SetDefaultLineColor(Color.None)

    def HideLayer(self):
        layer = self.current_layer()
        if layer is not None:
            layer.SetVisible(0)
        
    def ShowLayer(self):
        layer = self.current_layer()
        if layer is not None:
            layer.SetVisible(1)

    def LayerShowTable(self):
        layer = self.current_layer()
        if layer is not None:
            table = layer.table
            name = "table_view" + str(id(table))
            dialog = self.get_open_dialog(name)
            if dialog is None:
                dialog = tableview.LayerTableFrame(self, self.interactor, name,
                                                   _("Table: %s") % layer.Title(),
                                                   layer, table)
                self.add_dialog(name, dialog)
                dialog.Show(true)
            else:
                # FIXME: bring dialog to front here
                pass

    def Projection(self):
        map = self.canvas.Map()
        proj = map.projection
        if proj is None:
            proj4Dlg = proj4dialog.Proj4Dialog(NULL, None, map.BoundingBox())
        else:
            proj4Dlg = proj4dialog.Proj4Dialog(NULL, map.projection.params,
                                               map.BoundingBox())
        if proj4Dlg.ShowModal() == wxID_OK:
            params = proj4Dlg.GetParams()
            if params is not None:
                proj = Projection(params) 
            else:
                proj = None
            map.SetProjection(proj)
        proj4Dlg.Destroy()

    def Classify(self):

        #
        # the menu option for this should only be available if there
        # is a current layer, so we don't need to check if the 
        # current layer is None
        #

        layer = self.current_layer()
        name = "classifier" + str(id(layer))
        dialog = self.get_open_dialog(name)

        if dialog is None:
            dialog = classifier.Classifier(self, self.interactor,
                                           name, self.current_layer())
            self.add_dialog(name, dialog)
            dialog.Show()

    def ZoomInTool(self):
        self.canvas.ZoomInTool()

    def ZoomOutTool(self):
        self.canvas.ZoomOutTool()

    def PanTool(self):
        self.canvas.PanTool()

    def IdentifyTool(self):
        self.canvas.IdentifyTool()
        self.identify_view_on_demand(None, None)

    def LabelTool(self):
        self.canvas.LabelTool()

    def FullExtent(self):
        self.canvas.FitMapToWindow()

    def PrintMap(self):
        self.canvas.Print()

    def identify_view_on_demand(self, layer, shape):
        name = "identify_view"
        if self.canvas.CurrentTool() == "IdentifyTool":
            if not self.dialog_open(name):
                dialog = identifyview.IdentifyView(self, self.interactor, name)
                self.add_dialog(name, dialog)
                dialog.Show(true)
            else:
                # FIXME: bring dialog to front?
                pass

#
# Define all the commands available in the main window
#


# Helper functions to define common command implementations
def call_method(context, methodname, *args):
    """Call the mainwindow's method methodname with args *args"""
    apply(getattr(context.mainwindow, methodname), args)

def _method_command(name, title, method, helptext = "",
                    icon = "", sensitive = None):
    """Add a command implemented by a method of the mainwindow object"""
    registry.Add(Command(name, title, call_method, args=(method,),
                         helptext = helptext, icon = icon,
                         sensitive = sensitive))

def make_check_current_tool(toolname):
    """Return a function that tests if the currently active tool is toolname

    The returned function can be called with the context and returns
    true iff the currently active tool's name is toolname. It's directly
    usable as the 'checked' callback of a command.
    """
    def check_current_tool(context, name=toolname):
        return context.mainwindow.canvas.CurrentTool() == name
    return check_current_tool

def _tool_command(name, title, method, toolname, helptext = "",
                  icon = "", sensitive = None):
    """Add a tool command"""
    registry.Add(ToolCommand(name, title, call_method, args=(method,),
                             helptext = helptext, icon = icon,
                             checked = make_check_current_tool(toolname),
                             sensitive = sensitive))

def _has_selected_layer(context):
    """Return true if a layer is selected in the context"""
    return context.mainwindow.has_selected_layer()

def _can_remove_layer(context):
    return context.mainwindow.CanRemoveLayer()

def _has_tree_window_shown(context):
    """Return true if the tree window is shown"""
    return context.mainwindow.get_open_dialog("session_tree") is None

def _has_visible_map(context):
    """Return true iff theres a visible map in the mainwindow.

    A visible map is a map with at least one visible layer."""
    map = context.mainwindow.Map()
    if map is not None:
        for layer in map.Layers():
            if layer.Visible():
                return 1
    return 0


# File menu
_method_command("new_session", _("&New Session"), "NewSession")
_method_command("open_session", _("&Open Session"), "OpenSession")
_method_command("save_session", _("&Save Session"), "SaveSession")
_method_command("save_session_as", _("Save Session &As"), "SaveSessionAs")
_method_command("show_session_tree", _("Show Session &Tree"), "ShowSessionTree",
                sensitive = _has_tree_window_shown)
_method_command("exit", _("E&xit"), "Exit")

# Help menu
_method_command("help_about", _("&About"), "About")


# Map menu
_method_command("map_projection", _("Pro&jection"), "Projection")

_tool_command("map_zoom_in_tool", _("&Zoom in"), "ZoomInTool", "ZoomInTool",
              helptext = _("Switch to map-mode 'zoom-in'"), icon = "zoom_in",
              sensitive = _has_visible_map)
_tool_command("map_zoom_out_tool", _("Zoom &out"), "ZoomOutTool", "ZoomOutTool",
              helptext = _("Switch to map-mode 'zoom-out'"), icon = "zoom_out",
              sensitive = _has_visible_map)
_tool_command("map_pan_tool", _("&Pan"), "PanTool", "PanTool",
              helptext = _("Switch to map-mode 'pan'"), icon = "pan",
              sensitive = _has_visible_map)
_tool_command("map_identify_tool", _("&Identify"), "IdentifyTool",
              "IdentifyTool",
              helptext = _("Switch to map-mode 'identify'"), icon = "identify",
              sensitive = _has_visible_map)
_tool_command("map_label_tool", _("&Label"), "LabelTool", "LabelTool",
              helptext = _("Add/Remove labels"), icon = "label",
              sensitive = _has_visible_map)
_method_command("map_full_extent", _("&Full extent"), "FullExtent",
               helptext = _("Full Extent"), icon = "fullextent",
              sensitive = _has_visible_map)
_method_command("map_print", _("Prin&t"), "PrintMap",
                helptext = _("Print the map"))

# Layer menu
_method_command("layer_add", _("&Add Layer"), "AddLayer",
                helptext = _("Add a new layer to active map"))
_method_command("layer_remove", _("&Remove Layer"), "RemoveLayer",
                helptext = _("Remove selected layer(s)"),
                sensitive = _can_remove_layer)
_method_command("layer_fill_color", _("&Fill Color"), "LayerFillColor",
                helptext = _("Set the fill color of selected layer(s)"),
                sensitive = _has_selected_layer)
_method_command("layer_transparent_fill", _("&Transparent Fill"),
                "LayerTransparentFill",
                helptext = _("Do not fill the selected layer(s)"),
                sensitive = _has_selected_layer)
_method_command("layer_outline_color", _("&Outline Color"), "LayerOutlineColor",
                helptext = _("Set the outline color of selected layer(s)"),
                sensitive = _has_selected_layer)
_method_command("layer_no_outline", _("&No Outline"), "LayerNoOutline",
                helptext= _("Do not draw the outline of the selected layer(s)"),
                sensitive = _has_selected_layer)
_method_command("layer_raise", _("&Raise"), "RaiseLayer",
                helptext = _("Raise selected layer(s)"),
                sensitive = _has_selected_layer)
_method_command("layer_lower", _("&Lower"), "LowerLayer",
                helptext = _("Lower selected layer(s)"),
                sensitive = _has_selected_layer)
_method_command("layer_show", _("&Show"), "ShowLayer",
                helptext = _("Make selected layer(s) visible"),
                sensitive = _has_selected_layer)
_method_command("layer_hide", _("&Hide"), "HideLayer",
                helptext = _("Make selected layer(s) unvisible"),
                sensitive = _has_selected_layer)
_method_command("layer_show_table", _("Show Ta&ble"), "LayerShowTable",
                helptext = _("Show the selected layer's table"),
                sensitive = _has_selected_layer)

_method_command("layer_classifier", _("Classify"), "Classify",
                sensitive = _has_selected_layer)

# the menu structure
main_menu = Menu("<main>", "<main>",
                 [Menu("file", _("&File"),
                       ["new_session", "open_session", None,
                        "save_session", "save_session_as", None,
                        "show_session_tree", None,
                        "exit"]),
                  Menu("map", _("&Map"),
                       ["layer_add", "layer_remove",
                        None,
                        "map_projection",
                        None,
                        "map_zoom_in_tool", "map_zoom_out_tool",
                        "map_pan_tool", "map_identify_tool", "map_label_tool",
                        None,
                        "map_full_extent",
                        None,
                        "map_print"]),
                  Menu("layer", _("&Layer"),
                       ["layer_fill_color", "layer_transparent_fill",
                        "layer_outline_color", "layer_no_outline",
                        None,
                        "layer_raise", "layer_lower",
                        None,
                        "layer_show", "layer_hide",
                        None,
                        "layer_show_table",
                        None,
                        "layer_classifier"]),
                  Menu("help", _("&Help"),
                       ["help_about"])])

# the main toolbar

main_toolbar = Menu("<toolbar>", "<toolbar>",
                    ["map_zoom_in_tool", "map_zoom_out_tool", "map_pan_tool", 
                     "map_full_extent", None, 
                     "map_identify_tool", "map_label_tool"])
