# Copyright (c) 2001, 2003 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""Dialog for classifying how layers are displayed"""

__version__ = "$Revision: 1.28 $"

import copy

from Thuban.Model.table import FIELDTYPE_INT, FIELDTYPE_DOUBLE, \
     FIELDTYPE_STRING

from wxPython.wx import *
from wxPython.grid import *

from Thuban import _
from Thuban.common import *
from Thuban.UI.common import *

from Thuban.Model.classification import *

from Thuban.Model.color import Color

from Thuban.Model.layer import Layer, SHAPETYPE_ARC, SHAPETYPE_POLYGON, SHAPETYPE_POINT

from dialogs import NonModalDialog

# widget id's
ID_PROPERTY_SELECT = 4010
ID_CLASS_TABLE = 40011

ID_CLASSIFY_OK = 4001
ID_CLASSIFY_CANCEL = 4002
ID_CLASSIFY_ADD = 4003
ID_CLASSIFY_GENRANGE = 4004
ID_CLASSIFY_REMOVE = 4005
ID_CLASSIFY_MOVEUP = 4006
ID_CLASSIFY_MOVEDOWN = 4007
ID_CLASSIFY_APPLY = 4008

# table columns
COL_SYMBOL = 0
COL_VALUE  = 1
COL_LABEL  = 2

# indices into the client data lists in Classifier.fields
FIELD_CLASS = 0
FIELD_TYPE = 1
FIELD_NAME = 2

#
# this is a silly work around to ensure that the table that is
# passed into SetTable is the same that is returned by GetTable
#
import weakref
class ClassGrid(wxGrid):

    def __init__(self, parent):
        """Constructor.

        parent -- the parent window

        clazz -- the working classification that this grid should
                 use for display. 
        """

        #wxGrid.__init__(self, parent, ID_CLASS_TABLE, size = (340, 160))
        wxGrid.__init__(self, parent, ID_CLASS_TABLE)
        #self.SetTable(ClassTable(fieldData, layer.ShapeType(), self), True)

        EVT_GRID_CELL_LEFT_DCLICK(self, self._OnCellDClick)
        EVT_GRID_RANGE_SELECT(self, self._OnSelectedRange)
        EVT_GRID_SELECT_CELL(self, self._OnSelectedCell)

        self.currentSelection = []

    def CreateTable(self, clazz, shapeType):

        assert(isinstance(clazz, Classification))

        self.shapeType = shapeType
        table = self.GetTable()
        if table is None:
            w = self.GetDefaultColSize() * 3 + self.GetDefaultRowLabelSize() 
            h = self.GetDefaultRowSize() * 4 + self.GetDefaultColLabelSize()
            self.SetDimensions(-1, -1, w, h)
            self.SetSizeHints(w, h, -1, -1)
            self.SetTable(ClassTable(clazz, self.shapeType, self), True)
        else:
            table.Reset(clazz, self.shapeType)

        self.SetSelectionMode(wxGrid.wxGridSelectRows)
        self.ClearSelection()

    def GetCurrentSelection(self):
        """Return the currently highlighted rows as an increasing list
           of row numbers."""
        sel = copy.copy(self.currentSelection)
        sel.sort()
        return sel

    def SetCellRenderer(self, row, col):
        raise ValueError(_("Must not allow setting of renderer in ClassGrid!"))

    #
    # [Set|Get]Table is taken from http://wiki.wxpython.org
    # they are needed as a work around to ensure that the table
    # that is passed to SetTable is the one that is returned 
    # by GetTable.
    #
    def SetTable(self, object, *attributes): 
        self.tableRef = weakref.ref(object) 
        return wxGrid.SetTable(self, object, *attributes) 

    def GetTable(self): 
        try:
            return self.tableRef() 
        except:
            return None

    def DeleteSelectedRows(self):
        """Deletes all highlighted rows.
  
        If only one row is highlighted then after it is deleted the
        row that was below the deleted row is highlighted."""

        sel = self.GetCurrentSelection()

        # nothing to do
        if len(sel) == 0: return 

        # if only one thing is selected check if it is the default
        # data row, because we can't remove that
        if len(sel) == 1:
            #group = self.GetTable().GetValueAsCustom(sel[0], COL_SYMBOL, None)
            group = self.GetTable().GetClassGroup(sel[0])
            if isinstance(group, ClassGroupDefault):
                wxMessageDialog(self, 
                                "The Default group cannot be removed.",
                                style = wxOK | wxICON_EXCLAMATION).ShowModal()
                return
        

        self.ClearSelection()

        # we need to remove things from the bottom up so we don't
        # change the indexes of rows that will be deleted next
        sel.reverse()

        #
        # actually remove the rows
        #
        table = self.GetTable()
        for row in sel:
            table.DeleteRows(row)

        #
        # if there was only one row selected highlight the row
        # that was directly below it, or move up one if the
        # deleted row was the last row.
        #
        if len(sel) == 1:
            r = sel[0]
            if r > self.GetNumberRows() - 1:
                r = self.GetNumberRows() - 1
            self.SelectRow(r)
        
#
# XXX: This isn't working, and there is no way to deselect rows wxPython!
#
#   def DeselectRow(self, row):
#       self.ProcessEvent(
#           wxGridRangeSelectEvent(-1, 
#                                  wxEVT_GRID_RANGE_SELECT,
#                                  self,
#                                  (row, row), (row, row),
#                                  sel = False))

    def _OnCellDClick(self, event):
        """Handle a double on a cell."""

        r = event.GetRow()
        c = event.GetCol()
        if c == COL_SYMBOL:
            prop = self.GetTable().GetValueAsCustom(r, c, None)
            #prop = group.GetProperties()

            # get a new ClassGroupProperties object and copy the 
            # values over to our current object
            propDlg = SelectPropertiesDialog(NULL, prop, self.shapeType)
            if propDlg.ShowModal() == wxID_OK:
                new_prop = propDlg.GetClassGroupProperties()
                #prop.SetProperties(new_prop)
                self.GetTable().SetValueAsCustom(r, c, None, new_prop)
            propDlg.Destroy()

    #
    # _OnSelectedRange() and _OnSelectedCell() were borrowed
    # from http://wiki.wxpython.org to keep track of which
    # cells are currently highlighted
    #
    def _OnSelectedRange(self, event): 
        """Internal update to the selection tracking list""" 
        if event.Selecting(): 
            for index in range( event.GetTopRow(), event.GetBottomRow()+1): 
                if index not in self.currentSelection: 
                    self.currentSelection.append( index ) 
        else: 
            for index in range( event.GetTopRow(), event.GetBottomRow()+1): 
                while index in self.currentSelection: 
                    self.currentSelection.remove( index ) 
        #self.ConfigureForSelection() 

        event.Skip() 

    def _OnSelectedCell( self, event ): 
        """Internal update to the selection tracking list""" 
        self.currentSelection = [ event.GetRow() ] 
        #self.ConfigureForSelection() 
        event.Skip() 

class ClassTable(wxPyGridTableBase):
    """Represents the underlying data structure for the grid."""

    NUM_COLS = 3

    __col_labels = [_("Symbol"), _("Value"), _("Label")]

    def __init__(self, clazz, shapeType, view = None):
        """Constructor.

        shapeType -- the type of shape that the layer uses

        view -- a wxGrid object that uses this class for its table
        """

        wxPyGridTableBase.__init__(self)

        self.SetView(view)
        self.tdata = []

        self.Reset(clazz, shapeType)

    def Reset(self, clazz, shapeType):
        """Reset the table with the given data.

        This is necessary because wxWindows does not allow a grid's
        table to change once it has been intially set and so we
        need a way of modifying the data.

        clazz -- the working classification that this table should
                 use for display. This may be different from the
                 classification in the layer.

        shapeType -- the type of shape that the layer uses
        """

        assert(isinstance(clazz, Classification))

        self.GetView().BeginBatch()

        self.fieldType = clazz.GetFieldType()
        self.shapeType = shapeType

        old_len = len(self.tdata)

        self.tdata = []

        #
        # copy the data out of the classification and into our
        # array
        #
        for p in clazz:
            np = copy.deepcopy(p)
            self.__SetRow(-1, np)

 
        self.__Modified(-1)

        self.__NotifyRowChanges(old_len, len(self.tdata))

            
        self.GetView().EndBatch()

    def __NotifyRowChanges(self, curRows, newRows):
        #
        # silly message processing for updates to the number of
        # rows and columns
        #
        if newRows > curRows:
            msg = wxGridTableMessage(self,
                        wxGRIDTABLE_NOTIFY_ROWS_APPENDED,
                        newRows - curRows)    # how many
            self.GetView().ProcessTableMessage(msg)
            self.GetView().FitInside()
        elif newRows < curRows:
            msg = wxGridTableMessage(self,
                        wxGRIDTABLE_NOTIFY_ROWS_DELETED,
                        curRows - newRows,    # position
                        curRows - newRows)    # how many
            self.GetView().ProcessTableMessage(msg)
            self.GetView().FitInside()

    def __SetRow(self, row, group):
        """Set a row's data to that of the group.

        The table is considered modified after this operation.

        row -- if row is -1 or greater than the current number of rows
               then group is appended to the end.
        """

        # either append or replace
        if row == -1 or row >= self.GetNumberRows():
            self.tdata.append(group)
        else:
            self.tdata[row] = group

        self.__Modified()

    def GetColLabelValue(self, col):
        """Return the label for the given column."""
        return self.__col_labels[col]

    def GetRowLabelValue(self, row):
        """Return the label for the given row."""

        group = self.tdata[row]
        if isinstance(group, ClassGroupDefault):   return _("Default")
        if isinstance(group, ClassGroupSingleton): return _("Singleton")
        if isinstance(group, ClassGroupRange):     return _("Range")
        if isinstance(group, ClassGroupMap):       return _("Map")

        assert(False) # shouldn't get here
        return _("")

    def GetNumberRows(self):
        """Return the number of rows."""
        return len(self.tdata)

    def GetNumberCols(self):
        """Return the number of columns."""
        return self.NUM_COLS

    def IsEmptyCell(self, row, col):
        """Determine if a cell is empty. This is always false."""
        return False

    def GetValue(self, row, col):
        """Return the object that is used to represent the given
           cell coordinates. This may not be a string."""
        return self.GetValueAsCustom(row, col, None)

    def SetValue(self, row, col, value):
        """Assign 'value' to the cell specified by 'row' and 'col'.

        The table is considered modified after this operation.
        """

        self.SetValueAsCustom(row, col, None, value)
        self.__Modified()
       
    def GetValueAsCustom(self, row, col, typeName):
        """Return the object that is used to represent the given
           cell coordinates. This may not be a string.
 
        typeName -- unused, but needed to overload wxPyGridTableBase
        """

        group = self.tdata[row]

        if col == COL_SYMBOL:
            return group.GetProperties()

        if col == COL_LABEL:
            return group.GetLabel()

        # col must be COL_VALUE
        assert(col == COL_VALUE)

        if isinstance(group, ClassGroupDefault):
            return _("DEFAULT")
        elif isinstance(group, ClassGroupSingleton):
            return group.GetValue()
        elif isinstance(group, ClassGroupRange):
            return _("%s - %s") % (group.GetMin(), group.GetMax())

        assert(False) # shouldn't get here
        return None

    def __ParseInput(self, value):
        """Try to determine what kind of input value is 
           (string, number, or range)

        Returns a tuple of length one if there is a single
        value, or of length two if it is a range.
        """

        type = self.fieldType

        if type == FIELDTYPE_STRING:
            return (value,)
        elif type == FIELDTYPE_INT or type == FIELDTYPE_DOUBLE:

            if type == FIELDTYPE_INT:
                conv = lambda p: int(float(p))
            else:
                conv = lambda p: p

            #
            # first try to take the input as a single number
            # if there's an exception try to break it into
            # a range seperated by a '-'. take care to ignore
            # a leading '-' as that could be for a negative number.
            # then try to parse the individual parts. if there
            # is an exception here, let it pass up to the calling
            # function.
            #
            try:
                return (conv(Str2Num(value)),)
            except ValueError:
                i = value.find('-')
                if i == 0:
                    i = value.find('-', 1)

                return (conv(Str2Num(value[:i])), conv(Str2Num(value[i+1:])))

        assert(False) # shouldn't get here
        return (0,)
            

    def SetValueAsCustom(self, row, col, typeName, value):
        """Set the cell specified by 'row' and 'col' to 'value'.

        If column represents the value column, the input is parsed
        to determine if a string, number, or range was entered.
        A new ClassGroup may be created if the type of data changes.

        The table is considered modified after this operation.

        typeName -- unused, but needed to overload wxPyGridTableBase
        """

        assert(col >= 0 and col < self.GetNumberCols())
        assert(row >= 0 and row < self.GetNumberRows())

        group = self.tdata[row]

        mod = True # assume the data will change

        if col == COL_SYMBOL:
            group.SetProperties(value)
        elif col == COL_LABEL:
            group.SetLabel(value)
        elif col == COL_VALUE:
            if isinstance(group, ClassGroupDefault):
                # not allowed to modify the default value 
                pass
            elif isinstance(group, ClassGroupMap):
                # something special
                pass
            else: # SINGLETON, RANGE
                try:
                    dataInfo = self.__ParseInput(value)
                except ValueError:
                    # bad input, ignore the request
                    mod = False
                else:

                    changed = False
                    ngroup = group
                    props = group.GetProperties()

                    #
                    # try to update the values, which may include
                    # changing the underlying group type if the
                    # group was a singleton and a range was entered
                    #
                    if len(dataInfo) == 1:
                        if not isinstance(group, ClassGroupSingleton):
                            ngroup = ClassGroupSingleton(prop = props)
                            changed = True
                        ngroup.SetValue(dataInfo[0])
                    elif len(dataInfo) == 2:
                        if not isinstance(group, ClassGroupRange):
                            ngroup = ClassGroupRange(prop = props)
                            changed = True
                        ngroup.SetRange(dataInfo[0], dataInfo[1])
                    else:
                        assert(False)
                        pass

                    if changed:
                        ngroup.SetLabel(group.GetLabel())
                        self.SetClassGroup(row, ngroup)
        else:
            assert(False) # shouldn't be here
            pass

        if mod:
            self.__Modified()
            self.GetView().Refresh()

    def GetAttr(self, row, col, someExtraParameter):
        """Returns the cell attributes"""

        attr = wxGridCellAttr()
        #attr = wxPyGridTableBase.GetAttr(self, row, col, someExtraParameter)
 
        if col == COL_SYMBOL:
            # we need to create a new renderer each time, because 
            # SetRenderer takes control of the parameter
            attr.SetRenderer(ClassRenderer(self.shapeType))
            attr.SetReadOnly()

        return attr

    def GetClassGroup(self, row):
        """Return the ClassGroup object representing row 'row'."""

        return self.tdata[row] # self.GetValueAsCustom(row, COL_SYMBOL, None)

    def SetClassGroup(self, row, group):
        self.__SetRow(row, group)
        self.GetView().Refresh()

    def __Modified(self, mod = True):
        """Adjust the modified flag.

        mod -- if -1 set the modified flag to False, otherwise perform
               an 'or' operation with the current value of the flag and
               'mod'
        """

        if mod == -1:
            self.modified = False
        else:
            self.modified = mod or self.modified

    def IsModified(self):
        """True if this table is considered modified."""
        return self.modified

    def DeleteRows(self, pos, numRows = 1):
        """Deletes 'numRows' beginning at row 'pos'. 

        The row representing the default group is not removed.

        The table is considered modified if any rows are removed.
        """

        assert(pos >= 0)
        old_len = len(self.tdata)
        for row in range(pos, pos - numRows, -1):
            group = self.GetClassGroup(row)
            if not isinstance(group, ClassGroupDefault):
                self.tdata.pop(row)
                self.__Modified()
    
        if self.IsModified():
            self.__NotifyRowChanges(old_len, len(self.tdata))

    def AppendRows(self, numRows = 1):
        """Append 'numRows' empty rows to the end of the table.

        The table is considered modified if any rows are appended.
        """

        old_len = len(self.tdata)
        for i in range(numRows):
            np = ClassGroupSingleton()
            self.__SetRow(-1, np)

        if self.IsModified():
            self.__NotifyRowChanges(old_len, len(self.tdata))


class Classifier(NonModalDialog):
    
    def __init__(self, parent, interactor, name, layer):
        NonModalDialog.__init__(self, parent, interactor, name, 
                                _("Classifier: %s") % layer.Title())

        panel = wxPanel(self, -1, size=(100, 100))

        self.layer = layer

        self.originalClass = self.layer.GetClassification()
        field = self.originalClass.GetField()
        fieldType = self.originalClass.GetFieldType()

        topBox = wxBoxSizer(wxVERTICAL)
        panelBox = wxBoxSizer(wxVERTICAL)

        #panelBox.Add(wxStaticText(panel, -1, _("Layer: %s") % layer.Title()),
            #0, wxALIGN_LEFT | wxALL, 4) 
        panelBox.Add(wxStaticText(panel, -1, 
                                _("Layer Type: %s") % layer.ShapeType()),
            0, wxALIGN_LEFT | wxALL, 4) 


        #
        # make field combo box
        #
        self.fields = wxComboBox(panel, ID_PROPERTY_SELECT, "",
                                     style = wxCB_READONLY)

        self.num_cols = layer.table.field_count()
        # just assume the first field in case one hasn't been
        # specified in the file.
        self.__cur_field = 0 

        self.fields.Append("<None>")
        self.fields.SetClientData(0, None)

        for i in range(self.num_cols):
            type, name, len, decc = layer.table.field_info(i)
            self.fields.Append(name)

            if name == field:
                self.__cur_field = i + 1
                self.fields.SetClientData(i + 1, self.originalClass)
            else:
                self.fields.SetClientData(i + 1, None)


        ###########

        self.fieldTypeText = wxStaticText(panel, -1, "")
        panelBox.Add(self.fieldTypeText, 0, wxGROW | wxALIGN_LEFT | wxALL, 4) 

        propertyBox = wxBoxSizer(wxHORIZONTAL)
        propertyBox.Add(wxStaticText(panel, -1, _("Field: ")),
            0, wxALIGN_LEFT | wxALL, 4)
        propertyBox.Add(self.fields, 1, wxGROW|wxALL, 4)
        EVT_COMBOBOX(self, ID_PROPERTY_SELECT, self._OnFieldSelect)

        panelBox.Add(propertyBox, 0, wxGROW, 4)

        ###########
        #
        # Classification data table
        #

        controlBox = wxBoxSizer(wxHORIZONTAL)

        self.classGrid = ClassGrid(panel)
        self.__SetGridTable(self.__cur_field)

        controlBox.Add(self.classGrid, 1, wxGROW, 0)

        ###########
        #
        # Control buttons:
        #
        self.controlButtons = []

        controlButtonBox = wxBoxSizer(wxVERTICAL)

        button = wxButton(panel, ID_CLASSIFY_ADD, _("Add"))
        controlButtonBox.Add(button, 0, wxGROW | wxALL, 4)
        self.controlButtons.append(button)

        #button = wxButton(panel, ID_CLASSIFY_GENRANGE, _("Generate Ranges"))
        #controlButtonBox.Add(button, 0, wxGROW | wxALL, 4)
        #self.controlButtons.append(button)

        button = wxButton(panel, ID_CLASSIFY_MOVEUP, _("Move Up"))
        controlButtonBox.Add(button, 0, wxGROW | wxALL, 4)
        self.controlButtons.append(button)

        button = wxButton(panel, ID_CLASSIFY_MOVEDOWN, _("Move Down"))
        controlButtonBox.Add(button, 0, wxGROW | wxALL, 4)
        self.controlButtons.append(button)

        controlButtonBox.Add(60, 20, 0, wxGROW | wxALL | wxALIGN_BOTTOM, 4)

        button = wxButton(panel, ID_CLASSIFY_REMOVE, _("Remove"))
        controlButtonBox.Add(button, 0, wxGROW | wxALL | wxALIGN_BOTTOM, 4)
        self.controlButtons.append(button)

        controlBox.Add(controlButtonBox, 0, wxGROW, 10)
        panelBox.Add(controlBox, 1, wxGROW, 10)

        EVT_BUTTON(self, ID_CLASSIFY_ADD, self._OnAdd)
        EVT_BUTTON(self, ID_CLASSIFY_REMOVE, self._OnRemove)
        EVT_BUTTON(self, ID_CLASSIFY_GENRANGE, self._OnGenRange)
        EVT_BUTTON(self, ID_CLASSIFY_MOVEUP, self._OnMoveUp)
        EVT_BUTTON(self, ID_CLASSIFY_MOVEDOWN, self._OnMoveDown)

        ###########

        buttonBox = wxBoxSizer(wxHORIZONTAL)
        buttonBox.Add(wxButton(panel, ID_CLASSIFY_OK, _("OK")),
                      0, wxALL, 4)
        buttonBox.Add(60, 20, 0, wxALL, 4)
        buttonBox.Add(wxButton(panel, ID_CLASSIFY_APPLY, _("Apply")),
                      0, wxALL, 4)
        buttonBox.Add(60, 20, 0, wxALL, 4)
        buttonBox.Add(wxButton(panel, ID_CLASSIFY_CANCEL, _("Cancel")),
                      0, wxALL, 4)
        panelBox.Add(buttonBox, 0, wxALIGN_CENTER_HORIZONTAL|wxALIGN_BOTTOM, 0)

        EVT_BUTTON(self, ID_CLASSIFY_OK, self._OnOK)
        EVT_BUTTON(self, ID_CLASSIFY_APPLY, self._OnApply)
        EVT_BUTTON(self, ID_CLASSIFY_CANCEL, self._OnCancel)

        ###########

        self.fields.SetSelection(self.__cur_field)
        self.__SelectField(self.__cur_field)

        panel.SetAutoLayout(True)
        panel.SetSizer(panelBox)
        panelBox.SetSizeHints(panel) 

	topBox.Add(panel, 1, wxGROW, 0)
        panelBox.SetSizeHints(self)
        self.SetAutoLayout(True)
        self.SetSizer(topBox)

    def __BuildClassification(self, fieldIndex):

        numRows = self.classGrid.GetNumberRows()
        assert(numRows > 0) # there should always be a default row

        clazz = Classification()
        if fieldIndex == 0:
            fieldName = None
            fieldType = None
        else:
            fieldName = self.fields.GetString(fieldIndex)
            fieldType = self.layer.GetFieldType(fieldName)

        clazz.SetField(fieldName)
        clazz.SetFieldType(fieldType)


        table = self.classGrid.GetTable()
        clazz.SetDefaultGroup(table.GetClassGroup(0))

        for i in range(1, numRows):
            clazz.AddGroup(table.GetClassGroup(i))

        return clazz

    def __SetGridTable(self, fieldIndex):

        clazz = self.fields.GetClientData(fieldIndex)

        if clazz is None:
            clazz = Classification()
            clazz.SetDefaultGroup(
                ClassGroupDefault(
                    self.layer.GetClassification().
                               GetDefaultGroup().GetProperties()))

            fieldName = self.fields.GetString(fieldIndex)
            fieldType = self.layer.GetFieldType(fieldName)
            clazz.SetFieldType(fieldType)
                
        self.classGrid.CreateTable(clazz, self.layer.ShapeType())



    type2string = {None:             _("None"),
                   FIELDTYPE_STRING: _("Text"),
                   FIELDTYPE_INT:    _("Integer"),
                   FIELDTYPE_DOUBLE: _("Decimal")}

    def __SetFieldTypeText(self, fieldIndex):
        fieldName = self.fields.GetString(fieldIndex)
        fieldType = self.layer.GetFieldType(fieldName)

        assert(Classifier.type2string.has_key(fieldType))

        text = Classifier.type2string[fieldType]

        self.fieldTypeText.SetLabel(_("Field Type: %s") % text)

    def __SelectField(self, newIndex, oldIndex = -1):

        assert(oldIndex >= -1)

        if oldIndex != -1:
            clazz = self.__BuildClassification(oldIndex)
            self.fields.SetClientData(oldIndex, clazz)

        self.__SetGridTable(newIndex)

        enabled = newIndex != 0

        for b in self.controlButtons:
            b.Enable(enabled)

        self.__SetFieldTypeText(newIndex)

 
    def _OnFieldSelect(self, event): 
        index = self.fields.GetSelection()
        self.__SelectField(index, self.__cur_field)
        self.__cur_field = index

    def _OnApply(self, event):
        """Put the data from the table into a new Classification and hand
           it to the layer.
        """

        clazz = self.fields.GetClientData(self.__cur_field)

        #
        # only build the classification if there wasn't one to
        # to begin with or it has been modified
        #
        if clazz is None or self.classGrid.GetTable().IsModified():
            clazz = self.__BuildClassification(self.__cur_field)

        self.layer.SetClassification(clazz)

    def _OnOK(self, event):
        self._OnApply(event)
        self.OnClose(event)

    def _OnCancel(self, event):
        """The layer's current classification stays the same."""
        self.layer.SetClassification(self.originalClass)
        self.OnClose(event)

    def _OnAdd(self, event): 
        self.classGrid.AppendRows()

    def _OnRemove(self, event):
        self.classGrid.DeleteSelectedRows()

    def _OnGenRange(self, event):
        print "Classifier._OnGenRange()"

    def _OnMoveUp(self, event):
        sel = self.classGrid.GetCurrentSelection()

        if len(sel) == 1:
            i = sel[0]
            if i > 1:
                table = self.classGrid.GetTable()
                x = table.GetClassGroup(i - 1)
                y = table.GetClassGroup(i)
                table.SetClassGroup(i - 1, y)
                table.SetClassGroup(i, x)
                self.classGrid.ClearSelection()
                self.classGrid.SelectRow(i - 1)

    def _OnMoveDown(self, event):
        sel = self.classGrid.GetCurrentSelection()

        if len(sel) == 1:
            i = sel[0]
            table = self.classGrid.GetTable()
            if 0 < i < table.GetNumberRows() - 1:
                x = table.GetClassGroup(i)
                y = table.GetClassGroup(i + 1)
                table.SetClassGroup(i, y)
                table.SetClassGroup(i + 1, x)
                self.classGrid.ClearSelection()
                self.classGrid.SelectRow(i + 1)


ID_SELPROP_OK = 4001
ID_SELPROP_CANCEL = 4002
ID_SELPROP_SPINCTRL = 4002
ID_SELPROP_PREVIEW = 4003
ID_SELPROP_STROKECLR = 4004
ID_SELPROP_FILLCLR = 4005
ID_SELPROP_STROKECLRTRANS = 4006
ID_SELPROP_FILLCLRTRANS = 4007

class SelectPropertiesDialog(wxDialog):

    def __init__(self, parent, prop, shapeType):
        wxDialog.__init__(self, parent, -1, _("Select Properties"),
                          style = wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER)

        self.prop = ClassGroupProperties(prop)

        topBox = wxBoxSizer(wxVERTICAL)

        itemBox = wxBoxSizer(wxHORIZONTAL)

        # preview box
        previewBox = wxBoxSizer(wxVERTICAL)
        previewBox.Add(wxStaticText(self, -1, _("Preview:")),
            0, wxALIGN_LEFT | wxALL, 4)
        self.previewer = ClassDataPreviewer(None, self.prop, shapeType, 
                                            self, ID_SELPROP_PREVIEW, (40, 40))
        previewBox.Add(self.previewer, 1, wxGROW, 15)

        itemBox.Add(previewBox, 1, wxALIGN_LEFT | wxALL | wxGROW, 0)

        # control box
        ctrlBox = wxBoxSizer(wxVERTICAL)

        lineColorBox = wxBoxSizer(wxHORIZONTAL)
        lineColorBox.Add(
            wxButton(self, ID_SELPROP_STROKECLR, _("Change Line Color")),
            1, wxALL | wxGROW, 4)
        EVT_BUTTON(self, ID_SELPROP_STROKECLR, self._OnChangeLineColor)

        lineColorBox.Add(
            wxButton(self, ID_SELPROP_STROKECLRTRANS, _("Transparent")),
            1, wxALL | wxGROW, 4)
        EVT_BUTTON(self, ID_SELPROP_STROKECLRTRANS,
                   self._OnChangeLineColorTrans)

        ctrlBox.Add(lineColorBox, 0, 
                    wxALIGN_CENTER_HORIZONTAL | wxALL | wxGROW, 4)

        if shapeType != SHAPETYPE_ARC:
            fillColorBox = wxBoxSizer(wxHORIZONTAL)
            fillColorBox.Add(
                wxButton(self, ID_SELPROP_FILLCLR, _("Change Fill Color")),
                1, wxALL | wxGROW, 4)
            EVT_BUTTON(self, ID_SELPROP_FILLCLR, self._OnChangeFillColor)
            fillColorBox.Add(
                wxButton(self, ID_SELPROP_FILLCLRTRANS, _("Transparent")),
                1, wxALL | wxGROW, 4)
            EVT_BUTTON(self, ID_SELPROP_FILLCLRTRANS,
                       self._OnChangeFillColorTrans)
            ctrlBox.Add(fillColorBox, 0, 
                        wxALIGN_CENTER_HORIZONTAL | wxALL | wxGROW, 4)

        spinBox = wxBoxSizer(wxHORIZONTAL)
        spinBox.Add(wxStaticText(self, -1, _("Line Width: ")),
                0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL | wxALL, 4) 
        self.spinCtrl = wxSpinCtrl(self, ID_SELPROP_SPINCTRL, 
                                   min=1, max=10, 
                                   value=str(prop.GetLineWidth()),
                                   initial=prop.GetLineWidth())

        EVT_SPINCTRL(self, ID_SELPROP_SPINCTRL, self._OnSpin)

        spinBox.Add(self.spinCtrl, 0, wxALIGN_LEFT | wxALL, 4) 

        ctrlBox.Add(spinBox, 0, wxALIGN_RIGHT | wxALL, 0)
        itemBox.Add(ctrlBox, 0, wxALIGN_RIGHT | wxALL | wxGROW, 0)
        topBox.Add(itemBox, 1, wxALIGN_LEFT | wxALL | wxGROW, 0)

        #
        # Control buttons:
        #
        buttonBox = wxBoxSizer(wxHORIZONTAL)
        buttonBox.Add(wxButton(self, ID_SELPROP_OK, _("OK")),
                      0, wxALL, 4)
        buttonBox.Add(wxButton(self, ID_SELPROP_CANCEL, _("Cancel")),
                      0, wxALL, 4)
        topBox.Add(buttonBox, 0, wxALIGN_CENTER_HORIZONTAL|wxALIGN_BOTTOM, 10)
                                                                                
        EVT_BUTTON(self, ID_SELPROP_OK, self._OnOK)
        EVT_BUTTON(self, ID_SELPROP_CANCEL, self._OnCancel)
                                                                                
        self.SetAutoLayout(True)
        self.SetSizer(topBox)
        topBox.Fit(self)
        topBox.SetSizeHints(self)

    def _OnOK(self, event):
        self.EndModal(wxID_OK)

    def _OnCancel(self, event):
        self.EndModal(wxID_CANCEL)

    def _OnSpin(self, event):
        self.prop.SetLineWidth(self.spinCtrl.GetValue())
        self.previewer.Refresh()

    def __GetColor(self, cur):
        dialog = wxColourDialog(self)
        dialog.GetColourData().SetColour(Color2wxColour(cur))
        ret = None
        if dialog.ShowModal() == wxID_OK:
            ret = wxColour2Color(dialog.GetColourData().GetColour())

        dialog.Destroy()

        return ret
        
    def _OnChangeLineColor(self, event):
        clr = self.__GetColor(self.prop.GetLineColor())
        if clr is not None:
            self.prop.SetLineColor(clr)
        self.previewer.Refresh() # XXX: work around, see ClassDataPreviewer

    def _OnChangeLineColorTrans(self, event):
        self.prop.SetLineColor(Color.None)
        self.previewer.Refresh() # XXX: work around, see ClassDataPreviewer
        
    def _OnChangeFillColor(self, event):
        clr = self.__GetColor(self.prop.GetFill())
        if clr is not None:
            self.prop.SetFill(clr)
        self.previewer.Refresh() # XXX: work around, see ClassDataPreviewer

    def _OnChangeFillColorTrans(self, event):
        self.prop.SetFill(Color.None)
        self.previewer.Refresh() # XXX: work around, see ClassDataPreviewer

    def GetClassGroupProperties(self):
        return self.prop


class ClassDataPreviewer(wxWindow):

    def __init__(self, rect, prop, shapeType, 
                       parent = None, id = -1, size = wxDefaultSize):
        if parent is not None:
            wxWindow.__init__(self, parent, id, size=size)
            EVT_PAINT(self, self._OnPaint)

        self.rect = rect
        self.prop = prop
        self.shapeType = shapeType

    def _OnPaint(self, event):
        dc = wxPaintDC(self)

        # XXX: this doesn't seem to be having an effect:
        dc.DestroyClippingRegion() 

        self.Draw(dc, None)

    def Draw(self, dc, rect, prop = None, shapeType = None):

        if prop is None: prop = self.prop
        if shapeType is None: shapeType = self.shapeType

        if rect is None:
            x = y = 0
            w, h = self.GetClientSizeTuple()
        else:
            x = rect.GetX()
            y = rect.GetY()
            w = rect.GetWidth()
            h = rect.GetHeight()

        stroke = prop.GetLineColor()
        if stroke is Color.None:
            pen = wxTRANSPARENT_PEN
        else:
            pen = wxPen(Color2wxColour(stroke),
                        prop.GetLineWidth(),
                        wxSOLID)

        stroke = prop.GetFill()
        if stroke is Color.None:
            brush = wxTRANSPARENT_BRUSH
        else:
            brush = wxBrush(Color2wxColour(stroke), wxSOLID)

        dc.SetPen(pen)
        dc.SetBrush(brush)

        if shapeType == SHAPETYPE_ARC:
            dc.DrawSpline([wxPoint(x, y + h),
                           wxPoint(x + w/2, y + h/4),
                           wxPoint(x + w/2, y + h/4*3),
                           wxPoint(x + w, y)])

        elif shapeType == SHAPETYPE_POINT or \
             shapeType == SHAPETYPE_POLYGON:

            dc.DrawCircle(x + w/2, y + h/2,
                          (min(w, h) - prop.GetLineWidth())/2)

class ClassRenderer(wxPyGridCellRenderer):

    def __init__(self, shapeType):
        wxPyGridCellRenderer.__init__(self)
        self.previewer = ClassDataPreviewer(None, None, shapeType)

    def Draw(self, grid, attr, dc, rect, row, col, isSelected):
        data = grid.GetTable().GetClassGroup(row)

        dc.SetClippingRegion(rect.GetX(), rect.GetY(), 
                             rect.GetWidth(), rect.GetHeight())
        dc.SetPen(wxPen(wxLIGHT_GREY))
        dc.SetBrush(wxBrush(wxLIGHT_GREY, wxSOLID))
        dc.DrawRectangle(rect.GetX(), rect.GetY(), 
                         rect.GetWidth(), rect.GetHeight())

        if not isinstance(data, ClassGroupMap):
            self.previewer.Draw(dc, rect, data.GetProperties())

        if isSelected:
            dc.SetPen(wxPen(wxColour(0 * 255, 0 * 255, 0 * 255),
                      4, wxSOLID))
            dc.SetBrush(wxTRANSPARENT_BRUSH)
            dc.DrawRectangle(rect.GetX(), rect.GetY(), 
                             rect.GetWidth(), rect.GetHeight())

        dc.DestroyClippingRegion()

