# Copyright (C) 2001, 2002, 2003 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Thuban's application object.
"""

__version__ = "$Revision: 1.14 $"

import sys, os
import traceback

from wxPython.wx import *

from Thuban.Lib.connector import Publisher

from Thuban import _
from Thuban.Model.session import create_empty_session
from Thuban.Model.save import save_session
from Thuban.Model.load import load_session
from Thuban.Model.messages import MAPS_CHANGED

import view
import tree
from interactor import Interactor
import mainwindow

from messages import SESSION_REPLACED



class ThubanApplication(wxApp, Publisher):

    """
    Thuban's application class.

    All wxWindows programs have to have an instance of an application
    class derived from wxApp. In Thuban the application class holds
    references to the main window, the session and the interactor.
    """

    def OnInit(self):
        self.splash = self.splash_screen()
        if self.splash is not None:
            self.splash.Show()
        self.read_startup_files()
        self.interactor = Interactor(None)
        self.top = self.CreateMainWindow()
        self.SetTopWindow(self.top)
        if self.splash is None:
            self.ShowMainWindow()
        self.session = None
        self.create_session()
        return True

    def OnExit(self):
        """Clean up code.

        Extend this in derived classes if needed.
        """
        self.session.Destroy()
        self.interactor.Destroy()
        Publisher.Destroy(self)

    def MainLoop(self):
        """Call the inherited MainLoop method and then call OnExit.

        In wxPython OnExit isn't called automatically, unfortunately, so
        we do it here.
        """
        wxApp.MainLoop(self)
        self.OnExit()

    def read_startup_files(self):
        """Read the startup files."""
        # for now the startup file is ~/.thuban/thubanstart.py
        dir =os.path.expanduser("~/.thuban")
        if os.path.isdir(dir):
            sys.path.append(dir)
            try:
                import thubanstart
            except ImportError:
                tb = sys.exc_info()[2]
                try:
                    if tb.tb_next is not None:
                        # The ImportError exception was raised from
                        # inside the thubanstart module.
                        sys.stderr.write(_("Cannot import the thubanstart"
                                         "module\n"))
                        traceback.print_exc(None, sys.stderr)
                    else:
                        # There's no thubanstart module.
                        sys.stderr.write(_("No thubanstart module available\n"))
                finally:
                    # make sure we delete the traceback object,
                    # otherwise there's be circular references involving
                    # the current stack frame
                    del tb
            except:
                sys.stderr.write(_("Cannot import the thubanstart module\n"))
                traceback.print_exc(None, sys.stderr)
        else:
            # There's no .thuban directory
            sys.stderr.write(_("No ~/.thuban directory\n"))

    def splash_screen(self):
        """Create and return a splash screen.

        This method is called by OnInit to determine whether the
        application should have a splashscreen. If the application
        should display a splash screen override this method in a derived
        class and have it create and return the wxSplashScreen instance.
        The implementation of this method in the derived class should
        also arranged for ShowMainWindow to be called.

        The default implementation simply returns None so that no splash
        screen is shown and ShowMainWindow will be called automatically.
        """
        return None

    def ShowMainWindow(self):
        """Show the main window

        Normally this method is automatically called by OnInit to show
        the main window. However, if the splash_screen method has
        returned a splashscreen it is expected that the derived class
        also arranges for ShowMainWindow to be called at the appropriate
        time.
        """
        self.top.Show(True)
       
    def CreateMainWindow(self):
        """Create and return the main window for the application.

        Override this in subclasses to instantiate the Thuban mainwindow
        with different parameters or to use a different class for the
        main window.

        when this method is called by OnInit self.interactor (to be used
        for the interactor argument of the standard Thuban main window
        class) has already been instantiated.
        """
        msg = (_("This is the wxPython-based Graphical User Interface"
               " for exploring geographic data"))
        return mainwindow.MainWindow(NULL, -1, "Thuban", self, self.interactor,
                                     initial_message = msg)

    def Session(self):
        """Return the application's session object"""
        return self.session

    def SetSession(self, session):
        """Make session the new session.

        Issue SESSION_REPLACED after self.session has become the new
        session. After the session has been assigned call
        self.subscribe_session() with the new session and
        self.unsubscribe_session with the old one.
        """
        oldsession = self.session
        self.session = session
        self.subscribe_session(self.session)
        self.issue(SESSION_REPLACED)
        self.interactor.SetSession(session)
        self.maps_changed()
        if oldsession is not None:
            self.unsubscribe_session(oldsession)
            oldsession.Destroy()

    def subscribe_session(self, session):
        """Subscribe to some of the sessions channels.

        Extend this method in derived classes if you need additional
        channels.
        """
        session.Subscribe(MAPS_CHANGED, self.maps_changed)

    def unsubscribe_session(self, session):
        """Unsubscribe from the sessions channels.

        Extend this method in derived classes if you subscribed to
        additional channels in subscribe_session().
        """
        session.Unsubscribe(MAPS_CHANGED, self.maps_changed)

    def create_session(self):
        """Create a default session.

        Override this method in derived classes to instantiate the
        session differently or to use a different session class. Don't
        subscribe to channels here yet. Do that in the
        subscribe_session() method.
        """
        self.SetSession(create_empty_session())

    def OpenSession(self, filename):
        session = load_session(filename)
        session.SetFilename(filename)
        session.UnsetModified()
        self.SetSession(session)

    def SaveSession(self):
        save_session(self.session, self.session.filename)

    def maps_changed(self, *args):
        if self.session.HasMaps():
            self.top.SetMap(self.session.Maps()[0])
        else:
            self.top.SetMap(None)
