# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
# Jan-Oliver Wagner <jan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.12 $"

from Thuban.Lib.connector import Publisher

from messages import MAPS_CHANGED, EXTENSIONS_CHANGED, FILENAME_CHANGED, \
     LAYERS_CHANGED, MAP_PROJECTION_CHANGED, \
     LAYER_LEGEND_CHANGED, LAYER_PROJECTION_CHANGED, LAYER_VISIBILITY_CHANGED,\
     EXTENSION_CHANGED, EXTENSION_OBJECTS_CHANGED, CHANGED

from Thuban import _

from base import TitledObject, Modifiable

from map import Map


class Session(TitledObject, Modifiable):

    """A complete session.

    A Session consists of arbitrary numbers of maps, tables and extensions

    Session objects send the following events:

        TITLE_CHANGED -- The title has changed. Parameters: the session.

        FILENAME_CHANGED -- The filename has changed. No parameters.

        MAPS_CHANGED -- Maps were added, removed.

        EXTENSIONS_CHANGED -- Extensions were added, removed.

        LAYERS_CHANGED -- Same as the map's event of the same name.
                          It's simply resent from the session to make
                          subscriptions easier.

        CHANGED -- Generic changed event. Parameters: the session. The
                   event is always issued when any other changed event
                   is issused. This is useful for code that needs to be
                   notified whenever something in the session has
                   changed but it's too cumbersome or error-prone to
                   subscribe to all the individual events.
    """

    # message channels that have to be forwarded from maps contained in
    # the session.
    forwarded_channels = (
        # generic channels
        CHANGED,

        # map specific channels
        MAP_PROJECTION_CHANGED,
        LAYERS_CHANGED,

        # layer channels forwarded by the map
        LAYER_PROJECTION_CHANGED,
        LAYER_LEGEND_CHANGED,
        LAYER_VISIBILITY_CHANGED,

        # channels forwarded by an extension
        EXTENSION_CHANGED,
        EXTENSION_OBJECTS_CHANGED)

    def __init__(self, title):
        TitledObject.__init__(self, title)
        Modifiable.__init__(self)
        self.filename = None
        self.maps = []
        self.tables = []
        self.extensions = []

    def changed(self, channel = None, *args):
        """Like the inherited version but issue a CHANGED message as well.

        The CHANGED message is only issued if channel given is a
        different channel than CHANGED.
        """
        Modifiable.changed(self, channel, *args)
        if channel != CHANGED:
            self.issue(CHANGED, self)

    def SetFilename(self, filename):
        self.filename = filename
        self.changed(FILENAME_CHANGED)

    def Maps(self):
        return self.maps

    def HasMaps(self):
        return len(self.maps) > 0

    def AddMap(self, map):
        self.maps.append(map)
        for channel in self.forwarded_channels:
            map.Subscribe(channel, self.forward, channel)
        self.changed(MAPS_CHANGED)

    def RemoveMap(self, map):
        for channel in self.forwarded_channels:
            map.Unsubscribe(channel, self.forward, channel)
        self.maps.remove(map)
        self.changed(MAPS_CHANGED)
        map.Destroy()

    def Extensions(self):
        return self.extensions

    def HasExtensions(self):
        return len(self.extensions) > 0

    def AddExtension(self, extension):
        self.extensions.append(extension)
        for channel in self.forwarded_channels:
            extension.Subscribe(channel, self.forward, channel)
        self.changed(EXTENSIONS_CHANGED)

    def Destroy(self):
        for map in self.maps:
            map.Destroy()
        self.maps = []
        self.tables = []
        Modifiable.Destroy(self)

    def forward(self, *args):
        """Reissue events.

        If the channel the event is forwarded to is a changed-channel
        that is not the CHANGED channel issue CHANGED as well. An
        channel is considered to be a changed-channel if it's name ends
        with 'CHANGED'.
        """
        if len(args) > 1:
            args = (args[-1],) + args[:-1]
        apply(self.issue, args)
        channel = args[0]
        # It's a bit of a kludge to rely on the channel name for this.
        if channel.endswith("CHANGED") and channel != CHANGED:
            self.issue(CHANGED, self)

    def WasModified(self):
        """Return true if the session or one of the maps was modified"""
        if self.modified:
            return 1
        else:
            for map in self.maps:
                if map.WasModified():
                    return 1
        return 0

    def UnsetModified(self):
        """Unset the modified flag of the session and the maps"""
        Modifiable.UnsetModified(self)
        for map in self.maps:
            map.UnsetModified()

    def TreeInfo(self):
        items = []
        if self.filename is None:
            items.append(_("Filename:"))
        else:
            items.append(_("Filename: %s") % self.filename)

        if self.WasModified():
            items.append(_("Modified"))
        else:
            items.append(_("Unmodified"))

        items.extend(self.maps)
        items.extend(self.extensions)

        return (_("Session: %s") % self.title, items)


def create_empty_session():
    """Return an empty session useful as a starting point"""
    import os
    session = Session(_('unnamed session'))
    session.SetFilename(None)
    session.AddMap(Map(_('unnamed map')))
    session.UnsetModified()
    return session
