# Copyright (c) 2001, 2002, 2003 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Functions to save a session to a file
"""

__version__ = "$Revision: 1.11 $"

# fix for people using python2.1
from __future__ import nested_scopes

import os
import string

import Thuban.Lib.fileutil

from Thuban.Model.color import Color 

from Thuban.Model.classification import *

#
# one level of indention
#
TAB = "    "

def relative_filename(dir, filename):
    """Return a filename relative to dir for the absolute file name absname.

    This is almost the same as the function in fileutil, except that dir
    can be an empty string in which case filename will be returned
    unchanged.
    """
    if dir:
        return Thuban.Lib.fileutil.relative_filename(dir, filename)
    else:
        return filename

def escape(data):
    """Escape &, \", ', <, and > in a string of data.
    """
    data = string.replace(data, "&", "&amp;")
    data = string.replace(data, "<", "&lt;")
    data = string.replace(data, ">", "&gt;")
    data = string.replace(data, '"', "&quot;")
    data = string.replace(data, "'", "&apos;")
    return data

class Saver:

    """Class to serialize a session into an XML file.

    Applications built on top of Thuban may derive from this class and
    override or extend the methods to save additional information. This
    additional information should take the form of additional attributes
    or elements whose names are prefixed with a namespace. To define a
    namespace derived classes should extend the write_session method to
    pass the namespaces to the default implementation.
    """


    def __init__(self, session):
        self.session = session

    def write(self, file_or_filename):
        """Write the session to a file.

        The argument may be either a file object or a filename. If it's
        a filename, the file will be opened for writing. Files of
        shapefiles will be stored as filenames relative to the directory
        the file is stored in (as given by os.path.dirname(filename)) if
        they have a common parent directory other than the root
        directory.

        If the argument is a file object (which is determined by the
        presence of a write method) all filenames will be absolut
        filenames.
        """

        # keep track of how many levels of indentation to write
        self.indent_level = 0 
        # track whether an element is currently open. see open_element().
        self.element_open = 0

        if hasattr(file_or_filename, "write"):
            # it's a file object
            self.file = file_or_filename
            self.dir = ""
        else:
            filename = file_or_filename
            self.dir = os.path.dirname(filename)
            self.file = open(filename, 'w')
        self.write_header()
        self.write_session(self.session)

        assert(self.indent_level == 0)

    def write_attribs(self, attrs):
        for name, value in attrs.items():
            self.file.write(' %s="%s"' % (escape(name), escape(value)))
    
    def open_element(self, element, attrs = {}):

        #
        # we note when an element is opened so that if two open_element()
        # calls are made successively we can end the currently open
        # tag and will later write a proper close tag. otherwise,
        # if a close_element() call is made directly after an open_element()
        # call we will close the tag with a />
        #
        if self.element_open == 1:
            self.file.write(">\n")

        self.element_open = 1

        # Helper function to write an element open tag with attributes
        self.file.write("%s<%s" % (TAB*self.indent_level, element))
        self.write_attribs(attrs)

        self.indent_level += 1

    def close_element(self, element):
        self.indent_level -= 1
        assert(self.indent_level >= 0)

        # see open_element() for an explanation
        if self.element_open == 1:
            self.element_open = 0
            self.file.write("/>\n")
        else:
            self.file.write("%s</%s>\n" % (TAB*self.indent_level, element))

    def write_element(self, element, attrs = {}):
        """write an element that won't need a closing tag"""
        self.open_element(element, attrs)
        self.close_element(element)

    def write_header(self):
        """Write the XML header"""
        self.file.write('<?xml version="1.0" encoding="UTF-8"?>\n')
        self.file.write('<!DOCTYPE session SYSTEM "thuban.dtd">\n')

    def write_session(self, session, attrs = None, namespaces = ()):
        """Write the session and its contents

        By default, write a session element with the title attribute and
        call write_map for each map contained in the session.

        The optional argument attrs is for additional attributes and, if
        given, should be a mapping from attribute names to attribute
        values. The values should not be XML-escaped yet.

        The optional argument namespaces, if given, should be a sequence
        of (name, URI) pairs. The namespaces are written as namespace
        attributes into the session element. This is mainly useful for
        derived classes that need to store additional information in a
        thuban session file.
        """
        if attrs is None:
            attrs = {}
        attrs["title"] = session.title
        for name, uri in namespaces:
            attrs["xmlns:" + name] = uri
        self.open_element("session", attrs)
        for map in session.Maps():
            self.write_map(map)
        self.close_element("session")

    def write_map(self, map):
        """Write the map and its contents.

        By default, write a map element element with the title
        attribute, call write_projection to write the projection
        element, call write_layer for each layer contained in the map
        and finally call write_label_layer to write the label layer.
        """
        #write = self.file.write
        self.open_element('map title="%s"' % escape(map.title))
        self.write_projection(map.projection)
        for layer in map.Layers():
            self.write_layer(layer)
        self.write_label_layer(map.LabelLayer())
        self.close_element('map')

    def write_projection(self, projection):
        """Write the projection.
        """
        if projection and len(projection.params) > 0:
            self.open_element("projection")
            for param in projection.params:
                self.write_element('parameter value="%s"' % escape(param))
            self.close_element("projection")

    def write_layer(self, layer, attrs = None):
        """Write the layer.

        The optional argument attrs is for additional attributes and, if
        given, should be a mapping from attribute names to attribute
        values. The values should not be XML-escaped yet.
        """
        lc = layer.GetClassification()

        if attrs is None:
            attrs = {}

        attrs["title"]        = layer.title
        attrs["filename"]     = relative_filename(self.dir, layer.filename)
        attrs["stroke"]       = lc.GetDefaultLineColor().hex()
        attrs["stroke_width"] = str(lc.GetDefaultLineWidth())
        attrs["fill"]         = lc.GetDefaultFill().hex()

        self.open_element("layer", attrs)
        self.write_classification(layer)
        self.close_element("layer")

    def write_classification(self, layer, attrs = None):
        if attrs is None:
            attrs = {}

        lc = layer.GetClassification()

        field = lc.GetField()
 
        #
        # there isn't a classification of anything 
        # so don't do anything
        #
        if field is None: return

        attrs["field"] = field
        attrs["field_type"] = str(lc.GetFieldType())
        self.open_element("classification", attrs)


#       self.open_element("clnull")
#       write_class_data(lc.GetDefaultData())
#       self.close_element("clnull")
            
        # just playing now with lambdas and dictionaries

        types = [[lambda p: 'clnull', 
                  lambda p: 'clnull'],
                 [lambda p: 'clpoint value="%s"' % 
                             str(p.GetValue()),
                  lambda p: 'clpoint'],
                 [lambda p: 'clrange min="%s" max="%s"' % 
                             (str(p.GetMin()), 
                              (str(p.GetMax()))),
                  lambda p: 'clrange']]

        def write_class_group(group):
            type = -1
            if isinstance(group, ClassGroupDefault): type = 0
            elif isinstance(group, ClassGroupSingleton): type = 1
            elif isinstance(group, ClassGroupRange): type = 2
            elif isinstance(group, ClassGroupMap):   type = 3
            assert(type >= 0)

            if type <= 2: 
                data = group.GetProperties()
                dict = {'stroke'      : data.GetLineColor().hex(),
                        'stroke_width': str(data.GetLineWidth()),
                        'fill'        : data.GetFill().hex()}

                self.open_element(types[type][0](group))
                self.write_element("cldata", dict)
                self.close_element(types[type][1](group))
            else: pass # XXX: we need to handle maps

        for i in lc:
            write_class_group(i)

#       for i in lc:
#           t = i.GetType()
#           self.open_element(types[t][0](i))
#           write_class_data(i)
#           self.close_element(types[t][1](i))

#       for p in lc:
#           type = p.GetType()
#           if p == ClassData.DEFAULT:
#               lopen = lclose = 'clnull'
#           elif p == ClassData.POINTS:
#               lopen = 'clpoint value="%s"' % escape(str(p.GetValue()))
#               lclose = 'clpoint'
#           elif p == ClassData.RANGES:
#               lopen = 'clrange min="%s" max="%s"'
#                   % (escape(str(p.GetMin())), escape(str(p.GetMax()))))
#               lclose = 'clrange'

#           self.open_element(lopen)
#           write_class_data(p)
#           self.close_element(lclose)
            
#       if lc.points != {}:
#           for p in lc.points.values():
#               self.open_element('clpoint value="%s"' % 
#                   (escape(str(p.GetValue()))))
#               write_class_data(p)
#               self.close_element('clpoint')
#          
#       if lc.ranges != []:
#           for p in lc.ranges:
#               self.open_element('clrange min="%s" max="%s"'
#                   % (escape(str(p.GetMin())), escape(str(p.GetMax()))))
#               write_class_data(p)
#               self.close_element('clrange')

        self.close_element("classification")

    def write_label_layer(self, layer):
        """Write the label layer.
        """
        labels = layer.Labels()
        if labels:
            self.open_element('labellayer')
            for label in labels:
                self.write_element(('label x="%g" y="%g" text="%s"'
                                    ' halign="%s" valign="%s"')
                                % (label.x, label.y, label.text, label.halign,
                                   label.valign))
            self.close_element('labellayer')



def save_session(session, file, saver_class = None):
    """Save the session session to a file.

    The file argument may either be a filename or an open file object.

    The optional argument saver_class is the class to use to serialize
    the session. By default or if it's None, the saver class will be
    Saver.

    If writing the session is successful call the session's
    UnsetModified method
    """
    if saver_class is None:
        saver_class = Saver
    saver = saver_class(session)
    saver.write(file)

    # after a successful save consider the session unmodified.
    session.UnsetModified()
