# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.23 $"

import os
from math import log, ceil

from Thuban import _

import shapelib, shptree

from messages import LAYER_PROJECTION_CHANGED, LAYER_LEGEND_CHANGED, \
     LAYER_VISIBILITY_CHANGED

from color import Color

import classification

from table import Table

from base import TitledObject, Modifiable

class Shape:

    """Represent one shape"""

    def __init__(self, points):
        self.points = points
        #self.compute_bbox()

    def compute_bbox(self):
        xs = []
        ys = []
        for x, y in self.points:
            xs.append(x)
            ys.append(y)
        self.llx = min(xs)
        self.lly = min(ys)
        self.urx = max(xs)
        self.ury = max(ys)

    def Points(self):
        return self.points



# Shape type constants
SHAPETYPE_POLYGON = "polygon"
SHAPETYPE_ARC = "arc"
SHAPETYPE_POINT = "point"

# mapping from shapelib shapetype constants to our constants
shapelib_shapetypes = {shapelib.SHPT_POLYGON: SHAPETYPE_POLYGON,
                       shapelib.SHPT_ARC: SHAPETYPE_ARC,
                       shapelib.SHPT_POINT: SHAPETYPE_POINT}

shapetype_names = {SHAPETYPE_POINT: "Point",
                   SHAPETYPE_ARC: "Arc",
                   SHAPETYPE_POLYGON: "Polygon"}

class BaseLayer(TitledObject, Modifiable):

    """Base class for the layers."""

    def __init__(self, title, visible = 1):
        """Initialize the layer.

        title -- the title
        visible -- boolean. If true the layer is visible.
        """
        TitledObject.__init__(self, title)
        Modifiable.__init__(self)
        self.visible = visible

    def Visible(self):
        """Return true if layer is visible"""
        return self.visible

    def SetVisible(self, visible):
        """Set the layer's visibility."""
        self.visible = visible
        self.issue(LAYER_VISIBILITY_CHANGED, self)


class Layer(BaseLayer):

    """Represent the information of one geodata file (currently a shapefile)

    All children of the layer have the same type.

    A layer has fill and stroke colors. Colors should be instances of
    Color. They can also be None, indicating no fill or no stroke.

    The layer objects send the following events, all of which have the
    layer object as parameter:

        TITLE_CHANGED -- The title has changed.
        LAYER_PROJECTION_CHANGED -- the projection has changed.
        LAYER_LEGEND_CHANGED -- the fill or stroke attributes have changed

    """

    def __init__(self, title, filename, projection = None,
                 fill = Color.None, 
                 stroke = Color.Black, 
                 lineWidth = 1, 
                 visible = 1):
        """Initialize the layer.

        title -- the title
        filename -- the name of the shapefile
        projection -- the projection object. Its Inverse method is
               assumed to map the layer's coordinates to lat/long
               coordinates
        fill -- the fill color or Color.None if the shapes are not filled
        stroke -- the stroke color or Color.None if the shapes are not stroked
        visible -- boolean. If true the layer is visible.

        colors are expected to be instances of Color class
        """
        BaseLayer.__init__(self, title, visible = visible)

        # Make the filename absolute. The filename will be
        # interpreted relative to that anyway, but when saving a
        # session we need to compare absolute paths and it's usually
        # safer to always work with absolute paths.
        self.filename = os.path.abspath(filename)

        self.projection = projection
        self.shapefile = None
        self.shapetree = None
        self.open_shapefile()
        # shapetable is the table associated with the shapefile, while
        # table is the default table used to look up attributes for
        # display
        self.shapetable = Table(filename)
        self.table = self.shapetable

        #
        # this is really important so that when the classification class
        # tries to set its parent layer the variable will exist
        #
        self.__classification = None 
        self.__setClassLock = False


        self.SetClassification(None)

        self.__classification.SetDefaultLineColor(stroke)
        self.__classification.SetDefaultLineWidth(lineWidth)
        self.__classification.SetDefaultFill(fill)
        self.__classification.SetLayer(self)

        self.UnsetModified()

    def open_shapefile(self):
        if self.shapefile is None:
            self.shapefile = shapelib.ShapeFile(self.filename)
            numshapes, shapetype, mins, maxs = self.shapefile.info()
            self.numshapes = numshapes
            self.shapetype = shapelib_shapetypes[shapetype]

            # if there are shapes, set the bbox accordinly. Otherwise
            # set it to None.
            if self.numshapes:
                self.bbox = mins[:2] + maxs[:2]
            else:
                self.bbox = None

            # estimate a good depth for the quad tree. Each depth
            # multiplies the number of nodes by four, therefore we
            # basically take the base 4 logarithm of the number of
            # shapes.
            if self.numshapes < 4:
                maxdepth = 1
            else:
                maxdepth = int(ceil(log(self.numshapes / 4.0) / log(4)))

            self.shapetree = shptree.SHPTree(self.shapefile.cobject(), 2,
                                             maxdepth)

    def Destroy(self):
        BaseLayer.Destroy(self)
        if self.shapefile is not None:
            self.shapefile.close()
            self.shapefile = None
            self.shapetree = None
        self.SetClassification(None)
        self.table.Destroy()

    def BoundingBox(self):
        """Return the layer's bounding box in the intrinsic coordinate system.

        If the layer has no shapes, return None.
        """
        # The bbox will be set by open_shapefile just as we need it
        # here.
        self.open_shapefile()
        return self.bbox

    def LatLongBoundingBox(self):
        """Return the layer's bounding box in lat/long coordinates.

        Return None, if the layer doesn't contain any shapes.
        """
        bbox = self.BoundingBox()
        if bbox is not None:
            llx, lly, urx, ury = bbox
            if self.projection is not None:
                llx, lly = self.projection.Inverse(llx, lly)
                urx, ury = self.projection.Inverse(urx, ury)
            return llx, lly, urx, ury
        else:
            return None

    def GetFieldType(self, fieldName):
        self.open_shapefile()
        info = self.table.field_info_by_name(fieldName)
        if info is not None:
            return info[0]
        else:
            return None

    def NumShapes(self):
        """Return the number of shapes in the layer"""
        self.open_shapefile()
        return self.numshapes

    def ShapeType(self):
        """Return the type of the shapes in the layer.
        This is either SHAPETYPE_POINT, SHAPETYPE_ARC or SHAPETYPE_POLYGON.
        """
        self.open_shapefile()
        return self.shapetype

    def Shape(self, index):
        """Return the shape with index index"""
        self.open_shapefile()
        shape = self.shapefile.read_object(index)

        if self.shapetype == SHAPETYPE_POINT:
            points = shape.vertices()
        else:
            #for poly in shape.vertices():
            poly = shape.vertices()[0]
            points = []
            for x, y in poly:
                points.append((x, y))

        return Shape(points)

    def ShapesInRegion(self, box):
        """Return the ids of the shapes that overlap the box.

        Box is a tuple (left, bottom, right, top) in the coordinate
        system used by the layer's shapefile.
        """
        left, bottom, right, top = box
        return self.shapetree.find_shapes((left, bottom), (right, top))

    def SetProjection(self, projection):
        """Set the layer's projection"""
        self.projection = projection
        self.changed(LAYER_PROJECTION_CHANGED, self)

    def GetClassification(self):
        return self.__classification

    def SetClassification(self, clazz):
        """Set the classification to 'clazz'

        If 'clazz' is None a default classification is created
        """

        # prevent infinite recursion when calling SetLayer()
        if self.__setClassLock: return

        self.__setClassLock = True

        if clazz is None:
            if self.__classification is not None:
                self.__classification.SetLayer(None)
            self.__classification = classification.Classification()
        else:
            self.__classification = clazz
            try:
                self.__classification.SetLayer(self)
            except ValueError:
                self.__setClassLock = False
                raise ValueError

        self.changed(LAYER_LEGEND_CHANGED, self)

        self.__setClassLock = False

    def ClassChanged(self):
        """Called from the classification object when it has changed."""
        self.changed(LAYER_LEGEND_CHANGED, self)
 
    def TreeInfo(self):
        items = []

        if self.Visible():
            items.append(_("Shown"))
        else:
            items.append(_("Hidden"))
        items.append(_("Shapes: %d") % self.NumShapes())

        bbox = self.LatLongBoundingBox()
        if bbox is not None:
            items.append(_("Extent (lat-lon): (%g, %g, %g, %g)") % bbox)
        else:
            items.append(_("Extent (lat-lon):"))
        items.append(_("Shapetype: %s") % shapetype_names[self.ShapeType()])

        items.append(self.__classification)

        return (_("Layer '%s'") % self.Title(), items)

