# Copyright (c) 2001, 2003 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.17 $"

"""
A Classification provides a mapping from an input value
to data. This mapping can be specified in two ways. 
First, specific values can be associated with data. 
Second, ranges can be associated with data such that if 
an input value falls with a range that data is returned. 
If no mapping can be found then default data will 
be returned. Input values must be hashable objects

See the description of GetGroup() for more information 
on the mapping algorithm.
"""
  
# fix for people using python2.1
from __future__ import nested_scopes

import copy

from types import *

from messages import LAYER_PROJECTION_CHANGED, LAYER_LEGEND_CHANGED, \
     LAYER_VISIBILITY_CHANGED

from Thuban import _
from Thuban.Model.color import Color

import Thuban.Model.layer

# constants
RANGE_MIN  = 0
RANGE_MAX  = 1
RANGE_DATA = 2

class Classification:
    """Encapsulates the classification of layer. The Classification
    divides some kind of data into Groups which are associated with
    properties. Later the properties can be retrieved by matching
    data values to the appropriate group."""

    def __init__(self, layer = None, field = None):
        """Initialize a classification.

           layer -- the Layer object who owns this classification

           field -- the name of the data table field that 
                    is to be used to classify layer properties
        """
 
        self.layer = None
        self.field = None
        self.fieldType = None
        self.groups = []

        self.__setLayerLock = False

        self.SetDefaultGroup(ClassGroupDefault())

        self.SetLayer(layer)
        self.SetField(field)

    def __iter__(self):
        return ClassIterator(self.groups) 

    def __SendNotification(self):
        """Notify the layer that this class has changed."""
        if self.layer is not None:
            self.layer.ClassChanged()
    
    def SetField(self, field):
        """Set the name of the data table field to use.
         
           If there is no layer then the field type is set to None, 
           otherwise the layer is queried to find the type of the
           field data

           field -- if None then all values map to the default data 
        """

        if field == "": 
            field = None


        if field is None:
            if self.layer is not None:
                self.fieldType = None
        else:
            if self.layer is not None:
                fieldType = self.layer.GetFieldType(field)
                if fieldType is None:
                    raise ValueError("'%s' was not found in the layer's table."
                                     % self.field)

                #
                # unfortunately we cannot call SetFieldType() because it
                # requires the layer to be None
                #
                self.fieldType = fieldType
                #self.SetFieldType(fieldType)

        self.field = field

        self.__SendNotification()

    def GetField(self):
        """Return the name of the field."""
        return self.field

    def GetFieldType(self):
        """Return the field type."""
        return self.fieldType

    def SetFieldType(self, type):
        """Set the type of the field used by this classification.

        A ValueError is raised if the owning layer is not None and
        'type' is different from the current field type.
        """

        if type != self.fieldType:
            if self.layer is not None:
                raise ValueError()
            else:
                self.fieldType = type
                self.__SendNotification()

    def SetLayer(self, layer):
        """Set the owning Layer of this classification.
 
           A ValueError exception will be thrown either the field or
           field type mismatch the information in the layer's table.
        """

        # prevent infinite recursion when calling SetClassification()
        if self.__setLayerLock: return

        self.__setLayerLock = True

        if layer is None:
            if self.layer is not None:
                l = self.layer
                self.layer = None
                l.SetClassification(None)
        else:
            assert(isinstance(layer, Thuban.Model.layer.Layer))

            old_layer = self.layer

            self.layer = layer

            try:
                self.SetField(self.GetField()) # this sync's the fieldType
            except ValueError:
                self.layer = old_layer
                self.__setLayerLock = False
                raise ValueError
            else:
                self.layer.SetClassification(self)

        self.__setLayerLock = False

    def GetLayer(self):
        """Return the parent layer."""
        return self.layer

    def SetDefaultGroup(self, group):
        """Set the group to be used when a value can't be classified.

           group -- group that the value maps to.
        """

        assert(isinstance(group, ClassGroupDefault))
        self.AddGroup(group)

    def GetDefaultGroup(self):
        """Return the default group."""
        return self.groups[0]

    #
    # these SetDefault* methods are really only provided for 
    # some backward compatibility. they should be considered
    # for removal once all the classification code is finished.
    #

    def SetDefaultFill(self, fill):
        """Set the default fill color.

        fill -- a Color object.
        """
        assert(isinstance(fill, Color))
        self.GetDefaultGroup().GetProperties().SetFill(fill)
        self.__SendNotification()
        
    def GetDefaultFill(self):
        """Return the default fill color."""
        return self.GetDefaultGroup().GetProperties().GetFill()
        
    def SetDefaultLineColor(self, color):
        """Set the default line color.

        color -- a Color object.
        """
        assert(isinstance(color, Color))
        self.GetDefaultGroup().GetProperties().SetLineColor(color)
        self.__SendNotification()
        
    def GetDefaultLineColor(self):
        """Return the default line color."""
        return self.GetDefaultGroup().GetProperties().GetLineColor()
        
    def SetDefaultLineWidth(self, lineWidth):
        """Set the default line width.

        lineWidth -- an integer > 0.
        """
        assert(isinstance(lineWidth, IntType))
        self.GetDefaultGroup().GetProperties().SetLineWidth(lineWidth)
        self.__SendNotification()
        
    def GetDefaultLineWidth(self):
        """Return the default line width."""
        return self.GetDefaultGroup().GetProperties().GetLineWidth()
        
    def AddGroup(self, item):
        """Add a new ClassGroup item to the classification.

        item -- this must be a valid ClassGroup object
        """

        assert(isinstance(item, ClassGroup))

        if len(self.groups) > 0 and isinstance(item, ClassGroupDefault):
            self.groups[0] = item
        else:
            self.groups.append(item)

        self.__SendNotification()

    def GetGroup(self, value):
        """Return the associated group, or the default group.

           Groups are checked in the order the were added to the
           Classification.

           value -- the value to classify. If there is no mapping,
                    the field is None or value is None, 
                    return the default properties 
        """

        if self.GetField() is not None and value is not None:

            for i in range(1, len(self.groups)):
                group = self.groups[i]
                if group.Matches(value):
                    return group

        return self.GetDefaultGroup()

    def GetProperties(self, value):
        """Return the properties associated with the given value."""

        group = self.GetGroup(value)
        if isinstance(group, ClassGroupMap):
            return group.GetPropertiesFromValue(value)
        else:
            return group.GetProperties()

    def TreeInfo(self):
        items = []

        def build_color_item(text, color):
            if color is Color.None:
                return ("%s: %s" % (text, _("None")), None)

            return ("%s: (%.3f, %.3f, %.3f)" % 
                    (text, color.red, color.green, color.blue),
                    color)

        def build_item(group, string):
            label = group.GetLabel()
            if label == "":
                label = string
            else:
                label += " (%s)" % string

            props = group.GetProperties()
            i = []
            v = props.GetLineColor()
            i.append(build_color_item(_("Line Color"), v))
            v = props.GetLineWidth()
            i.append(_("Line Width: %s") % v)
            v = props.GetFill()
            i.append(build_color_item(_("Fill"), v))
            return (label, i)

        for p in self:
            if isinstance(p, ClassGroupDefault):
                items.append(build_item(self.GetDefaultGroup(), _("'DEFAULT'")))
            elif isinstance(p, ClassGroupSingleton):
                items.append(build_item(p, str(p.GetValue())))
            elif isinstance(p, ClassGroupRange):
                items.append(build_item(p, "%s - %s" % 
                                           (p.GetMin(), p.GetMax())))

        return (_("Classification"), items)
 
class ClassIterator:
    """Allows the Groups in a Classifcation to be interated over.

    The items are returned in the following order:
        default data, singletons, ranges, maps
    """

    def __init__(self, data): #default, points, ranges, maps):
        """Constructor.

        default -- the default group
 
        points -- a list of singleton groups

        ranges -- a list of range groups
 
        maps -- a list of map groups
        """

        self.data = data #[default, points, ranges, maps]
        self.data_index = 0
        #self.data_iter = iter(self.data)
        #self.iter = None

    def __iter__(self):
        return self

    def next(self):
        """Return the next item."""

        if self.data_index >= len(self.data):
            raise StopIteration
        else:
            d = self.data[self.data_index]
            self.data_index += 1
            return d
        
#       if self.iter is None:
#           try:
#               self.data_item = self.data_iter.next()
#               self.iter = iter(self.data_item)
#           except TypeError:
#               return self.data_item

#       try:
#           return self.iter.next()
#       except StopIteration:
#           self.iter = None
#           return self.next()
      
class ClassGroupProperties:
    """Represents the properties of a single Classification Group.
  
    These are used when rendering a layer."""

    def __init__(self, props = None):
        """Constructor.

        props -- a ClassGroupProperties object. The class is copied if
                 prop is not None. Otherwise, a default set of properties
                 is created such that: line color = Color.Black, line width = 1,
                 and fill color = Color.None
        """

        self.stroke = None
        self.strokeWidth = 0
        self.fill = None

        if props is not None:
            self.SetProperties(props)
        else:
            self.SetLineColor(Color.Black)
            self.SetLineWidth(1)
            self.SetFill(Color.None)

    def SetProperties(self, props):
        """Set this class's properties to those in class props."""

        assert(isinstance(props, ClassGroupProperties))
        self.SetLineColor(props.GetLineColor())
        self.SetLineWidth(props.GetLineWidth())
        self.SetFill(props.GetFill())
        
    def GetLineColor(self):
        """Return the line color as a Color object."""
        return self.stroke

    def SetLineColor(self, color):
        """Set the line color.

        color -- the color of the line. This must be a Color object.
        """

        assert(isinstance(color, Color))
        self.stroke = color

    def GetLineWidth(self):
        """Return the line width."""
        return self.strokeWidth

    def SetLineWidth(self, lineWidth):
        """Set the line width.

        lineWidth -- the new line width. This must be > 0.
        """
        assert(isinstance(lineWidth, IntType))
        if (lineWidth < 1):
            raise ValueError(_("lineWidth < 1"))

        self.strokeWidth = lineWidth

    def GetFill(self):
        """Return the fill color as a Color object."""
        return self.fill
 
    def SetFill(self, fill):
        """Set the fill color.

        fill -- the color of the fill. This must be a Color object.
        """

        assert(isinstance(fill, Color))
        self.fill = fill

    def __eq__(self, other):
        """Return true if 'props' has the same attributes as this class"""

        return isinstance(other, ClassGroupProperties)   \
            and self.stroke      == other.GetLineColor() \
            and self.strokeWidth == other.GetLineWidth() \
            and self.fill        == other.GetFill()

    def __ne__(self, other): 
        return not self.__eq__(other)

    def __copy__(self):
        return ClassGroupProperties(self)

class ClassGroup:
    """A base class for all Groups within a Classification"""

    def __init__(self, label = ""):
        """Constructor.

        label -- A string representing the Group's label
        """

        self.label = None

        self.SetLabel(label)

    def GetLabel(self):
        """Return the Group's label."""
        return self.label
 
    def SetLabel(self, label):
        """Set the Group's label.

        label -- a string representing the Group's label. This must
                 not be None.
        """
        assert(isinstance(label, StringType))
        self.label = label

    def Matches(self, value):
        """Determines if this Group is associated with the given value.

        Returns False. This needs to be overridden by all subclasses.
        """
        return False

    def GetProperties(self):
        """Return the properties associated with the given value.

        Returns None. This needs to be overridden by all subclasses.
        """
        return None
 
    
class ClassGroupSingleton(ClassGroup):
    """A Group that is associated with a single value."""

    def __init__(self, value = 0, prop = None, label = ""):
        """Constructor.

        value -- the associated value.

        prop -- a ClassGroupProperites object. If prop is None a default
                 set of properties is created.

        label -- a label for this group.
        """
        ClassGroup.__init__(self, label)

        self.prop = None
        self.value = None

        self.SetValue(value)
        self.SetProperties(prop)

    def __copy__(self):
        return ClassGroupSingleton(self.GetValue(), 
                                   self.GetProperties(), 
                                   self.GetLabel())

    def __deepcopy__(self, memo):
        return ClassGroupSingleton(copy.copy(self.GetValue()), 
                                   copy.copy(self.GetProperties()), 
                                   copy.copy(self.GetLabel()))

    def GetValue(self):
        """Return the associated value."""
        return self.value

    def SetValue(self, value):
        """Associate this Group with the given value."""
        self.value = value

    def Matches(self, value):
        """Determine if the given value matches the associated Group value."""

        """Returns True if the value matches, False otherwise."""

        return self.value == value

    def GetProperties(self):
        """Return the Properties associated with this Group."""

        return self.prop

    def SetProperties(self, prop):
        """Set the properties associated with this Group.

        prop -- a ClassGroupProperties object. if prop is None, 
                a default set of properties is created.
        """

        if prop is None: prop = ClassGroupProperties()
        assert(isinstance(prop, ClassGroupProperties))
        self.prop = prop

    def __eq__(self, other):
        return isinstance(other, ClassGroupSingleton) \
            and self.GetProperties() == other.GetProperties() \
            and self.GetValue() == other.GetValue()

    def __ne__(self, other):
        return not self.__eq__(other)

class ClassGroupDefault(ClassGroup):
    """The default Group. When values do not match any other
       Group within a Classification, the properties from this
       class are used."""

    def __init__(self, prop = None, label = ""):
        """Constructor.

        prop -- a ClassGroupProperites object. If prop is None a default
                 set of properties is created.

        label -- a label for this group.
        """

        ClassGroup.__init__(self, label)
        self.SetProperties(prop)

    def __copy__(self):
        return ClassGroupDefault(self.GetProperties(), self.GetLabel())

    def __deepcopy__(self, memo):
        return ClassGroupDefault(copy.copy(self.GetProperties()), 
                                 copy.copy(self.GetLabel()))

    def Matches(self, value):
        return True

    def GetProperties(self):
        """Return the Properties associated with this Group."""
        return self.prop

    def SetProperties(self, prop):
        """Set the properties associated with this Group.

        prop -- a ClassGroupProperties object. if prop is None, 
                a default set of properties is created.
        """

        if prop is None: prop = ClassGroupProperties()
        assert(isinstance(prop, ClassGroupProperties))
        self.prop = prop

    def __eq__(self, other):
        return isinstance(other, ClassGroupDefault) \
            and self.GetProperties() == other.GetProperties()

    def __ne__(self, other):
        return not self.__eq__(other)

class ClassGroupRange(ClassGroup):
    """A Group that represents a range of values that map to the same
       set of properties."""

    def __init__(self, min = 0, max = 1, prop = None, label = ""):
        """Constructor.

        The minumum value must be strictly less than the maximum.

        min -- the minimum range value

        max -- the maximum range value

        prop -- a ClassGroupProperites object. If prop is None a default
                 set of properties is created.

        label -- a label for this group.
        """

        ClassGroup.__init__(self, label)

        self.min = self.max = 0
        self.prop = None

        self.SetRange(min, max)
        self.SetProperties(prop)

    def __copy__(self):
        return ClassGroupRange(self.GetMin(), 
                               self.GetMax(), 
                               self.GetProperties(), 
                               self.GetLabel())

    def __deepcopy__(self, memo):
        return ClassGroupRange(copy.copy(self.GetMin()), 
                               copy.copy(self.GetMax()), 
                               copy.copy(self.GetProperties()), 
                               copy.copy(self.GetLabel()))

    def GetMin(self):
        """Return the range's minimum value."""
        return self.min

    def SetMin(self, min):
        """Set the range's minimum value.
     
        min -- the new minimum. Note that this must be less than the current
               maximum value. Use SetRange() to change both min and max values.
        """
     
        self.SetRange(min, self.max)

    def GetMax(self):
        """Return the range's maximum value."""
        return self.max

    def SetMax(self, max):
        """Set the range's maximum value.
     
        max -- the new maximum. Note that this must be greater than the current
               minimum value. Use SetRange() to change both min and max values.
        """
        self.SetRange(self.min, max)

    def SetRange(self, min, max):
        """Set a new range.

        Note that min must be strictly less than max.

        min -- the new minimum value
        min -- the new maximum value
        """

        if min >= max:
            raise ValueError(_("ClassGroupRange: %i(min) >= %i(max)!") %
                             (min, max))
        self.min = min
        self.max = max

    def GetRange(self):
        """Return the range as a tuple (min, max)"""
        return (self.min, self.max)

    def Matches(self, value):
        """Determine if the given value lies with the current range.

        The following check is used: min <= value < max.
        """

        return self.min <= value < self.max

    def GetProperties(self):
        """Return the Properties associated with this Group."""
        return self.prop

    def SetProperties(self, prop):
        """Set the properties associated with this Group.

        prop -- a ClassGroupProperties object. if prop is None, 
                a default set of properties is created.
        """
        if prop is None: prop = ClassGroupProperties()
        assert(isinstance(prop, ClassGroupProperties))
        self.prop = prop

    def __eq__(self, other):
        return isinstance(other, ClassGroupRange) \
            and self.GetProperties() == other.GetProperties() \
            and self.GetRange() == other.GetRange()

    def __ne__(self, other):
        return not self.__eq__(other)

class ClassGroupMap(ClassGroup):
    """Currently, this class is not used."""

    FUNC_ID = "id"

    def __init__(self, map_type = FUNC_ID, func = None, prop = None, label=""):
        ClassGroup.__init__(self, label)

        self.map_type = map_type
        self.func = func

        if self.func is None:
            self.func = func_id

    def Map(self, value):
        return self.func(value)

    def GetProperties(self):
        return None

    def GetPropertiesFromValue(self, value):
        pass

    #
    # built-in mappings
    #
    def func_id(value):
        return value

