# Copyright (c) 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test the Session class
"""

__version__ = "$Revision: 1.1 $"
# $Source: /thubanrepository/thuban/test/test_session.py,v $
# $Id: test_session.py,v 1.1 2002/09/20 15:51:17 bh Exp $

import os
import unittest

import support
support.initthuban()

from Thuban.Model.messages import CHANGED, MAPS_CHANGED, FILENAME_CHANGED, \
     MAP_PROJECTION_CHANGED, LAYERS_CHANGED, \
     LAYER_VISIBILITY_CHANGED, LAYER_LEGEND_CHANGED
from Thuban.Model.session import Session
from Thuban.Model.map import Map
from Thuban.Model.layer import Layer
from Thuban.Model.proj import Projection
from Thuban.Model.color import Color


class TestSessionSimple(unittest.TestCase):

    """Very simple test cases for Session"""

    def test_initial_state(self):
        """Test Session's initial state"""
        session = Session("Test Session")
        self.assertEquals(session.Title(), "Test Session")
        self.assertEquals(session.Maps(), [])
        self.assertEquals(session.filename, None)
        self.failIf(session.HasMaps())
        self.failIf(session.WasModified())
        session.Destroy()


class TestSessionBase(unittest.TestCase, support.SubscriberMixin):

    """Base class for Session test cases that test the messages"""

    def setUp(self):
        """
        Clear the message list, create a session and subscribe to its messages

        Bind the session to self.session.
        """
        self.clear_messages()

        # Create a Session and subscribe to all interesting channels.
        self.session = Session("Test Session")
        for channel in (CHANGED, MAPS_CHANGED, FILENAME_CHANGED,
                        MAP_PROJECTION_CHANGED, LAYERS_CHANGED,
                        LAYER_VISIBILITY_CHANGED, LAYER_LEGEND_CHANGED):
            self.session.Subscribe(channel,
                                   self.subscribe_with_params, channel)

    def tearDown(self):
        """Destroy self.session and clear the message list"""
        self.session.Destroy()
        self.clear_messages()


class TestSessionMessages(TestSessionBase):

    """Simple Session test cases that test messges"""

    def test_add_map(self):
        """Test Session.AddMap"""
        self.failIf(self.session.WasModified())
        map = Map("Some Map")
        self.session.AddMap(map)
        self.assert_(self.session.HasMaps())
        self.assert_(self.session.WasModified())
        self.assertEquals(self.session.Maps(), [map])
        self.check_messages([(MAPS_CHANGED,),
                             (self.session, CHANGED)])

    def test_set_filename(self):
        """Test Session.SetFilename"""
        self.session.SetFilename("session_set_filename_test")
        self.session.filename = "session_set_filename_test"
        self.check_messages([(FILENAME_CHANGED,),
                             (self.session, CHANGED)])


class TestSessionWithContent(TestSessionBase):

    """Session test cases that start with a filled session."""

    def setUp(self):
        """Extend the inherited method to add a non-empty map to self.session
        """
        TestSessionBase.setUp(self)
        self.arc_layer = Layer("Roads",
                               os.path.join("..", "Data", "iceland",
                                            "roads-line.shp"))
        self.poly_layer = Layer("Political",
                                os.path.join("..", "Data", "iceland",
                                             "political.shp"))
        self.map = Map("A Map")
        self.map.AddLayer(self.arc_layer)
        self.map.AddLayer(self.poly_layer)
        self.session.AddMap(self.map)
        self.session.UnsetModified()
        self.clear_messages()

    def test_remove_map(self):
        """Test Session.RemoveMap"""
        self.session.RemoveMap(self.map)
        self.assert_(self.session.WasModified())
        self.failIf(self.session.HasMaps())
        self.check_messages([(MAPS_CHANGED,),
                             (self.session, CHANGED)])

    def test_tree_info(self):
        """Test Session.TreeInfo"""
        self.assertEquals(self.session.TreeInfo(),
                          ('Session: Test Session',
                           ['Filename:',
                            'Unmodified',
                            self.map]))

    def test_forward_map_projection(self):
        """Test Session forwarding of Map.SetProjection messages"""
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        self.map.SetProjection(proj)
        self.check_messages([(self.map, MAP_PROJECTION_CHANGED),
                             (self.session, CHANGED)])
        self.assert_(self.session.WasModified())

    def test_forward_map_projection(self):
        """Test Session forwarding of Map.SetProjection messages"""
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        self.map.SetProjection(proj)
        self.assert_(self.session.WasModified())
        self.check_messages([(self.map, MAP_PROJECTION_CHANGED),
                             (self.session, CHANGED)])

    def test_forwarding_fill(self):
        """Test Session's forwarding of Layer.SetFill messages"""
        self.poly_layer.SetFill(Color(0.0, 0.5, 1.0))
        self.assert_(self.session.WasModified())
        self.check_messages([(self.poly_layer, LAYER_LEGEND_CHANGED),
                             (self.session, CHANGED)])

    def test_forwarding_stroke(self):
        """Test Session's forwarding of Layer.SetStroke messages"""
        self.poly_layer.SetStroke(Color(0.0, 0.5, 1.0))
        self.assert_(self.session.WasModified())
        self.check_messages([(self.poly_layer, LAYER_LEGEND_CHANGED),
                             (self.session, CHANGED)])

    def test_forwarding_stroke_width(self):
        """Test Session's forwarding of Layer.SetStrokeWidth messages"""
        self.poly_layer.SetStrokeWidth(3)
        self.assert_(self.session.WasModified())
        self.check_messages([(self.poly_layer, LAYER_LEGEND_CHANGED),
                             (self.session, CHANGED)])

    def test_forwarding_visibility(self):
        """Test Session's forwarding of Layer.SetVisible messages"""
        self.poly_layer.SetVisible(0)
        # Currently changing the visibility of a layer doesn't change
        # the modification flag.
        self.failIf(self.session.WasModified())
        self.check_messages([(self.poly_layer, LAYER_VISIBILITY_CHANGED),
                             (self.session, CHANGED)])

    def test_unset_modified_map(self):
        """Test Session.UnsetModified with map level changes"""
        self.failIf(self.session.WasModified())
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        self.map.SetProjection(proj)
        self.assert_(self.session.WasModified())
        self.session.UnsetModified()
        self.failIf(self.session.WasModified())

    def test_unset_modified_layer(self):
        """Test Session.UnsetModified with layer level changes"""
        self.failIf(self.session.WasModified())
        self.poly_layer.SetStrokeWidth(3)
        self.assert_(self.session.WasModified())
        self.session.UnsetModified()
        self.failIf(self.session.WasModified())
        self.check_messages([(self.poly_layer, LAYER_LEGEND_CHANGED),
                             (self.session, CHANGED),
                             (CHANGED,)])


if __name__ == "__main__":
    unittest.main()
