# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Classes for display of a map and interaction with it
"""

__version__ = "$Revision: 1.23 $"

from math import hypot

from wxPython.wx import wxWindow,\
     wxPaintDC, wxColour, wxClientDC, wxINVERT, wxTRANSPARENT_BRUSH, wxFont,\
     EVT_PAINT, EVT_LEFT_DOWN, EVT_LEFT_UP, EVT_MOTION, EVT_LEAVE_WINDOW


from wxPython import wx

from wxproj import point_in_polygon_shape, shape_centroid


from Thuban.Model.messages import MAP_PROJECTION_CHANGED, \
     LAYERS_CHANGED, LAYER_LEGEND_CHANGED, LAYER_VISIBILITY_CHANGED
from Thuban.Model.layer import SHAPETYPE_POLYGON, SHAPETYPE_ARC, \
     SHAPETYPE_POINT
from Thuban.Model.label import ALIGN_CENTER, ALIGN_TOP, ALIGN_BOTTOM, \
     ALIGN_LEFT, ALIGN_RIGHT
from Thuban.Lib.connector import Publisher

from renderer import ScreenRenderer, PrinterRender

import labeldialog

from messages import SELECTED_SHAPE, VIEW_POSITION


#
#   The tools
#

class Tool:

    """
    Base class for the interactive tools
    """

    def __init__(self, view):
        """Intitialize the tool. The view is the canvas displaying the map"""
        self.view = view
        self.start = self.current = None
        self.dragging = 0
        self.drawn = 0

    def Name(self):
        """Return the tool's name"""
        return ''

    def drag_start(self, x, y):
        self.start = self.current = x, y
        self.dragging = 1

    def drag_move(self, x, y):
        self.current = x, y

    def drag_stop(self, x, y):
        self.current = x, y
        self.dragging = 0

    def Show(self, dc):
        if not self.drawn:
            self.draw(dc)
        self.drawn = 1

    def Hide(self, dc):
        if self.drawn:
            self.draw(dc)
        self.drawn = 0

    def draw(self, dc):
        pass

    def MouseDown(self, event):
        self.drag_start(event.m_x, event.m_y)

    def MouseMove(self, event):
        if self.dragging:
            self.drag_move(event.m_x, event.m_y)

    def MouseUp(self, event):
        if self.dragging:
            self.drag_move(event.m_x, event.m_y)

    def Cancel(self):
        self.dragging = 0


class RectTool(Tool):

    """Base class for tools that draw rectangles while dragging"""

    def draw(self, dc):
        sx, sy = self.start
        cx, cy = self.current
        dc.DrawRectangle(sx, sy, cx - sx, cy - sy)

class ZoomInTool(RectTool):

    """The Zoom-In Tool"""

    def Name(self):
        return "ZoomInTool"

    def proj_rect(self):
        """return the rectangle given by start and current in projected
        coordinates"""
        sx, sy = self.start
        cx, cy = self.current
        left, top = self.view.win_to_proj(sx, sy)
        right, bottom = self.view.win_to_proj(cx, cy)
        return (min(left, right), min(top, bottom),
                max(left, right), max(top, bottom))

    def MouseUp(self, event):
        if self.dragging:
            Tool.MouseUp(self, event)
            sx, sy = self.start
            cx, cy = self.current
            if sx == cx or sy == cy:
                # Just a mouse click or a degenerate rectangle. Simply
                # zoom in by a factor of two
                # FIXME: For a click this is the desired behavior but should we
                # really do this for degenrate rectagles as well or
                # should we ignore them?
                self.view.ZoomFactor(2, center = (cx, cy))
            else:
                # A drag. Zoom in to the rectangle
                self.view.FitRectToWindow(self.proj_rect())


class ZoomOutTool(RectTool):

    """The Zoom-Out Tool"""

    def Name(self):
        return "ZoomOutTool"

    def MouseUp(self, event):
        if self.dragging:
            Tool.MouseUp(self, event)
            sx, sy = self.start
            cx, cy = self.current
            if sx == cx or sy == cy:
                # Just a mouse click or a degenerate rectangle. Simply
                # zoom out by a factor of two.
                # FIXME: For a click this is the desired behavior but should we
                # really do this for degenrate rectagles as well or
                # should we ignore them?
                self.view.ZoomFactor(0.5, center = (cx, cy))
            else:
                # A drag. Zoom out to the rectangle
                self.view.ZoomOutToRect((min(sx, cx), min(sy, cy),
                                         max(sx, cx), max(sy, cy)))


class PanTool(Tool):

    """The Pan Tool"""

    def Name(self):
        return "PanTool"

    def MouseMove(self, event):
        if self.dragging:
            Tool.MouseMove(self, event)
            sx, sy = self.start
            x, y = self.current
            width, height = self.view.GetSizeTuple()

            bitmapdc = wx.wxMemoryDC()
            bitmapdc.SelectObject(self.view.bitmap)

            dc = self.view.drag_dc
            dc.Blit(0, 0, width, height, bitmapdc, sx - x, sy - y)

    def MouseUp(self, event):
        if self.dragging:
            Tool.MouseUp(self, event)
            sx, sy = self.start
            cx, cy = self.current
            self.view.Translate(cx - sx, cy - sy)

class IdentifyTool(Tool):

    """The "Identify" Tool"""

    def Name(self):
        return "IdentifyTool"

    def MouseUp(self, event):
        self.view.SelectShapeAt(event.m_x, event.m_y)


class LabelTool(Tool):

    """The "Label" Tool"""

    def Name(self):
        return "LabelTool"

    def MouseUp(self, event):
        self.view.LabelShapeAt(event.m_x, event.m_y)




class MapPrintout(wx.wxPrintout):

    """
    wxPrintout class for printing Thuban maps
    """

    def __init__(self, map):
        wx.wxPrintout.__init__(self)
        self.map = map

    def GetPageInfo(self):
        return (1, 1, 1, 1)

    def HasPage(self, pagenum):
        return pagenum == 1

    def OnPrintPage(self, pagenum):
        if pagenum == 1:
            self.draw_on_dc(self.GetDC())

    def draw_on_dc(self, dc):
        width, height = self.GetPageSizePixels()
        llx, lly, urx, ury = self.map.ProjectedBoundingBox()
        scalex = width / (urx - llx)
        scaley = height / (ury - lly)
        scale = min(scalex, scaley)
        offx = 0.5 * (width - (urx + llx) * scale)
        offy = 0.5 * (height + (ury + lly) * scale)

        resx, resy = self.GetPPIPrinter()
        renderer = PrinterRender(dc, scale, (offx, offy), resolution = resx)
        renderer.RenderMap(self.map)
        return wx.true


class MapCanvas(wxWindow, Publisher):

    """A widget that displays a map and offers some interaction"""

    def __init__(self, parent, winid, interactor):
        wxWindow.__init__(self, parent, winid)
        self.SetBackgroundColour(wxColour(255, 255, 255))

        # the map displayed in this canvas. Set with SetMap()
        self.map = None

        # scale and offset describe the transformation from projected
        # coordinates to window coordinates.
        self.scale = 1.0
        self.offset = (0, 0)

        # whether the user is currently dragging the mouse, i.e. moving
        # the mouse while pressing a mouse button
        self.dragging = 0

        # the currently active tool
        self.tool = None

        # The current mouse position of the last OnMotion event or None
        # if the mouse is outside the window.
        self.current_position = None

        # the bitmap serving as backing store
        self.bitmap = None

        # the interactor
        self.interactor = interactor
        self.interactor.Subscribe(SELECTED_SHAPE, self.shape_selected)

        # keep track of which layers/shapes are selected to make sure we
        # only redraw when necessary
        self.last_selected_layer = None
        self.last_selected_shape = None

        # subscribe the WX events we're interested in
        EVT_PAINT(self, self.OnPaint)
        EVT_LEFT_DOWN(self, self.OnLeftDown)
        EVT_LEFT_UP(self, self.OnLeftUp)
        EVT_MOTION(self, self.OnMotion)
        EVT_LEAVE_WINDOW(self, self.OnLeaveWindow)
        wx.EVT_SIZE(self, self.OnSize)

    def __del__(self):
        wxWindow.__del__(self)
        Publisher.__del__(self)

    def OnPaint(self, event):
        dc = wxPaintDC(self)
        if self.map is not None and self.map.HasLayers():
            self.do_redraw()
        else:
            # If we've got no map or if the map is empty, simply clear
            # the screen.

            # XXX it's probably possible to get rid of this. The
            # background color of the window is already white and the
            # only thing we may have to do is to call self.Refresh()
            # with a true argument in the right places.
            dc.BeginDrawing()
            dc.Clear()
            dc.EndDrawing()

    def do_redraw(self):
        # This should only be called if we have a non-empty map.

        # Get the window size.
        width, height = self.GetSizeTuple()

        # If self.bitmap's still there, reuse it. Otherwise redraw it
        if self.bitmap is not None:
            bitmap = self.bitmap
        else:
            bitmap = wx.wxEmptyBitmap(width, height)
            dc = wx.wxMemoryDC()
            dc.SelectObject(bitmap)
            dc.BeginDrawing()

            # clear the background
            dc.SetBrush(wx.wxWHITE_BRUSH)
            dc.SetPen(wx.wxTRANSPARENT_PEN)
            dc.DrawRectangle(0, 0, width, height)

            if 1: #self.interactor.selected_map is self.map:
                selected_layer = self.interactor.selected_layer
                selected_shape = self.interactor.selected_shape
            else:
                selected_layer = None
                selected_shape = None

            # draw the map into the bitmap
            renderer = ScreenRenderer(dc, self.scale, self.offset)

            # Pass the entire bitmap as update region to the renderer.
            # We're redrawing the whole bitmap, after all.
            renderer.RenderMap(self.map, (0, 0, width, height),
                               selected_layer, selected_shape)

            dc.EndDrawing()
            dc.SelectObject(wx.wxNullBitmap)
            self.bitmap = bitmap

        # blit the bitmap to the screen
        dc = wx.wxMemoryDC()
        dc.SelectObject(bitmap)
        clientdc = wxClientDC(self)
        clientdc.BeginDrawing()
        clientdc.Blit(0, 0, width, height, dc, 0, 0)
        clientdc.EndDrawing()

    def Print(self):
        printer = wx.wxPrinter()
        printout = MapPrintout(self.map)
        printer.Print(self, printout, wx.true)
        printout.Destroy()

    def SetMap(self, map):
        redraw_channels = (LAYERS_CHANGED, LAYER_LEGEND_CHANGED,
                           LAYER_VISIBILITY_CHANGED)
        if self.map is not None:
            for channel in redraw_channels:
                self.map.Unsubscribe(channel, self.full_redraw)
            self.map.Unsubscribe(MAP_PROJECTION_CHANGED,
                                 self.projection_changed)
        self.map = map
        if self.map is not None:
            for channel in redraw_channels:
                self.map.Subscribe(channel, self.full_redraw)
            self.map.Subscribe(MAP_PROJECTION_CHANGED, self.projection_changed)
        self.FitMapToWindow()
        # force a redraw. If map is not empty, it's already been called
        # by FitMapToWindow but if map is empty it hasn't been called
        # yet so we have to explicitly call it.
        self.full_redraw()

    def Map(self):
        """Return the map displayed by this canvas"""
        return self.map

    def redraw(self, *args):
        self.Refresh(0)

    def full_redraw(self, *args):
        self.bitmap = None
        self.redraw()

    def projection_changed(self, *args):
        self.FitMapToWindow()
        self.full_redraw()

    def set_view_transform(self, scale, offset):
        self.scale = scale
        self.offset = offset
        self.full_redraw()

    def proj_to_win(self, x, y):
        """\
        Return the point in  window coords given by projected coordinates x y
        """
        offx, offy = self.offset
        return (self.scale * x + offx, -self.scale * y + offy)

    def win_to_proj(self, x, y):
        """\
        Return the point in projected coordinates given by window coords x y
        """
        offx, offy = self.offset
        return ((x - offx) / self.scale, (offy - y) / self.scale)

    def FitRectToWindow(self, rect):
        """Fit the rectangular region given by rect into the window.
        
        Set scale so that rect (in projected coordinates) just fits into
        the window and center it.
        """
        width, height = self.GetSizeTuple()
        llx, lly, urx, ury = rect
        if llx == urx or lly == ury:
            # zero with or zero height. Do Nothing
            return
        scalex = width / (urx - llx)
        scaley = height / (ury - lly)
        scale = min(scalex, scaley)
        offx = 0.5 * (width - (urx + llx) * scale)
        offy = 0.5 * (height + (ury + lly) * scale)
        self.set_view_transform(scale, (offx, offy))

    def FitMapToWindow(self):
        """Fit the map to the window
        
        Set the scale so that the map fits exactly into the window and
        center it in the window.
        """
        bbox = self.map.ProjectedBoundingBox()
        if bbox is not None:
            self.FitRectToWindow(bbox)

    def ZoomFactor(self, factor, center = None):
        """Multiply the zoom by factor and center on center.

        The optional parameter center is a point in window coordinates
        that should be centered. If it is omitted, it defaults to the
        center of the window
        """
        width, height = self.GetSizeTuple()
        scale = self.scale * factor
        offx, offy = self.offset
        if center is not None:
            cx, cy = center
        else:
            cx = width / 2
            cy = height / 2
        offset = (factor * (offx - cx) + width / 2,
                  factor * (offy - cy) + height / 2)
        self.set_view_transform(scale, offset)

    def ZoomOutToRect(self, rect):
        """Zoom out to fit the currently visible region into rect.

        The rect parameter is given in window coordinates
        """
        # determine the bbox of the displayed region in projected
        # coordinates
        width, height = self.GetSizeTuple()
        llx, lly = self.win_to_proj(0, height - 1)
        urx, ury = self.win_to_proj(width - 1, 0)

        sx, sy, ex, ey = rect
        scalex = (ex - sx) / (urx - llx)
        scaley = (ey - sy) / (ury - lly)
        scale = min(scalex, scaley)

        offx = 0.5 * ((ex + sx) - (urx + llx) * scale)
        offy = 0.5 * ((ey + sy) + (ury + lly) * scale)
        self.set_view_transform(scale, (offx, offy))

    def Translate(self, dx, dy):
        """Move the map by dx, dy pixels"""
        offx, offy = self.offset
        self.set_view_transform(self.scale, (offx + dx, offy + dy))

    def ZoomInTool(self):
        """Start the zoom in tool"""
        self.tool = ZoomInTool(self)

    def ZoomOutTool(self):
        """Start the zoom out tool"""
        self.tool = ZoomOutTool(self)

    def PanTool(self):
        """Start the pan tool"""
        self.tool = PanTool(self)

    def IdentifyTool(self):
        """Start the identify tool"""
        self.tool = IdentifyTool(self)

    def LabelTool(self):
        """Start the label tool"""
        self.tool = LabelTool(self)

    def CurrentTool(self):
        """Return the name of the current tool or None if no tool is active"""
        return self.tool and self.tool.Name() or None

    def CurrentPosition(self):
        """Return current position of the mouse in projected coordinates.

        The result is a 2-tuple of floats with the coordinates. If the
        mouse is not in the window, the result is None.
        """
        if self.current_position is not None:
            x, y = self.current_position
            return self.win_to_proj(x, y)
        else:
            return None

    def set_current_position(self, event):
        """Set the current position from event

        Should be called by all events that contain mouse positions
        especially EVT_MOTION. The event paramete may be None to
        indicate the the pointer left the window.
        """
        if event is not None:
            self.current_position = (event.m_x, event.m_y)
        else:
            self.current_position = None
        self.issue(VIEW_POSITION)

    def OnLeftDown(self, event):
        self.set_current_position(event)
        if self.tool is not None:
            self.drag_dc = wxClientDC(self)
            self.drag_dc.SetLogicalFunction(wxINVERT)
            self.drag_dc.SetBrush(wxTRANSPARENT_BRUSH)
            self.CaptureMouse()
            self.tool.MouseDown(event)
            self.tool.Show(self.drag_dc)
            self.dragging = 1

    def OnLeftUp(self, event):
        self.set_current_position(event)
        if self.dragging:
            self.ReleaseMouse()
            self.tool.Hide(self.drag_dc)
            self.tool.MouseUp(event)
            self.drag_dc = None
        self.dragging = 0

    def OnMotion(self, event):
        self.set_current_position(event)
        if self.dragging:
            self.tool.Hide(self.drag_dc)
            self.tool.MouseMove(event)
            self.tool.Show(self.drag_dc)

    def OnLeaveWindow(self, event):
        self.set_current_position(None)

    def OnSize(self, event):
        # the window's size has changed. We have to get a new bitmap. If
        # we want to be clever we could try to get by without throwing
        # everything away. E.g. when the window gets smaller, we could
        # either keep the bitmap or create the new one from the old one.
        # Even when the window becomes larger some parts of the bitmap
        # could be reused.
        self.full_redraw()

    def shape_selected(self, layer, shape):
        """Redraw the map.

        Receiver for the SELECTED_SHAPE messages. Try to redraw only
        when necessary.
        """
        # A redraw is necessary when the display has to change, which
        # means that either the status changes from having no selection
        # to having a selection shape or vice versa, or when the fact
        # whether there is a selection at all doesn't change, when the
        # shape which is selected has changed (which means that layer or
        # shapeid changes).
        if ((shape is not None or self.last_selected_shape is not None)
            and (shape != self.last_selected_shape
                 or layer != self.last_selected_layer)):
            self.full_redraw()

        # remember the selection so we can compare when it changes again.
        self.last_selected_layer = layer
        self.last_selected_shape = shape

    def unprojected_rect_around_point(self, x, y, dist):
        """return a rect dist pixels around (x, y) in unprojected corrdinates

        The return value is a tuple (minx, miny, maxx, maxy) suitable a
        parameter to a layer's ShapesInRegion method.
        """
        map_proj = self.map.projection
        if map_proj is not None:
            inverse = map_proj.Inverse
        else:
            inverse = None

        xs = []
        ys = []
        for dx, dy in ((-1, -1), (1, -1), (1, 1), (-1, 1)):
            px, py = self.win_to_proj(x + dist * dx, y + dist * dy)
            if inverse:
                px, py = inverse(px, py)
            xs.append(px)
            ys.append(py)
        return (min(xs), min(ys), max(xs), max(ys))

    def find_shape_at(self, px, py, select_labels = 0, searched_layer = None):
        """Determine the shape at point px, py in window coords

        Return the shape and the corresponding layer as a tuple (layer,
        shape).

        If the optional parameter select_labels is true (default false)
        search through the labels. If a label is found return it's index
        as the shape and None as the layer.

        If the optional parameter searched_layer is given (or not None
        which it defaults to), only search in that layer.
        """
        map_proj = self.map.projection
        if map_proj is not None:
            forward = map_proj.Forward
        else:
            forward = None

        scale = self.scale
        offx, offy = self.offset

        if select_labels:
            labels = self.map.LabelLayer().Labels()

            if labels:
                dc = wxClientDC(self)
                font = wxFont(10, wx.wxSWISS, wx.wxNORMAL, wx.wxNORMAL)
                dc.SetFont(font)
                for i in range(len(labels) - 1, -1, -1):
                    label = labels[i]
                    x = label.x
                    y = label.y
                    text = label.text
                    if forward:
                        x, y = forward(x, y)
                    x = x * scale + offx
                    y = -y * scale + offy
                    width, height = dc.GetTextExtent(text)
                    if label.halign == ALIGN_LEFT:
                        # nothing to be done
                        pass
                    elif label.halign == ALIGN_RIGHT:
                        x = x - width
                    elif label.halign == ALIGN_CENTER:
                        x = x - width/2
                    if label.valign == ALIGN_TOP:
                        # nothing to be done
                        pass
                    elif label.valign == ALIGN_BOTTOM:
                        y = y - height
                    elif label.valign == ALIGN_CENTER:
                        y = y - height/2
                    if x <= px < x + width and y <= py <= y + height:
                        return None, i

        if searched_layer:
            layers = [searched_layer]
        else:
            layers = self.map.Layers()

        for layer_index in range(len(layers) - 1, -1, -1):
            layer = layers[layer_index]

            # search only in visible layers
            if not layer.Visible():
                continue

            filled = layer.fill is not None
            stroked = layer.stroke is not None

            layer_proj = layer.projection
            if layer_proj is not None:
                inverse = layer_proj.Inverse
            else:
                inverse = None

            shapetype = layer.ShapeType()

            select_shape = -1

            # Determine the ids of the shapes that overlap a tiny area
            # around the point. For layers containing points we have to
            # choose a larger size of the box we're testing agains so
            # that we take the size of the markers into account
            # FIXME: Once the markers are more flexible this part has to
            # become more flexible too, of course
            if shapetype == SHAPETYPE_POINT:
                box = self.unprojected_rect_around_point(px, py, 5)
            else:
                box = self.unprojected_rect_around_point(px, py, 1)
            shape_ids = layer.ShapesInRegion(box)
            shape_ids.reverse()

            if shapetype == SHAPETYPE_POLYGON:
                for i in shape_ids:
                    result = point_in_polygon_shape(layer.shapefile.cobject(),
                                                    i,
                                                    filled, stroked,
                                                    map_proj, layer_proj,
                                                    scale, -scale, offx, offy,
                                                    px, py)
                    if result:
                        select_shape = i
                        break
            elif shapetype == SHAPETYPE_ARC:
                for i in shape_ids:
                    result = point_in_polygon_shape(layer.shapefile.cobject(),
                                                    i, 0, 1,
                                                    map_proj, layer_proj,
                                                    scale, -scale, offx, offy,
                                                    px, py)
                    if result < 0:
                        select_shape = i
                        break
            elif shapetype == SHAPETYPE_POINT:
                for i in shape_ids:
                    shape = layer.Shape(i)
                    x, y = shape.Points()[0]
                    if inverse:
                        x, y = inverse(x, y)
                    if forward:
                        x, y = forward(x, y)
                    x = x * scale + offx
                    y = -y * scale + offy
                    if hypot(px - x, py - y) < 5:
                        select_shape = i
                        break

            if select_shape >= 0:
                return layer, select_shape
        return None, None

    def SelectShapeAt(self, x, y, layer = None):
        """\
        Select and return the shape and its layer at window position (x, y)

        If layer is given, only search in that layer. If no layer is
        given, search through all layers.

        Return a tuple (layer, shapeid). If no shape is found, return
        (None, None).
        """
        layer, shape = result = self.find_shape_at(x, y, searched_layer=layer)
        # If layer is None, then shape will also be None. We don't want
        # to deselect the currently selected layer, so we simply select
        # the already selected layer again.
        if layer is None:
            layer = self.interactor.SelectedLayer()
        self.interactor.SelectLayerAndShape(layer, shape)
        return result

    def LabelShapeAt(self, x, y):
        """Add or remove a label at window position x, y.

        If there's a label at the given position, remove it. Otherwise
        determine the shape at the position, run the label dialog and
        unless the user cancels the dialog, add a laber.
        """
        ox = x; oy = y
        label_layer = self.map.LabelLayer()
        layer, shape_index = self.find_shape_at(x, y, select_labels = 1)
        if layer is None and shape_index is not None:
            # a label was selected
            label_layer.RemoveLabel(shape_index)
        elif layer is not None:
            text = labeldialog.run_label_dialog(self, layer.table, shape_index)
            if text:
                proj = self.map.projection
                if proj is not None:
                    map_proj = proj
                else:
                    map_proj = None
                proj = layer.projection
                if proj is not None:
                    layer_proj = proj
                else:
                    layer_proj = None

                shapetype = layer.ShapeType()
                if shapetype == SHAPETYPE_POLYGON:
                    x, y = shape_centroid(layer.shapefile.cobject(),
                                          shape_index,
                                          map_proj, layer_proj, 1, 1, 0, 0)
                    if map_proj is not None:
                        x, y = map_proj.Inverse(x, y)
                else:
                    shape = layer.Shape(shape_index)
                    if shapetype == SHAPETYPE_POINT:
                        x, y = shape.Points()[0]
                    else:
                        # assume SHAPETYPE_ARC
                        points = shape.Points()
                        x, y = points[len(points) / 2]
                    if layer_proj is not None:
                        x, y = layer_proj.Inverse(x, y)
                if shapetype == SHAPETYPE_POINT:
                    halign = ALIGN_LEFT
                    valign = ALIGN_CENTER
                elif shapetype == SHAPETYPE_POLYGON:
                    halign = ALIGN_CENTER
                    valign = ALIGN_CENTER
                elif shapetype == SHAPETYPE_ARC:
                    halign = ALIGN_LEFT
                    valign = ALIGN_CENTER
                label_layer.AddLabel(x, y, text,
                                     halign = halign, valign = valign)
