# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Functions to save a session to a file
"""

__version__ = "$Revision: 1.4 $"

import os
import string

import Thuban.Lib.fileutil

def relative_filename(dir, filename):
    """Return a filename relative to dir for the absolute file name absname.

    This is almost the same as the function in fileutil, except that dir
    can be an empty string in which case filename will be returned
    unchanged.
    """
    if dir:
        return Thuban.Lib.fileutil.relative_filename(dir, filename)
    else:
        return filename

def escape(data):
    """Escape &, \", ', <, and > in a string of data.
    """
    data = string.replace(data, "&", "&amp;")
    data = string.replace(data, "<", "&lt;")
    data = string.replace(data, ">", "&gt;")
    data = string.replace(data, '"', "&quot;")
    data = string.replace(data, "'", "&apos;")
    return data

class Saver:

    """Class to serialize a session into an XML file.

    Applications built on top of Thuban may derive from this class and
    override or extend the methods to save additinal information. This
    additional information should take the form of additional attributes
    or elements whose names are prefixed with a namespace. To define a
    namespace derived classes should extend the write_session method to
    pass the namespaces to the default implementation.
    """

    def __init__(self, session):
        self.session = session

    def write(self, file_or_filename):
        """Write the session to a file.

        The argument may be either a file object or a filename. If it's
        a filename, the file will be opened for writing. Files of
        shapefiles will be stored as filenames relative to the directory
        the file is stored in (as given by os.path.dirname(filename)) if
        they have a common parent directory other than the root
        directory.

        If the argument is a file object (which is determined by the
        presence of a write method) all filenames will be absolut
        filenames.
        """
        if hasattr(file_or_filename, "write"):
            # it's a file object
            self.file = file_or_filename
            self.dir = ""
        else:
            filename = file_or_filename
            self.dir = os.path.dirname(filename)
            self.file = open(filename, 'w')
        self.write_header()
        self.write_session(self.session)

    def write_element(self, element, attrs, empty = 0, indentation = ""):
        # Helper function to write an element open tag with attributes
        self.file.write("%s<%s" % (indentation, element))
        for name, value in attrs.items():
            self.file.write(' %s="%s"' % (escape(name), escape(value)))
        if empty:
            self.file.write("/>\n")
        else:
            self.file.write(">\n")

    def write_header(self):
        """Write the XML header"""
        write = self.file.write
        write('<?xml version="1.0" encoding="UTF-8"?>\n')
        write('<!DOCTYPE session SYSTEM "thuban.dtd">\n')

    def write_session(self, session, attrs = None, namespaces = ()):
        """Write the session and its contents

        By default, write a session element with the title attribute and
        call write_map for each map contained in the session.

        The optional argument attrs is for additional attributes and, if
        given, should be a mapping from attribute names to attribute
        values. The values should not be XML-escaped yet.

        The optional argument namespaces, if given, should be a sequence
        of (name, URI) pairs. The namespaces are written as namespace
        attributes into the session element. This is mainly useful for
        derived classes that need to store additional information in a
        thuban session file.
        """
        if attrs is None:
            attrs = {}
        attrs["title"] = session.title
        for name, uri in namespaces:
            attrs["xmlns:" + name] = uri
        self.write_element("session", attrs)
        for map in session.Maps():
            self.write_map(map)
        self.file.write('</session>\n')

    def write_map(self, map):
        """Write the map and its contents.

        By default, write a map element element with the title
        attribute, call write_projection to write the projection
        element, call write_layer for each layer contained in the map
        and finally call write_label_layer to write the label layer.
        """
        write = self.file.write
        write('\t<map title="%s">\n' % escape(map.title))
        self.write_projection(map.projection)
        for layer in map.Layers():
            self.write_layer(layer)
        self.write_label_layer(map.LabelLayer())
        write('\t</map>\n')

    def write_projection(self, projection):
        """Write the projection.
        """
        if projection and len(projection.params) > 0:
            self.file.write('\t\t<projection>\n')
            for param in projection.params:
                self.file.write('\t\t\t<parameter value="%s"/>\n'
                                % escape(param))
            self.file.write('\t\t</projection>\n')

    def write_layer(self, layer, attrs = None):
        """Write the layer.

        The optional argument attrs is for additional attributes and, if
        given, should be a mapping from attribute names to attribute
        values. The values should not be XML-escaped yet.
        """
        if attrs is None:
            attrs = {}
        attrs["title"] = layer.title
        attrs["filename"] = relative_filename(self.dir, layer.filename)
        attrs["stroke_width"] = str(layer.stroke_width)
        fill = layer.fill
        if fill is None:
            attrs["fill"] = "None"
        else:
            attrs["fill"] = fill.hex()
        stroke = layer.stroke
        if stroke is None:
            attrs["stroke"] = "None"
        else:
            attrs["stroke"] = stroke.hex()
        self.write_element("layer", attrs, empty = 1, indentation = "\t\t")

    def write_label_layer(self, layer):
        """Write the label layer.
        """
        labels = layer.Labels()
        if labels:
            self.file.write('\t\t<labellayer>\n')
            for label in labels:
                self.file.write(('\t\t\t<label x="%g" y="%g" text="%s"'
                                 ' halign="%s" valign="%s"/>\n')
                                % (label.x, label.y, label.text, label.halign,
                                   label.valign))
            self.file.write('\t\t</labellayer>\n')



def save_session(session, file, saver_class = None):
    """Save the session session to a file.

    The file argument may either be a filename or an open file object.

    The optional argument saver_class is the class to use to serialize
    the session. By default or if it's None, the saver class will be
    Saver.

    If writing the session is successful call the session's
    UnsetModified method
    """
    if saver_class is None:
        saver_class = Saver
    saver = saver_class(session)
    saver.write(file)

    # after a successful save consider the session unmodified.
    session.UnsetModified()
