# Copyright (C) 2001, 2002 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with GRASS for details.

"""
Parser for thuban session files.
"""

__version__ = "$Revision: 1.4 $"

import sys, string, os

import xml.sax
import xml.sax.handler
from xml.sax import make_parser, ErrorHandler

from Thuban.Model.session import Session
from Thuban.Model.map import Map
from Thuban.Model.layer import Layer
from Thuban.Model.color import Color
from Thuban.Model.proj import Projection


def parse_color(color):
    """Return the color object for the string color.

    Color may be either 'None' or of the form '#RRGGBB' in the usual
    HTML color notation
    """
    color = string.strip(color)
    if color == "None":
        result = None
    elif color[0] == '#':
        if len(color) == 7:
            r = string.atoi(color[1:3], 16) / 255.0
            g = string.atoi(color[3:5], 16) / 255.0
            b = string.atoi(color[5:7], 16) / 255.0
            result = Color(r, g, b)
        else:
            raise ValueError("Invalid hexadecimal color specification %s"
                             % color)
    else:
        raise ValueError("Invalid color specification %s" % color)
    return result


class ProcessSession(xml.sax.handler.ContentHandler):

    # Dictionary mapping element names (or (URI, element name) pairs for
    # documents using amespaces) to method names. The methods should
    # accept the same parameters as the startElement (or startElementNS)
    # methods. The start_dispatcher is used by the default startElement
    # and startElementNS methods to call a method for the open tag of an
    # element.
    start_dispatcher = {}

    # end_dispatcher works just like start_dispatcher but it's used by
    # endElement and endElementNS. The method whose names it maps to
    # should accept the same parameters as endElement and endElementNS.
    end_dispatcher = {}


    def __init__(self, directory):
        """Inititialize the Sax handler.

        The directory parameter should be the directory containing the
        session file. It's needed to interpret embedded relative
        filenames.
        """
        self.directory = directory
        self.chars = ''
        self.theSession = None
        self.aMap = None
        self.aLayer = None

    def startElementNS(self, name, qname, attrs):
        """Call the method given for name in self.start_dispatcher
        """
        if name[0] is None:
            method_name = self.start_dispatcher.get(name[1])
        else:
            # Dispatch with namespace
            method_name = self.start_dispatcher.get(name)
        if method_name is not None:
            getattr(self, method_name)(name, qname, attrs)

    def endElementNS(self, name, qname):
        """Call the method given for name in self.end_dispatcher
        """
        if name[0] is None:
            method_name = self.end_dispatcher.get(name[1])
        else:
            # Dispatch with namespace
            method_name = self.end_dispatcher.get(name)
        if method_name is not None:
            getattr(self, method_name)(name, qname)

    def start_session(self, name, qname, attrs):
        self.theSession = Session(attrs.get((None, 'title'), None))
    start_dispatcher['session'] = "start_session"

    def end_session(self, name, qname):
        pass
    end_dispatcher['session'] = "end_session"

    def start_map(self, name, qname, attrs):
        """Start a map."""
        self.aMap = Map(attrs.get((None, 'title'), None))
    start_dispatcher['map'] = "start_map"

    def end_map(self, name, qname):
        self.theSession.AddMap(self.aMap)
    end_dispatcher['map'] = "end_map"

    def start_projection(self, name, qname, attrs):
        self.ProjectionParams = [ ]
    start_dispatcher['projection'] = "start_projection"

    def end_projection(self, name, qname):
        self.aMap.SetProjection(Projection(self.ProjectionParams))
    end_dispatcher['projection'] = "end_projection"

    def start_parameter(self, name, qname, attrs):
        s = attrs.get((None, 'value'))
        s = str(s) # we can't handle unicode in proj
        self.ProjectionParams.append(s)
    start_dispatcher['parameter'] = "start_parameter"

    def start_layer(self, name, qname, attrs, layer_class = Layer):
        """Start a layer

        Instantiate a layer of class layer_class from the attributes in
        attrs which may be a dictionary as well as the normal SAX attrs
        object and bind it to self.aLayer.
        """
        title = attrs.get((None, 'title'), "")
        filename = attrs.get((None, 'filename'), "")
        filename = os.path.join(self.directory, filename)
        fill = parse_color(attrs.get((None, 'fill'), "None"))
        stroke = parse_color(attrs.get((None, 'stroke'), "#000000"))
        stroke_width = int(attrs.get((None, 'stroke_width'), "1"))
        self.aLayer = layer_class(title, filename, fill = fill,
                                  stroke = stroke, stroke_width = stroke_width)
    start_dispatcher['layer'] = "start_layer"

    def end_layer(self, name, qname):
        self.aMap.AddLayer(self.aLayer)
    end_dispatcher['layer'] = "end_layer"

    def start_table(self, name, qname, attrs):
        print "table title: %s" % attrs.get('title', None)
    start_dispatcher['table'] = "start_table"

    def end_table(self, name, qname):
        pass
    end_dispatcher['table'] = "end_table"

    def start_labellayer(self, name, qname, attrs):
        self.aLayer = self.aMap.LabelLayer()
    start_dispatcher['labellayer'] = "start_labellayer"

    def start_label(self, name, qname, attrs):
        x = float(attrs[(None, 'x')])
        y = float(attrs[(None, 'y')])
        text = attrs[(None, 'text')]
        halign = attrs[(None, 'halign')]
        valign = attrs[(None, 'valign')]
        self.aLayer.AddLabel(x, y, text, halign = halign, valign = valign)
    start_dispatcher['label'] = "start_label"

    def characters(self, chars):
        pass


def load_session(filename):
    """Load a Thuban session from the file object file"""
    dir = os.path.dirname(filename)
    file = open(filename)
    handler = ProcessSession(dir)

    parser = make_parser()
    parser.setContentHandler(handler)
    parser.setErrorHandler(ErrorHandler())
    parser.setFeature(xml.sax.handler.feature_namespaces, 1)
    parser.parse(file)

    session = handler.theSession
    # Newly loaded session aren't modified
    session.UnsetModified()

    return session

