/* Copyright (c) 2001 by Intevation GmbH
 * Authors:
 * Bernhard Herzog <bh@intevation.de>
 *
 * This program is free software under the GPL (>=v2)
 * Read the file COPYING coming with Thuban for details.
 */

/* module to read a shape from a shapefile with shapelib, project the
 * data with proj4 and draw it on a wxWindows DC
 */

#include <stdlib.h>
#include <math.h>

#include <Python.h>

#include <shapefil.h>

/* PVALUE is a typef in projects.h and is used in winreg.h on Windows as
 * well. We don't need it, so we temporarily rename the one from
 * projects.h */
#define PVALUE PROJ_PVALUE

#include <projects.h>

/* undefine it again */
#undef PVALUE

#include <wx/wx.h>


/* helper function to extract the pointer value from a swig ptr string
 */
static void *
decode_pointer(char *string)
{
  unsigned long p = 0;

  /* Pointer values must start with leading underscore */
  if (*string == '_')
  {
      string++;
      /* Extract hex value from pointer */
      while (*string)
      {
	  if ((*string >= '0') && (*string <= '9'))
	    p = (p << 4) + (*string - '0');
	  else if ((*string >= 'a') && (*string <= 'f'))
	    p = (p << 4) + ((*string - 'a') + 10);
	  else
	    break;
	  string++;
      }
  }
  return (void*)p;
}

/* Return the object wrapped by the SWIG shadow object shadow */
static void*
get_pointer(PyObject* shadow)
{
    PyObject * string;
    void * result = NULL;

    string = PyObject_GetAttrString(shadow, "this");
    if (string && PyString_Check(string))
    {
	result = decode_pointer(PyString_AsString(string));
    }
    Py_XDECREF(string);

    return result;
}

/* Write the projection for the Python object object into *output and
 * return true. If unsuccessful set a Python exception, do not modify
 * *output and return false.
 *
 * If object is None, *output will be NULL, if it's a cobject, its
 * voidptr is assumed to be a PJ*, otherwise, call its cobject() method
 * which is expected to return a CObject with the PJ*.
 */
static int
extract_projection(PyObject * object, PJ**output)
{
    void * value = NULL;
    /* to_decref may hold an object reference that has to be DECREF'd
     * before return */
    PyObject * to_decref = NULL;
    
    if (Py_None == object)
    {
	value = NULL;
    }
    else
    {
	PyObject *cobject;

	if (!PyCObject_Check(object))
	{
	    cobject = PyObject_CallMethod(object, "cobject", NULL);
	    if (!cobject)
		return 0;
	    /* cobject has to be DECREF'd before return */
	    to_decref = cobject;
	}
	else
	    cobject = object;

	/* now cobject should be a CObject containing the PJ* pointer */
	if (PyCObject_Check(cobject))
	{
	    value = PyCObject_AsVoidPtr(cobject);
	}
	else
	{
	    PyErr_SetString(PyExc_TypeError,
			    "The projection must be either None, "
			    "a projection object or a cobject with a "
			    "PJ pointer");
	    Py_XDECREF(to_decref);
	    return 0;
	}
    }

    *output = (PJ*)value;
    return 1;
}


/* Project the point (x, y) with the projections forward and inverse and
 * scale it with scalex, scaley and offset it by offx, offy and store in
 * *xdest, and *ydest.
 */
static void
project_point(double * xdest, double *ydest,
	      PJ* forward, PJ* inverse,
	      double scalex, double scaley, double offx, double offy,
	      double x, double y)
{
    projUV proj_point;
    if (forward)
    {
	proj_point.u = x * DEG_TO_RAD;
	proj_point.v = y * DEG_TO_RAD;
	proj_point = pj_fwd(proj_point, forward);
	*xdest = (int)(scalex * proj_point.u + offx);
	*ydest = (int)(scaley * proj_point.v + offy);
    }
    else
    {
	*xdest = (int)(scalex * x + offx);
	*ydest = (int)(scaley * y + offy);
    }
}


/* Return the projected points as a wxPoint array. The array returned
 * has a length of num_vertices + num_parts - 1
 *
 * The return value is suitable as a parameter to DrawPolygon with a
 * non-bull brush so that the shape will be filled correctly, even
 * shapes with holes or otherwise consisting of multiple parts. This is
 * achieved by connecting the start/end points of the parts with lines
 * in such a way that these lines are given twice, but in opposite
 * directions. This trick works on bothj X11 and Windows 2000 (perhaps
 * on other windows versions too). This trick is the reason why the
 * returned array has num_vertices + num_parts - 1 items.
 */
static wxPoint *
project_points(int num_vertices, int num_parts,
	       double * xs, double * ys, int * part_start,
	       PJ* forward, PJ* inverse,
	       double scalex, double scaley, double offx, double offy)
{
    int i;
    int num_points = num_vertices + num_parts - 1;
    wxPoint* points = (wxPoint*)malloc(num_points * sizeof(wxPoint));
    if (!points)
    {
	PyErr_NoMemory();
	return NULL;
    }

    for (i = 0; i < num_vertices; i++)
    {
	projUV proj_point;
	if (forward)
	{
	    proj_point.u = xs[i] * DEG_TO_RAD;
	    proj_point.v = ys[i] * DEG_TO_RAD;
	    proj_point = pj_fwd(proj_point, forward);
	    points[i].x = (int)(scalex * proj_point.u + offx);
	    points[i].y = (int)(scaley * proj_point.v + offy);
	}
	else
	{
	    points[i].x = (int)(scalex * xs[i] + offx);
	    points[i].y = (int)(scaley * ys[i] + offy);
	}
    }

    /* link all the parts so that we can draw a multipart polygon with
     * one DrawPolgon call. If the points array as a whole is used for
     * DrawPolgon, the connections between the start points of the
     * subparts are already in the polygon, but only in one direction.
     * Here we add the same connections, but in the opposite order and
     * direction.
     */
    for (i = num_parts - 1; i > 0; i--)
    {
	points[num_vertices + num_parts - 1 - i] = points[part_start[i]];
    }

    return points;
}


static PyObject *
draw_polygon_shape(PyObject *, PyObject * args)
{
    int shape_index = 0;
    PyObject * handle_cobject;
    PyObject * forward_cobject;
    PyObject * inverse_cobject;
    PyObject * brush_shadow;
    PyObject * pen_shadow;
    PyObject * dc_shadow;
    double scalex, scaley, offx, offy;
    SHPHandle handle;
    PJ * forward = NULL;
    PJ * inverse = NULL;
    SHPObject * shape;
    wxPoint* points;
    wxDC * dc;
    wxPen * pen;
    wxBrush * brush;
    int num_points;

    if (!PyArg_ParseTuple(args, "O!iOOOOOdddd",
			  &PyCObject_Type, &handle_cobject,
			  &shape_index, &dc_shadow,
			  &pen_shadow, &brush_shadow,
			  &forward_cobject, &inverse_cobject,
			  &scalex, &scaley, &offx, &offy))
	return NULL;

    handle = (SHPHandle)PyCObject_AsVoidPtr(handle_cobject);
    dc = (wxDC*)get_pointer(dc_shadow);
    if (!dc)
    {
	PyErr_SetString(PyExc_TypeError,
			"third argument must be a wxDC instance");
	return NULL;
    }

    if (pen_shadow == Py_None)
    {
	pen = NULL;
    }
    else
    {
	pen = (wxPen*)get_pointer(pen_shadow);
	if (!pen)
	{
	    PyErr_SetString(PyExc_TypeError,
		       "fourth argument must be a wxPen instance or None");
	    return NULL;
	}
    }

    if (brush_shadow == Py_None)
    {
	brush = NULL;
    }
    else
    {
	brush = (wxBrush*)get_pointer(brush_shadow);
	if (!brush)
	{
	    PyErr_SetString(PyExc_TypeError,
			"fifth argument must be a wxBrush instance or None");
	    return NULL;
	}
    }

    if (!extract_projection(forward_cobject, &forward))
	return NULL;

    if (!extract_projection(inverse_cobject, &inverse))
	return NULL;

    shape = SHPReadObject(handle, shape_index);


    
    num_points = shape->nVertices + shape->nParts - 1;
    points = project_points(shape->nVertices, shape->nParts,
			    shape->padfX, shape->padfY, shape->panPartStart,
			    forward, inverse, scalex, scaley, offx, offy);
    if (brush && brush != wxTRANSPARENT_BRUSH)
    {
	dc->SetPen(*wxTRANSPARENT_PEN);
	dc->SetBrush(*brush);
	dc->DrawPolygon(num_points, points, 0, 0);
    }
    if (pen && pen != wxTRANSPARENT_PEN)
    {
	dc->SetPen(*pen);
	dc->SetBrush(*wxTRANSPARENT_BRUSH);
	for (int i = 0; i < shape->nParts; i++)
	{
	    int length;

	    if (i < shape->nParts - 1)
		length = shape->panPartStart[i + 1] - shape->panPartStart[i];
	    else
		length = shape->nVertices - shape->panPartStart[i];
	    dc->DrawPolygon(length, points + shape->panPartStart[i], 0, 0);
	}
    }
    
    free(points);

    SHPDestroyObject(shape);

    Py_INCREF(Py_None);
    return Py_None;
}



/* determine whether the line fom (SX, SY) to (EX, EY) is `hit' by a
 * click at (PX, PY), or whether a polygon containing this line is hit
 * in the interior at (PX, PY).
 *
 * Return -1 if the line it self his hit. Otherwise, return +1 if a
 * horizontal line from (PX, PY) to (-Infinity, PY) intersects the line
 * and 0 if it doesn't.
 *
 * The nonnegative return values can be used to determine whether (PX, PY)
 * is an interior point of a polygon according to the even-odd rule.
 */

#define PREC_BITS 4

static int
test_line(int sx, int sy, int ex, int ey, int px, int py)
{
    long vx, vy, dx, dy, len, dist, not_horizontal;

    if (ey < sy)
    {
	dist = ex; ex = sx; sx = dist;
	dist = ey; ey = sy; sy = dist;
    }
    not_horizontal = ey > sy + (2 << PREC_BITS);
    if (not_horizontal && (py >= ey || py < sy))
	return 0;

    vx = ex - sx; vy = ey - sy;

    len = (long)sqrt(vx * vx + vy * vy);
    if (!len)
	/* degenerate case of coincident end points. Assumes that some
	 * other part of the code has already determined whether the end
	 * point is hit.
	 */
	return 0;

    dx = px - sx; dy = py - sy;
    dist = vx * dy - vy * dx;
    if ((not_horizontal || (px >= sx && px <= ex) || (px >= ex && px <= sx))
	&& abs(dist) <= (len << (PREC_BITS + 1)))
	return -1;

    /* horizontal lines (vy == 0) always return 0 here. */
    return vy && py < ey && py >= sy && dx * abs(vy) > vx * abs(dy);
}



static PyObject *
point_in_polygon_shape(PyObject *, PyObject * args)
{
    int shape_index = 0;
    PyObject * handle_cobject;
    PyObject * forward_cobject;
    PyObject * inverse_cobject;
    int filled, stroked;
    double scalex, scaley, offx, offy;
    int px, py;
    SHPHandle handle;
    PJ * forward = NULL;
    PJ * inverse = NULL;
    SHPObject * shape;
    wxPoint* points;
    int num_points;
    int cross_count, linehit;
    int result;

    if (!PyArg_ParseTuple(args, "O!iiiOOddddii",
			  &PyCObject_Type, &handle_cobject,
			  &shape_index, &filled, &stroked,
			  &forward_cobject, &inverse_cobject,
			  &scalex, &scaley, &offx, &offy, &px, &py))
	return NULL;

    handle = (SHPHandle)PyCObject_AsVoidPtr(handle_cobject);

    if (!extract_projection(forward_cobject, &forward))
	return NULL;

    if (!extract_projection(inverse_cobject, &inverse))
	return NULL;

    shape = SHPReadObject(handle, shape_index);
    
    num_points = shape->nVertices + shape->nParts - 1;
    points = project_points(shape->nVertices, shape->nParts,
			    shape->padfX, shape->padfY, shape->panPartStart,
			    forward, inverse, scalex, scaley, offx, offy);

    long scaled_px = (px << PREC_BITS) + 1, scaled_py = (py << PREC_BITS) + 1;

    cross_count = 0; linehit = 0;
    for (int part = 0; part < shape->nParts; part++)
    {
	int start, end;

	if (part < shape->nParts - 1)
	{
	    start = shape->panPartStart[part];
	    end = shape->panPartStart[part + 1] - 1;
	}
	else
	{
	    start = shape->panPartStart[part];
	    end = shape->nVertices - 1;
	}

	for (int vertex = start; vertex < end; vertex++)
	{
	    //printf("test_line: %d, %d: %d, %d -- %d, %d",
	    //	   px, py, points[vertex].x, points[vertex].y,
	    //	   points[vertex + 1].x, points[vertex + 1].y);
	    result = test_line(points[vertex].x << PREC_BITS,
				points[vertex].y << PREC_BITS,
				points[vertex + 1].x << PREC_BITS,
				points[vertex + 1].y << PREC_BITS,
				scaled_px, scaled_py);
	    //printf(":\t%d\n", result);
	    if (result < 0)
	    {
		linehit = 1;
		break;
	    }
	    cross_count += result;
	}
	if (linehit)
	    break;
    }

    free(points);

    SHPDestroyObject(shape);

    if (filled)
    {
	if (stroked && linehit)
	    result = -1;
	else
	    result = cross_count % 2;
    }
    else if (stroked)
    {
	if (linehit)
	    result = -1;
	else
	    result = 0;
    }
    else
	result = 0;
		

    return PyInt_FromLong(result);
}


static PyObject*
shape_centroid(PyObject *self, PyObject *args)
{
    int shape_index = 0;
    PyObject * handle_cobject;
    PyObject * forward_cobject;
    PyObject * inverse_cobject;
    double scalex, scaley, offx, offy;
    SHPHandle handle;
    PJ * forward = NULL;
    PJ * inverse = NULL;
    SHPObject * shape;
    double centroidx = 0, centroidy = 0;
    double sum = 0, area;

    if (!PyArg_ParseTuple(args, "O!iOOdddd",
			  &PyCObject_Type, &handle_cobject,
			  &shape_index, &forward_cobject, &inverse_cobject,
			  &scalex, &scaley, &offx, &offy))
	return NULL;

    handle = (SHPHandle)PyCObject_AsVoidPtr(handle_cobject);

    if (!extract_projection(forward_cobject, &forward))
	return NULL;

    if (!extract_projection(inverse_cobject, &inverse))
	return NULL;

    shape = SHPReadObject(handle, shape_index);

    for (int part = 0; part < shape->nParts; part++)
    {
	int start, end;
	double lastx, lasty, x, y;

	if (part < shape->nParts - 1)
	{
	    start = shape->panPartStart[part];
	    end = shape->panPartStart[part + 1];
	}
	else
	{
	    start = shape->panPartStart[part];
	    end = shape->nVertices;
	}

	project_point(&lastx, &lasty, forward, inverse,
		      scalex, scaley, offx, offy,
		      shape->padfX[start],
		      shape->padfY[start]);
	for (int vertex = start + 1; vertex < end; vertex++)
	{
	    project_point(&x, &y, forward, inverse,
			  scalex, scaley, offx, offy,
			  shape->padfX[vertex],
			  shape->padfY[vertex]);
	    area = x * lasty - lastx * y;
	    centroidx += area * (x + lastx);
	    centroidy += area * (y + lasty);
	    sum += area;
	    lastx = x;
	    lasty = y;
	}
    }

    SHPDestroyObject(shape);

    return Py_BuildValue("dd", centroidx / (3 * sum), centroidy / (3 * sum));
}



static PyMethodDef wxproj_methods[] = {
    { "draw_polygon_shape", draw_polygon_shape, METH_VARARGS },
    { "point_in_polygon_shape", point_in_polygon_shape, METH_VARARGS },
    { "shape_centroid", shape_centroid, METH_VARARGS },
    {NULL, NULL}
};


#ifdef __cplusplus
extern "C" 
#endif
void initwxproj(void)
{
    Py_InitModule("wxproj", wxproj_methods);
}



