#! /usr/bin/python
# Copyright (c) 2001 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.5 $"

from wxPython.wx import *

from Thuban.Model.messages import MAPS_CHANGED, MAP_PROJECTION_CHANGED, \
     LAYERS_CHANGED, LAYER_LEGEND_CHANGED, LAYER_VISIBILITY_CHANGED
from Thuban.Model.layer import Layer, shapetype_names
from Thuban.Model.map import Map

from dialogs import NonModalDialog
from messages import SESSION_CHANGED, SELECTED_LAYER

def color_string(color):
    if color is None:
        return "None"
    return "(%.3f, %.3f, %.3f)" % (color.red, color.green, color.blue)


class SessionTreeCtrl(wxTreeCtrl):

    def __init__(self, parent, ID, app):
        # Use the WANTS_CHARS style so the panel doesn't eat the Return key.
        wxTreeCtrl.__init__(self, parent, ID)

        self.app = app
        # boolean to indicate that we manipulate the selection ourselves
        # so that we can ignore the selection events generated
        self.changing_selection = 0

        # Dictionary mapping layer id's to tree items
        self.layer_to_item = {}

        self.app.Subscribe(SESSION_CHANGED, self.session_changed)
        self.app.interactor.Subscribe(SELECTED_LAYER, self.layer_selected)
        # pretend the session has changed to build the initial tree
        self.session_changed()

        EVT_TREE_SEL_CHANGED(self, self.GetId(), self.OnSelChanged)

    def update_tree(self, *args):
        """Clear and rebuild the tree"""
        self.DeleteAllItems()
        session = self.app.session
        root = self.AddRoot("Session: %s" % session.title)
        self.layer_to_item.clear()
        for map in session.Maps():
            mapitem = self.AppendItem(root, "Map: %s" % map.title)
            self.SetPyData(mapitem, map)
            if map.projection and len(map.projection.params) > 0:
                projectionitem = self.AppendItem(mapitem, "Projection")
                for param in map.projection.params:
                    parameteritem = self.AppendItem(projectionitem, str(param))
                self.Expand(projectionitem)

            layers = map.Layers()
            for layer_index in range(len(layers) - 1, -1, -1):
                layer = layers[layer_index]
                idata = wxTreeItemData()
                idata.SetData(layer)
                layeritem = self.AppendItem(mapitem,
                                            "Layer '%s'" % layer.Title(),
                                            data = idata)
                self.layer_to_item[id(layer)] = layeritem
                if layer is self.app.interactor.selected_layer:
                    self.SelectItem(layeritem)
                if isinstance(layer, Layer):
                    if layer.Visible():
                        text = "Shown"
                    else:
                        text = "Hidden"
                    self.AppendItem(layeritem, text)
                    self.AppendItem(layeritem, "Shapes: %d" %layer.NumShapes())
                    self.AppendItem(layeritem, ("Extents: (%g, %g, %g, %g)" 
                                                % layer.LatLongBoundingBox()))
                    self.AppendItem(layeritem,
                                    "Shapetype: %s"
                                    % shapetype_names[layer.ShapeType()])
                    self.AppendItem(layeritem,
                                    "Fill: " + color_string(layer.fill))
                    self.AppendItem(layeritem,
                                    "Outline: " + color_string(layer.stroke))
                self.Expand(layeritem)
            self.Expand(mapitem)
        self.Expand(root)

    def session_changed(self, *args):
        for channel in (MAPS_CHANGED,
                        MAP_PROJECTION_CHANGED,
                        LAYERS_CHANGED,
                        LAYER_LEGEND_CHANGED,
                        LAYER_VISIBILITY_CHANGED):
            self.app.session.Subscribe(channel, self.update_tree)
        self.update_tree()

    def normalize_selection(self):
        """Select the layer or map containing currently selected item"""
        item = self.GetSelection()
        while item.IsOk():
            object = self.GetPyData(item)
            if isinstance(object, Layer) or isinstance(object, Map):
                break
            item = self.GetItemParent(item)

        self.changing_selection = 1
        try:
            self.SelectItem(item)
        finally:
            self.changing_selection = 0

    def SelectedLayer(self):
        """Return the layer object currently selected in the tree.
        Return None if no layer is selected"""
        layer = self.GetPyData(self.GetSelection())
        if isinstance(layer, Layer):
            return layer
        return None

    def OnSelChanged(self, event):
        if self.changing_selection:
            # we're changing the selection ourselves (probably through
            # self.normalize_selection(). ignore the event.
            return
        self.normalize_selection()
        # SelectedLayer returns None if no layer is selected. Since
        # passing None to interactor.SelectLayer deselects the layer we
        # can simply pass the result of SelectedLayer on in all cases
        self.app.interactor.SelectLayer(self.SelectedLayer())

    def layer_selected(self, layer):
        item = self.layer_to_item.get(id(layer))
        if item is not None and item != self.GetSelection():
            self.SelectItem(item)


class SessionTreeView(NonModalDialog):

    """Non modal dialog showing the session as a tree"""

    def __init__(self, parent, app, name):
        NonModalDialog.__init__(self, parent, app.interactor, name, "Session")
        self.tree = SessionTreeCtrl(self, -1, app)

    def OnClose(self, event):
        #self.interactor.Unsubscribe(SELECTED_SHAPE, self.select_shape)
        NonModalDialog.OnClose(self, event)

