# Copyright (c) 2001 by Intevation GmbH
# Authors:
# Frank Koormann    <frank.koormann@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Dialogs to the geographic projection library PROJ4
"""

__version__ = "$Revision: 1.3 $"

from wxPython.wx import *
from string import split

ID_PROJECTION_EDIT   = 4010
ID_PROJECTION_SELECT = 4011
ID_PROJECTION_OK     = 4001
ID_PROJECTION_CANCEL = 4002

ID_UTM_DIALOG_OK     = 4101
ID_UTM_DIALOG_CANCEL = 4102

projectionDict = {'None' : 'None', 'UTM' : 'utm'}
projectionMapping = {'None' : 'None', 'utm' : 'UTM'}

class Proj4Dialog(wxDialog):

    """Let the user select a projection and specify related parameters"""
    
    def __init__(self, parent, projectionParamsList):
        wxDialog.__init__(self, parent, -1, "Projection",
                          style = wxRESIZE_BORDER)

        self.dialogLayout()
        self.projectionParams={}
        combo = self.projection
        if projectionParamsList is not None:
            for element in projectionParamsList:
                key, val = split(element,'=')
                self.projectionParams[key] = val
            if self.projectionParams.has_key('proj'):
                proj = projectionMapping[self.projectionParams['proj']]
                combo.SetSelection(combo.FindString(proj))
            else:
                combo.SetSelection(combo.FindString('None'))
            self.UpdateProjectionInfo()
        else:
            self.projection.SetValue('None')
            self.UpdateProjectionInfo()


    def dialogLayout(self):
        topBox = wxBoxSizer(wxVERTICAL)

        # Projection selection:
        projectionBox = wxBoxSizer(wxHORIZONTAL)
        projectionBox.Add(wxStaticText(self, -1, "Projection"), 
            0, wxALIGN_CENTER|wxALL, 4)
        
        self.projection = wxComboBox(self, ID_PROJECTION_SELECT, "", 
                                     style = wxCB_READONLY)
        for projection in projectionDict.keys():
            self.projection.Append(projection)
        projectionBox.Add(self.projection, 0, wxALL, 4)
        projectionBox.Add(wxButton(self, ID_PROJECTION_EDIT, "Edit"), 
                          0, wxALL, 4) 
        EVT_COMBOBOX(self, ID_PROJECTION_SELECT, self.OnProj4Select)
        EVT_BUTTON(self, ID_PROJECTION_EDIT, self.OnEdit)
        
        topBox.Add(projectionBox, 0, 0)

        # Info about current projection:
        self.projectionInfo = wxTextCtrl(self, -1, "", 
                                         style = wxTE_MULTILINE|wxTE_READONLY,
                                         size = (-1, 60))
        topBox.Add(self.projectionInfo,1,wxEXPAND|wxALL, 4)

        # Control buttons: 
        buttonBox = wxBoxSizer( wxHORIZONTAL ) 
        buttonBox.Add(wxButton(self, ID_PROJECTION_OK, "OK"), 
                      0, wxALL, 4) 
        buttonBox.Add(wxButton(self, ID_PROJECTION_CANCEL, "Cancel"), 
                      0, wxALL, 4) 
        topBox.Add(buttonBox, 0, wxALIGN_CENTER_HORIZONTAL|wxALL, 10)

        EVT_BUTTON(self, ID_PROJECTION_OK, self.OnProj4OK)
        EVT_BUTTON(self, ID_PROJECTION_CANCEL, self.OnProj4Cancel)

        self.SetAutoLayout(true)
        self.SetSizer(topBox)
        topBox.Fit(self)
        topBox.SetSizeHints(self)

    def OnProj4Select(self, event):
        projection = self.projection.GetStringSelection()
        if projection == 'None':
            self.projectionParams = {}
            self.UpdateProjectionInfo()

    def OnEdit(self, event):
        projection = self.projection.GetStringSelection()
        if projection is not 'None':
            dialogname = getattr(self, 'launch' + projection + 'Dialog', None)
            if dialogname is not None:
                if dialogname(self):
                    self.UpdateProjectionInfo()
                else:
                    pass
        else:
            self.projectionParams = {}

    def UpdateProjectionInfo(self):
        self.projectionInfo.Clear()
        for key in self.projectionParams.keys():
            self.projectionInfo.AppendText(key+": "
                +str(self.projectionParams[key])+"\n")
        

    def OnProj4OK(self, event):
        self.EndModal(wxID_OK)

    def OnProj4Cancel(self, event):
        self.EndModal(wxID_CANCEL)

    def launchUTMDialog(self, parent):
        dlg = UTMDialog(parent, parent.projectionInfo )
        if dlg.ShowModal():
            return true
        else:
            return false

    def GetParams(self):
        if len(self.projectionParams.keys()) > 0:    
            projection = []
            for key in self.projectionParams.keys():
                projection.append(key+"="+str(self.projectionParams[key]))
        else:
            projection=None

        return projection



class UTMDialog(wxDialog):

    """Let the user specify parameters for UTM projection (Zone, Spheroid)"""
    
    def __init__(self, parent, projection):
        wxDialog.__init__(self, parent, -1, "Projection: UTM Parameters",
                          wxDefaultPosition, wxSize(200, 100))
        self.parent = parent
        self.dialogLayout()
        if self.parent.projectionParams.has_key('zone'):
            text = str(self.parent.projectionParams['zone'])
            self.zone.SetSelection(self.zone.FindString(text))
        if self.parent.projectionParams.has_key('ellps'):
            text = str(self.parent.projectionParams['ellps'])
            self.ellps.SetSelection(self.ellps.FindString(text))

    def dialogLayout(self):
        topBox = wxBoxSizer(wxVERTICAL)

        zoneBox = wxBoxSizer(wxHORIZONTAL)
        zoneBox.Add(wxStaticText(self, -1, "UTM Zone"), 
                    0, wxALIGN_CENTER|wxALL, 4)
        self.zone = wxComboBox(self, -1, "", style = wxCB_READONLY)
        for zone in range(1,61):
            self.zone.Append(str(zone))
        zoneBox.Add(self.zone, 0, wxALIGN_CENTER|wxALL, 4)

        topBox.Add(zoneBox, 1, wxEXPAND|wxALL, 4)

        ellipsoidBox = wxBoxSizer(wxHORIZONTAL)
        ellipsoidBox.Add(wxStaticText(self, -1, "Ellipsoid"), 
                         0, wxALIGN_CENTER|wxALL, 4)
        self.ellps = wxComboBox(self, -1, "", style = wxCB_READONLY)
        for ellps in ["clrk66", "GRS80"]:
            self.ellps.Append(ellps)
        ellipsoidBox.Add(self.ellps, 0, wxALIGN_CENTER|wxALL, 4)

        topBox.Add(ellipsoidBox, 1, wxEXPAND|wxALL, 4)

        buttonBox = wxBoxSizer(wxHORIZONTAL)
        buttonBox.Add(wxButton(self, ID_UTM_DIALOG_OK, "OK"), 
                      0, wxALL, 4) 
        buttonBox.Add(wxButton(self, ID_UTM_DIALOG_CANCEL, "Cancel"), 
                      0, wxALL, 4) 
        topBox.Add(buttonBox, 0, wxALIGN_CENTER_HORIZONTAL|wxALL, 10)
        EVT_BUTTON(self, ID_UTM_DIALOG_OK, self.OnOK)
        EVT_BUTTON(self, ID_UTM_DIALOG_CANCEL, self.OnCancel)

        self.SetAutoLayout(true)
        self.SetSizer(topBox)
        topBox.Fit(self)
        topBox.SetSizeHints(self)

    def OnOK(self, event):
        self.parent.projectionParams = {}
        self.parent.projectionParams['zone'] = self.zone.GetStringSelection()
        self.parent.projectionParams['ellps'] = self.ellps.GetStringSelection()
        self.parent.projectionParams['proj'] = "utm"
        self.Close(true)

    def OnCancel(self, event):
        self.Close(false)

