# Copyright (c) 2001 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.3 $"

from Thuban.Lib.connector import Publisher

from messages import MAPS_CHANGED, LAYERS_CHANGED, MAP_PROJECTION_CHANGED, \
     LAYER_LEGEND_CHANGED, FILENAME_CHANGED

from base import TitledObject, Modifiable

from map import Map


class Session(TitledObject, Modifiable):

    """A complete session.

    A Session consists of arbitrary numbers of maps and tables

    Session objects send the following events:

        TITLE_CHANGED -- The title has changed. Parameters: the session.

        FILENAME_CHANGED -- The filename has changed. No parameters.

        MAPS_CHANGED -- Maps were added, removed.

        LAYERS_CHANGED -- Same as the map's event of the same name.
                          It's simply resent from the session to make
                          subscriptions easier.
    """

    def __init__(self, title):
        TitledObject.__init__(self, title)
        Modifiable.__init__(self)
        self.filename = None
        self.maps = []
        self.tables = []

    def SetFilename(self, filename):
        self.filename = filename
        self.changed(FILENAME_CHANGED)

    def Maps(self):
        return self.maps

    def HasMaps(self):
        return len(self.maps) > 0

    def AddMap(self, map):
        self.maps.append(map)
        for channel in (LAYERS_CHANGED, MAP_PROJECTION_CHANGED,
                        LAYER_LEGEND_CHANGED):
            map.Subscribe(channel, self.forward, channel)
        self.changed(MAPS_CHANGED)

    def Destroy(self):
        for map in self.maps:
            map.Destroy()
        self.maps = []
        self.tables = []
        Publisher.Destroy(self)

    def forward(self, *args):
        """Reissue events"""
        if len(args) > 1:
            args = (args[-1],) + args[:-1]
        apply(self.issue, args)

    def WasModified(self):
        """Return true if the session or one of the maps was modified"""
        if self.modified:
            return 1
        else:
            for map in self.maps:
                if map.WasModified():
                    return 1
        return 0

    def UnsetModified(self):
        """Unset the modified flag of the session and the maps"""
        Modifiable.UnsetModified(self)
        for map in self.maps:
            map.UnsetModified()



def create_empty_session():
    """Return an empty session useful as a starting point"""
    session = Session('unnamed session')
    session.SetFilename('unnamed.session')
    session.AddMap(Map('unnamed map'))
    session.UnsetModified()
    return session
