# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Functions to save a session to a file
"""

__version__ = "$Revision: 1.2 $"

import os
import string

from Thuban.Lib.fileutil import relative_filename

def escape(data):
    """Escape &, \", ', <, and > in a string of data.
    """
    data = string.replace(data, "&", "&amp;")
    data = string.replace(data, "<", "&lt;")
    data = string.replace(data, ">", "&gt;")
    data = string.replace(data, '"', "&quot;")
    data = string.replace(data, "'", "&apos;")
    return data

def save_session(session, filename):
    """Save the session session to the file given by filename"""
    dir = os.path.dirname(filename)
    file = open(filename, 'w')
    write = file.write
    write('<?xml version="1.0" encoding="UTF-8"?>\n')
    write('<!DOCTYPE session SYSTEM "thuban.dtd">\n')
    write('<session title="%s">\n' % escape(session.title))
    for map in session.Maps():
        write('\t<map title="%s">\n' % escape(map.title))
        if map.projection and len(map.projection.params) > 0:
            write('\t\t<projection>\n')
            for param in map.projection.params:
                write('\t\t\t<parameter value="%s"/>\n' % escape(param))
            write('\t\t</projection>\n')

        for layer in map.Layers():
            fill = layer.fill
            if fill is None:
                fill = "None"
            else:
                fill = fill.hex()
            stroke = layer.stroke
            if stroke is None:
                stroke = "None"
            else:
                stroke = stroke.hex()
            write(('\t\t<layer title="%s" filename="%s"'
                   ' fill="%s" stroke="%s" stroke_width="%d"/>\n') %
                  (escape(layer.title),
                   escape(relative_filename(dir, layer.filename)),
                   fill, stroke, layer.stroke_width))
        labels = map.LabelLayer().Labels()
        if labels:
            write('\t\t<labellayer>\n')
            for label in labels:
                write(('\t\t\t<label x="%g" y="%g" text="%s"'
                       ' halign="%s" valign="%s"/>\n')
                      % (label.x, label.y, label.text, label.halign,
                         label.valign))
            write('\t\t</labellayer>\n')
        write('\t</map>\n')
    write('</session>\n')

    # after a successful save consider the session unmodified.
    session.UnsetModified()
