# Copyright (c) 2001 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.2 $"

from messages import LAYERS_CHANGED, MAP_PROJECTION_CHANGED, \
     CHANGED, LAYER_PROJECTION_CHANGED, LAYER_LEGEND_CHANGED, \
     LAYER_VISIBILITY_CHANGED

from base import TitledObject, Modifiable

from label import LabelLayer



class Map(TitledObject, Modifiable):

    """Represent a map. A map is simply a list of layers.

    Map objects send the following message types:

        TITLE_CHANGED -- The title has changed. Parameter: the map.

        LAYERS_CHANGED -- Layers were added, removed or rearranged.
                        Parameters: the map

        MAP_PROJECTION_CHANGED -- the map's projection has changed.
                        Parameter: the map
    """

    forwarded_channels = (LAYER_PROJECTION_CHANGED,
                          LAYER_LEGEND_CHANGED,
                          LAYER_VISIBILITY_CHANGED)

    def __init__(self, title, projection = None):
        """Initialize the map."""
        TitledObject.__init__(self, title)
        self.layers = []
        self.label_layer = LabelLayer("Labels")
        self.label_layer.Subscribe(CHANGED, self.forward, LAYERS_CHANGED)
        self.projection = projection

    def Destroy(self):
        for layer in self.layers:
            layer.Destroy()
        self.label_layer.Unsubscribe(CHANGED, self.forward, LAYERS_CHANGED)
        Modifiable.Destroy(self)

    def AddLayer(self, layer):
        self.layers.append(layer)
        for channel in self.forwarded_channels:
            layer.Subscribe(channel, self.forward, channel)
        self.changed(LAYERS_CHANGED, self)

    def RemoveLayer(self, layer):
        for channel in self.forwarded_channels:
            layer.Unsubscribe(channel, self.forward, channel)
        self.layers.remove(layer)
        self.changed(LAYERS_CHANGED, self)
        layer.Destroy()

    def LabelLayer(self):
        """Return the Map's label layer"""
        return self.label_layer

    def Layers(self):
        return self.layers

    def HasLayers(self):
        return len(self.layers) > 0

    def RaiseLayer(self, layer):
        index = self.layers.index(layer)
        if index < len(self.layers) - 1:
            del self.layers[index]
            self.layers.insert(index + 1, layer)
        self.changed(LAYERS_CHANGED, self)

    def LowerLayer(self, layer):
        index = self.layers.index(layer)
        if index > 0:
            del self.layers[index]
            self.layers.insert(index - 1, layer)
        self.changed(LAYERS_CHANGED, self)

    def BoundingBox(self):
        if not self.layers:
            return None
        llx = []
        lly = []
        urx = []
        ury = []
        for layer in self.layers:
            if layer is self.label_layer:
                continue
            left, bottom, right, top = layer.LatLongBoundingBox()
            llx.append(left)
            lly.append(bottom)
            urx.append(right)
            ury.append(top)
        return (min(llx), min(lly), max(urx), max(ury))

    def ProjectedBoundingBox(self):
        # This simply returns the rectangle given by the projected
        # corners of the non-projected bbox.
        bbox = self.BoundingBox()
        if bbox is not None and self.projection is not None:
            bbox = self.projection.ForwardBBox(bbox)
        return bbox

    def SetProjection(self, projection):
        self.projection = projection
        self.changed(MAP_PROJECTION_CHANGED, self)

    def forward(self, *args):
        """Reissue events"""
        if len(args) > 1:
            args = (args[-1],) + args[:-1]
        apply(self.issue, args)

    def WasModified(self):
        """Return true if the map or one of the layers was modified"""
        if self.modified:
            return 1
        else:
            for layer in self.layers:
                if layer.WasModified():
                    return 1
            return self.label_layer.WasModified()

    def UnsetModified(self):
        """Unset the modified flag of the map and the layers"""
        Modifiable.UnsetModified(self)
        for layer in self.layers:
            layer.UnsetModified()
        self.label_layer.UnsetModified()


