# Copyright (C) 2001, 2002 by Intevation GmbH
# Authors:
# Jan-Oliver Wagner <jan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with GRASS for details.

"""
Parser for thuban session files.
"""

__version__ = "$Revision: 1.2 $"

import sys, string, os
from Thuban.Model.session import Session
from Thuban.Model.map import Map
from Thuban.Model.layer import Layer
from Thuban.Model.color import Color
from Thuban.Model.proj import Projection

oldPython=0

if not sys.__dict__.has_key("version_info"):
    # We can assume to have python 1.5.2 or lower here now
    oldPython=1

if oldPython:
    try:
        from xml.sax.saxexts import make_parser
        from xml.sax.saxlib import HandlerBase
        from xml.sax import saxutils
    except ImportError:
        sys.stdout.write(("You need to have Python-XML installed or"
                          " a modern Python!\n"
                          "Check www.python.org/sigs/xml-sig/\n\n"))
        raise
else:
    # Do the python 2.0 standard xml thing and map it on the old names
    import xml.sax
    import xml.sax.handler
    HandlerBase=xml.sax.handler.ContentHandler
    from xml.sax import make_parser

class testSAXContentHandler(HandlerBase):
# SAX compliant
    def characters(self, ch, start, length):
        pass
    
def test_for_broken_SAX():
    ch=testSAXContentHandler()
    try:
        xml.sax.parseString("""<?xml version="1.0"?>
            <child1 name="paul">Text goes here</child1>
        """,ch)
    except TypeError:
        return 1
    return 0


def parse_color(color):
    """
    Return the color object for the string color. Color may be either
    'None' or of the form '#RRGGBB' in the usual HTML color notation
    """
    color = string.strip(color)
    if color == "None":
        result = None
    elif color[0] == '#':
        if len(color) == 7:
            r = string.atoi(color[1:3], 16) / 255.0
            g = string.atoi(color[3:5], 16) / 255.0
            b = string.atoi(color[5:7], 16) / 255.0
            result = Color(r, g, b)
        else:
            raise ValueError("Invalid hexadecimal color specification %s"
                             % color)
    else:
        raise ValueError("Invalid color specification %s" % color)
    return result


class ProcessSession(HandlerBase):

    def __init__(self, directory):
        """Inititialize the Sax handler.

        directory is the directory containing the session file. It's
        needed to interpret embedded relative filenames
        """
        self.directory = directory
        self.chars = ''
        self.theSession = None
        self.aMap = None
        self.aLayer = None

    def startElement(self, name, attrs):
        if name == 'session':
            self.theSession = Session(attrs.get('title', None))
        elif name == 'map':
            self.aMap = Map(attrs.get('title', None))
        elif name == 'projection':
            self.ProjectionParams = [ ]
        elif name == 'parameter':
            self.ProjectionParams.append(attrs.get('value', None))
        elif name == 'layer':
            title = attrs.get('title', "")
            filename = attrs.get('filename', "")
            filename = os.path.join(self.directory, filename)
            fill = parse_color(attrs.get('fill', "None"))
            stroke = parse_color(attrs.get('stroke', "#000000"))
            stroke_width = int(attrs.get("stroke_width", "1"))
            self.aLayer = Layer(title, filename, fill = fill, stroke = stroke,
                                stroke_width = stroke_width) 
        elif name == 'table':
            print "table title: %s" % attrs.get('title', None)
        elif name == 'labellayer':
            self.aLayer = self.aMap.LabelLayer()
        elif name == 'label':
            x = float(attrs['x'])
            y = float(attrs['y'])
            text = attrs['text']
            halign = attrs['halign']
            valign = attrs['valign']
            self.aLayer.AddLabel(x, y, text, halign = halign, valign = valign)
        

    if not oldPython and test_for_broken_SAX():
        # works with python 2.0, but is not SAX compliant
        def characters(self, ch):
            self.my_characters(ch)
    else:
        # SAX compliant
        def characters(self, ch, start, length):
            self.my_characters(ch[start:start+length])

    def my_characters(self, ch):
        self.chars = self.chars + ch

    def endElement(self, name): 
        # If it's not a parameter element, ignore it
        if name == 'session':
            #print "end of session"
            pass
        if name == 'map':
            self.theSession.AddMap(self.aMap)
        if name == 'projection':
            self.aMap.SetProjection(Projection(self.ProjectionParams))
        if name == 'layer':
            self.aMap.AddLayer(self.aLayer)
        if name == 'table':
            #print "end of table"
            pass

def load_session(filename):
    """Load a Thuban session from the file object file"""
    dir = os.path.dirname(filename)
    file = open(filename)
    handler = ProcessSession(dir)

    if oldPython:
        parser = make_parser()
        parser.setDocumentHandler(handler)
        parser.setErrorHandler(saxutils.ErrorPrinter())
        parser.parseFile(file)
        parser.close()
    else:
        xml.sax.parse(file,handler)
    session = handler.theSession
    # Newly loaded session aren't modified
    session.UnsetModified()

    return session

if __name__ == "__main__":
    # find out the command to run
    if len(sys.argv) > 1:
        print "usage: cat <file> | " + sys.argv[0]
    else:
        parseSession(sys.stdin)
