# Copyright (c) 2001, 2002 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.4 $"

import shapelib

from messages import LAYER_PROJECTION_CHANGED, LAYER_LEGEND_CHANGED, \
     LAYER_VISIBILITY_CHANGED

from color import Color
# Some predefined colors for internal use
_black = Color(0, 0, 0)


from table import Table

from base import TitledObject, Modifiable

class Shape:

    """Represent one shape"""

    def __init__(self, points):
        self.points = points
        #self.compute_bbox()

    def compute_bbox(self):
        xs = []
        ys = []
        for x, y in self.points:
            xs.append(x)
            ys.append(y)
        self.llx = min(xs)
        self.lly = min(ys)
        self.urx = max(xs)
        self.ury = max(ys)

    def Points(self):
        return self.points



# Shape type constants
SHAPETYPE_POLYGON = "polygon"
SHAPETYPE_ARC = "arc"
SHAPETYPE_POINT = "point"

# mapping from shapelib shapetype constants to our constants
shapelib_shapetypes = {shapelib.SHPT_POLYGON: SHAPETYPE_POLYGON,
                       shapelib.SHPT_ARC: SHAPETYPE_ARC,
                       shapelib.SHPT_POINT: SHAPETYPE_POINT}

shapetype_names = {SHAPETYPE_POINT: "Point",
                   SHAPETYPE_ARC: "Arc",
                   SHAPETYPE_POLYGON: "Polygon"}

class BaseLayer(TitledObject, Modifiable):

    """Base class for the layers."""

    def __init__(self, title, visible = 1):
        """Initialize the layer.

        title -- the title
        visible -- boolean. If true the layer is visible.
        """
        TitledObject.__init__(self, title)
        Modifiable.__init__(self)
        self.visible = visible

    def Visible(self):
        """Return true if layer is visible"""
        return self.visible

    def SetVisible(self, visible):
        """Set the layer's visibility."""
        self.visible = visible
        self.issue(LAYER_VISIBILITY_CHANGED, self)
        
        
class Layer(BaseLayer):

    """Represent the information of one geodata file (currently a shapefile)

    All children of the layer have the same type.

    A layer has fill and stroke colors. Colors should be instances of
    Color. They can also be None, indicating no fill or no stroke.

    The layer objects send the following events, all of which have the
    layer object as parameter:

        TITLE_CHANGED -- The title has changed.
        LAYER_PROJECTION_CHANGED -- the projection has changed.
        LAYER_LEGEND_CHANGED -- the fill or stroke attributes have changed

    """

    def __init__(self, title, filename, projection = None,
                 fill = None, stroke = _black, stroke_width = 1, visible = 1):
        """Initialize the layer.

        title -- the title
        filename -- the name of the shapefile
        projection -- the projection object. Its Inverse method is
               assumed to map the layer's coordinates to lat/long
               coordinates
        fill -- the fill color or None if the shapes are not filled
        stroke -- the stroke color or None if the shapes are not stroked
        visible -- boolean. If true the layer is visible.

        colors are expected to be instances of Color class
        """
        BaseLayer.__init__(self, title, visible = visible)
        self.filename = filename
        self.projection = projection
        self.fill = fill
        self.stroke = stroke
        self.stroke_width = stroke_width
        self.shapefile = None
        self.open_shapefile()
        # shapetable is the table associated with the shapefile, while
        # table is the default table used to look up attributes for
        # display
        self.shapetable = Table(filename)
        self.table = self.shapetable

    def open_shapefile(self):
        if self.shapefile is None:
            self.shapefile = shapelib.ShapeFile(self.filename)
            numshapes, shapetype, mins, maxs = self.shapefile.info()
            self.numshapes = numshapes
            self.shapetype = shapelib_shapetypes[shapetype]
            self.bbox = mins[:2] + maxs[:2]

    def BoundingBox(self):
        """Return the bounding box of the layer's shapes in their default
        coordinate system"""
        self.open_shapefile()
        return self.bbox

    def LatLongBoundingBox(self):
        """Return the layer's bounding box in lat/long coordinates"""
        llx, lly, urx, ury = self.BoundingBox()
        if self.projection is not None:
            llx, lly = self.projection.Inverse(llx, lly)
            urx, ury = self.projection.Inverse(urx, ury)
        return llx, lly, urx, ury

    def NumShapes(self):
        """Return the number of shapes in the layer"""
        self.open_shapefile()
        return self.numshapes

    def ShapeType(self):
        """Return the type of the shapes in the layer.
        This is either SHAPETYPE_POINT, SHAPETYPE_ARC or SHAPETYPE_POLYGON.
        """
        self.open_shapefile()
        return self.shapetype

    def Shape(self, index):
        """Return the shape with index index"""
        self.open_shapefile()
        shape = self.shapefile.read_object(index)
        if self.shapetype == SHAPETYPE_POINT:
            points = shape.vertices()
        else:
            #for poly in shape.vertices():
            poly = shape.vertices()[0]
            points = []
            for x, y in poly:
                points.append((x, y))
        return Shape(points)

    def SetProjection(self, projection):
        """Set the layer's projection"""
        self.projection = projection
        self.changed(LAYER_PROJECTION_CHANGED, self)

    def SetFill(self, fill):
        """Set the layer's fill color. None means the shapes are not filled"""
        self.fill = fill
        self.changed(LAYER_LEGEND_CHANGED, self)

    def SetStroke(self, stroke):
        """Set the layer's stroke color. None means the shapes are not
        stroked."""
        self.stroke = stroke
        self.changed(LAYER_LEGEND_CHANGED, self)

    def SetStrokeWidth(self, width):
        """Set the layer's stroke width."""
        self.stroke_width = width
        self.changed(LAYER_LEGEND_CHANGED, self)
