# Copyright (C) 2000, 2001 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the LGPL (>=v2)
# Read the file COPYING coming with MapIt! for details.

"""
Module to read and manage the marker information

The marker information is stored in a text file with line oriented
format. Empty lines and lines starting with a hash (#) are ignored. The
other lines contain either object or group definitions.

A group is defined with a line starting with the keyword "group:",
followed by four comma separated values. For example (the examples here
are in German because the example data is in German at the moment):

       group: db,bahnhof,http://bahnhof/,Bahnhfe

The first value is the name of the symbol to use. The filename of the
symbol image is built from the values of the configuration variables
marker_dir (the directory) and marker_ext (file name extension). A
good file format for the symbol images is PNG because it can contain
transparency information.

The second value is the identifier of the group. This identifier has
to be unique and should only contain characters that can be used in
URLs and HTML without quoting (even though MapIt! treats them
properly).

The third parameter is a URL to be used in the client side image map
for all markers of the group that don't define their own URL. The URL
may be empty.

The fourth value is the label of the group. It is used in the list box
where the user can select which objects to show in the map. This label
doesn't have to be unique, although several objects and groups with
the same label will be confusing for the user.

Only the first three commas are significant, so that the label may
contain commas without special quoting.

An object is defined by a line with six comma separated values. For
example:

      458862396.00,583250959.00,db,bahnhof,http://bahnhof/,Berlin-Hermsdorf

The first two values are the coordinates of the object. The two
following values have the same meaning as the symbol name and
identifier of a group. If the identifier of an object is the same as
the identifier of a previously defined group, the object is associated
with that group. Otherwise the identifier is only used for that
object. The last two values are again a URL and the label used in the
HTML-page.

If a marker contained in group does not define a URL, i.e. if the
field is empty, the URL of the group is used, if the group has a URL
associated with it.

Only the first five commas are significant, so that the label may
contain commas without special quoting.


The main contents of this module is the class MarkerInfo that reads the
marker file and represents the information contained in it.

"""

__version__ = "$Revision: 1.14 $"

import os, sys
import stat
from string import split, strip
from types import StringType

import PIL.Image

# Import the PngImagePlugin directly to work around a bug in PIL's
# plugin code that doesn't find all plugins if PIL is only usable as a
# package, i.e. if the PIL directory itself is not in the path.
import PIL.PngImagePlugin

class Marker:

    """Describe a single marker.

    Public instance variables:

        x, y - integers. The position of the marker in world coordinates.

        icon - basename of the image file to use as an icon without
               the extension

        name - unique name for the label. Used internally to identify
               the marker

        url - A url to be used in the client side image map

        label - Human readable name for the marker.
    """

    def __init__(self, x, y, name, icon, url, label):
        self.x = x
        self.y = y
        self.icon = icon
        self.name = name
        self.url = url
        self.label = label

    # provide a markers() method for both markers and marker groups so
    # they can be treated in a more uniform way without having to
    # special case them.
    def markers(self):
        return (self,)

class MarkerGroup:

    """Describe a whole group of markers"""

    def __init__(self, name, icon, url, label):
        self.icon = icon
        self.name = name
        self.url = url
        self.label = label
        self._markers = []

    def markers(self):
        return self._markers

    def add_marker(self, marker):
        """Add marker to the group.

        If the marker does not define an icon or a url, use self's icon
        or url.
        """
        if not marker.icon:
            marker.icon = self.icon
        if not marker.url:
            marker.url = self.url
        self._markers.append(marker)


class MarkerInfo:

    """Read and manage the marker descriptions and icons

    Public Methods:

        __init__(filename, image_dir, image_ext)

                Read the marker definitions from the file given by
                filename. image_dir is the directory where the marker
                icons are found, image_ext is the file extension of the
                images.

        marker_image(marker)

                Return the image associated with the Marker object marker.

        update_if_changed()

                If the marker file has changed since it was last read
                update self and return true, otherwise return false

    Public Instance Variables:

        marker - list of Marker instances. One for each marker defined
                 in the marker file. The order is the same as in the
                 marker file.

        name_to_marker - Dictionary mapping the names of the markers to
                         Marker objects.

    Private Instace Variables:

        marker_images - Dictionary to cache the icons
        image_dir - The value of the __init__ parameter image_dir
        image_ext - The value of the __init__ parameter image_ext
        
    """

    def __init__(self, filename, image_dir, image_ext):
        self.filename = filename
        self.mtime = None
        self.marker = []
        self.image_dir = image_dir
        self.image_ext = image_ext
        self.init_caches()
        self.read_marker_data(self.filename)

    def init_caches(self):
        """Initialize the caches for images and other information"""
        self.name_to_marker = {}
        self.marker_images = {}
        
    def clear_caches(self):
        """Clear the caches"""
        self.name_to_marker.clear()
        self.marker_images.clear()

    def read_marker_data(self, filename):
        """Read the marker definitions from the file given by filename"""
        self.mtime = os.stat(filename)[stat.ST_MTIME]
        groups = {}
        lines = open(filename).readlines()
        for line in lines:
            line = strip(line)
            if line and line[0] != '#':
                # it's not an empty line or a comment
                if line[:6] == 'group:':
                    # a group definition
                    icon, name, url, label = split(strip(line[6:]), ',', 3)
                    icon = strip(icon)
                    name = strip(name)
                    url = strip(url)
                    label = strip(label)
                    g = MarkerGroup(name, icon, url, label)
                    groups[name] = g
                    self.name_to_marker[name] = g
                    self.marker.append(g)
                else:
                    # it must be a single marker definition
                    items = split(line, ',', 5)
                    if len(items) == 6:
                        x, y, icon, name, url, label = items
                    else:
                        # Wrong number of items on the line. File
                        # doesn't confor to the markerdefs format.
                        if len(items) in (3, 5):
                            # Looks like the file follows the old format
                            # (pre MapIt 1.0, see tools/convertmarkerdefs)
                            message = ("Invalid markerdefs line in %s."
                                       " Perhaps an old markerdefs format.") \
                                       % filename
                            raise TypeError(message)
                        else:
                            raise TypeError("Invalid markerdefs line in %s"
                                            % filename)
                    x = float(x)
                    y = float(y)
                    icon = strip(icon)
                    name = strip(name)
                    url = strip(url)
                    label = strip(label)
                    marker = Marker(x, y, name, icon, url, label)
                    g = groups.get(name)
                    if g is not None:
                        g.add_marker(marker)
                    else:
                        self.name_to_marker[name] = marker
                        self.marker.append(marker)

    def update_if_changed(self):
        """If the marker file has changed update self and return true,
        otherwise return false"""
        mtime = os.stat(self.filename)[stat.ST_MTIME]
        if mtime > self.mtime:
            del self.marker[:]
            self.clear_caches()
            self.read_marker_data(self.filename)
            return 1
        return 0

    def marker_image(self, marker, scale = ''):
        """Return the PIL image for the marker marker.

        The marker argument can be a marker object or a string. If it's
        a string, assume that it's the icon name to use. If it's a
        marker, use marker.icon as icon name.

        If the scale argument is given, look for the icon image for the
        given scale and use the standard icon for that marker if it
        isn't found. The basename of the image file for the scale is of
        the form <icon>-<scale> where <icon> is the icon name and scale
        is the scale argument.

        The actual image loading is done through the load_marker_image
        method.
        """
        if isinstance(marker, StringType):
            icon = marker
        else:
            icon = marker.icon

        # First, try to load the scale specific marker image
        image = None
        if scale:
            scale_icon = icon + '-' + scale
            try:
                image = self.load_marker_image(scale_icon)
            except IOError:
                pass
        # If there's no scale specific image, so try the standard image
        if not image:
            image = self.load_marker_image(icon)
            # If we're loading a scale specific image, put the image
            # into the cache under the scale-specific name too.
            if scale:
                self.marker_images[scale_icon] = image
        return image

    def load_marker_image(self, basename):
        """Return the marker image with basename basename. The image
        filename is built by appending self.image_ext to basename and
        joining it with self.image_dir. Maintain a cache of loaded
        images for speed.

        If the image file can't be found raise an IOError exception. The
        exception is actualyl raised from within PIL, so it's
        conceivable that it may change in future PIL or Python versions.
        """
        image = self.marker_images.get(basename)
        if image is None:
            filename = os.path.join(self.image_dir, basename + self.image_ext)
            image = PIL.Image.open(filename)
            self.marker_images[basename] = image
        return image

    def check_marker_images(self):
        for marker in self.marker:
            for m in marker.markers():
                try:
                    image = self.marker_image(m)
                except:
                    return "Unknown Markerimage %s:\n%s" \
                           % (m.icon, sys.exc_info()[1])
        return ""

                    

                
