# Copyright (C) 2000, 2001, 2002 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the LGPL (>=v2)
# Read the file COPYING coming with MapIt! for details.

"""
Module to generate the raster map.

The entry point is the function generate_map.
"""

__version__ = "$Revision: 1.14 $"

import operator, time

import os, sys
from string import split, strip

import PIL.Image

# explicitly import the Png plugin makes loading of PNG images
# substantially faster in a normal CGI setting. The reason is that PIL
# only preloads a few plugins for common formats, but not for PNG. When
# an image is loaded that is not in one of the preloaded formats, all
# remaining plugins are loaded before the correct one is used. That
# takes a lot of time (Well, about half a second on my system, but
# that's a bit too much already in a normal CGI setting)
import PIL.PngImagePlugin

import mapconfig, tileinfo, common

# We have to deal with several coordinate systems:
#
# - World coordinates. prefix: world_
#
# - Pixel coordinates in several images:
#
#   - The result image. prefix result_
#
#   - the source images. prefix source_
#
# Note: the positive y axis of the pixel coordinate systems points down
# while the y axis of the world coordinate system points up.
#
# Other values:
#
# The tile resolution in world units / pixel. prefix res


def generate_map(parameters):
    """Return the map image described by parameters"""
    world_x = parameters.world_x
    world_y = parameters.world_y
    tileset = parameters.get_tile_set()

    result_width = parameters.width
    result_height = parameters.height
    tile_image_width, tile_image_height = tileset.tile_image_size
    resx, resy = tileset.tile_resolution
    
    # compute the world coords of edges of the result
    world_width = result_width * resx
    world_left = world_x - world_width / 2
    world_right = world_left + world_width
    world_height = result_height * resy
    world_top = world_y + world_height / 2
    world_bottom = world_top - world_height

    # Find the tiles that cover the result image
    tiles = []
    for tile in tileset.tiles:
        if tile.llx <= world_right and tile.urx >= world_left  \
           and tile.lly <= world_top and tile.ury >= world_bottom:
            tiles.append(tile)

    # Create the result image
    image = PIL.Image.new("RGB", (result_width, result_height), 0xFFFFFFFF)

    # Iterate through the tiles previously found and cut and paste the
    # appropriate region of the tile image into the output image
    load_times = []
    for tile in tiles:
        result_tile_left = int((tile.llx - world_left) / resx)
        result_tile_top = int((world_top - tile.ury) / resy)
        
        source_width = result_width
        source_height = result_height
        if result_tile_left <= 0:
            source_left = -result_tile_left
            result_left = 0
        else:
            source_left = 0
            result_left = result_tile_left
            
        if result_tile_top <= 0:
            source_top = -result_tile_top
            result_top = 0
        else:
            source_top = 0
            result_top = result_tile_top

        w = min(tile_image_width - source_left, result_width - result_left)
        h = min(tile_image_height - source_top, result_height - result_top)
        load_start = time.clock()
        filename = tileset.tile_filename(tile)
        source_image = PIL.Image.open(filename)
        # Explizitly load the image to measure performance.
        source_image.load() 
        load_times.append(time.clock() - load_start)
        source_image = source_image.crop((source_left, source_top,
                                          source_left + w, source_top + h))
        image.paste(source_image, (result_left, result_top))
        source_image = None

    # put the markers into the image. Simply iterate through all the
    # available markers and test whether they're in the result image.
    markers = parameters.markerinfo
    for name in parameters.marker.keys():
        for marker in markers.name_to_marker[name].markers():
            x = marker.x
            y = marker.y
            marker_image = markers.marker_image(marker, str(parameters.scale))
            marker_width, marker_height = marker_image.size
            # convert to pixel coordinates. x and y define the center of the
            # marker image, so we have to subtract half the size of the
            # marker image because PIL's paste method requires the
            # coordinates of the top-left corner.
            result_x = int((x - world_left) / resx - marker_width / 2)
            result_y = int((world_top - y) / resy - marker_height / 2)
            # Test whether the marker image overlaps the result image
            # and ignore the marker if it doesn't
            if result_x + marker_width < 0 or result_x > result_width \
               or result_y + marker_height < 0 or result_y > result_height:
                # The marker lies outside of the image
                continue
            if marker_image.mode == "RGBA":
                image.paste(marker_image, (result_x, result_y), marker_image)
            else:
                image.paste(marker_image, (result_x, result_y))
    # finally, place the stamp on the image
    filename = parameters.stamp
    try:
        source_image = PIL.Image.open(filename)
    except IOError:
        source_image = None
    else:
        s_width, s_height = source_image.size
        if source_image.mode == "RGBA":
            image.paste(source_image, 
                        (1, parameters.height - 1 - s_height), source_image)
        else:
            image.paste(source_image, (1, parameters.height - 1 - s_height))
        source_image = None

    # make it indexed
    if mapconfig.output_indexed:
        image = image.convert(mode = 'P', palette = PIL.Image.ADAPTIVE,
                              dither = PIL.Image.NONE)

    return image


# logging
def write_log_entry(filename):
    if filename:
        file = open(filename, "a", 0)
        file.write('.')
        file.close()
