#! /usr/bin/python
# Copyright (C) 2000 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the LGPL (>=v2)
# Read the file COPYING coming with MapIt! for details.

"""Usage: checkconfig.py

Run various tests on the mapit installation:

 1. test whether all tile images are readable and can be decoded by the
    imaging library

 2. test whether all marker images are readable and can be decoded by the
    imaging library

 3. check the syntax and presence of the mapit frontends implemented in
    python.

    If this test produces messages about files it can't compile or
    import that does not necessarily mean that you can't run mapit
    because you may not need that particular file for the method you use
    to integrate mapit with your webserver. For instance modmapit will
    be reported as problematic if mod_python is not installed but you
    don't need that file if you run mapit as a normal CGI script.

checkconfig prints quite detailed messages about the success and failure
of the tests it runs. For a better overview it prints a summary of the
tests, listing all files for which tests failed.

"""


import os, sys
import string
import traceback
import PIL.Image

if __name__ == '__main__':
    # assuming we're run directly in the mapit source tree, sys.path[0]
    # should be the directory where mapitserver.py is located, so use
    # that to add ../base to the path
    sys.path.insert(0, os.path.join(sys.path[0], '..', 'base'))


import tileinfo, markerinfo, mapconfig

def check_tiles(tilesets):
    """try to open and load all images in the MapInfo instance tilesets
    print diagnostics about it to stdout and return a tuple with two
    lists (failed_open, failed_load). The first contains the filenames
    of the tiles that couldn't be opened (permission denied, file
    doesn't exist, etc) and the second lists all files that couldn't be
    loaded (because of PIL bugs (some PNG images) or because they're
    corrupted). """
    failed_on_open = []
    failed_on_load = []
    print "checking tilesets"
    for scale in tilesets.scales:
        print "  tileset for scale", scale
        set = tilesets.get_tile_set(scale)
        for tile in set.tiles:
            filename = set.tile_filename(tile)
            print "    image", filename
            try:
                image = PIL.Image.open(filename)
                print "      open successful"
            except:
                failed_on_open.append(filename)
                print "      open failed:"
                traceback.print_exc(0)
            try:
                image.load()
                print "      load successful"
            except:
                failed_on_load.append(filename)
                print "      load failed:"
                traceback.print_exc(0)
    return failed_on_open, failed_on_load

def run_tile_check():
    """Check the tile set configuration and return a string summarizing
    the result
    """
    try:
        tilesets = tileinfo.MapInfo(mapconfig.tile_dir, mapconfig.tile_ext)
    except:
        summary = "Can't read tilesets from %s:" % mapconfig.tile_dir
        print summary
        traceback.print_exc()
        return summary

    failed_on_open, failed_on_load = check_tiles(tilesets)
    summary = []
    if failed_on_open:
        summary.append("%d tile image file(s) can't be opened:"
                       % len(failed_on_open))
        for filename in failed_on_open:
            summary.append(filename)
    else:
        summary.append("all tile image files can be opened")
    if failed_on_load:
        summary.append("%d tile image file(s) can't be loaded:"
                       % len(failed_on_load))
        for filename in failed_on_load:
            summary.append(filename)
    else:
        summary.append("all tile image files can be loaded")
    return string.join(summary, "\n")

def check_markers(markerdefs):
    failed_on_open = []
    failed_on_load = []
    cache = {}
    print "checking marker definitions"
    for marker in markerdefs.marker:
        is_group = isinstance(marker, markerinfo.MarkerGroup)
        if is_group:
            print "  checking marker group", marker.label
        for m in marker.markers():
            print "   checking marker", m.label
            if cache.has_key(m.icon):
                print "    marker image already checked"
                continue
            filename = os.path.join(markerdefs.image_dir,
                                    m.icon + markerdefs.image_ext)
            print "   marker %s -> filename %s" % (m.icon, filename) 
            try:
                image = markerdefs.marker_image(m)
                print "      open successful"
            except:
                failed_on_open.append(filename)
                print "      open failed:"
                traceback.print_exc(0)
            try:
                image.load()
                print "      load successful"
            except:
                failed_on_load.append(filename)
                print "      load failed:"
                traceback.print_exc(0)
            cache[m.icon] = 1
    return failed_on_open, failed_on_load

def run_marker_check():
    try:
        markerdefs = markerinfo.MarkerInfo(mapconfig.marker_locations,
                                           mapconfig.marker_dir,
                                           mapconfig.marker_ext)
    except:
        summary = "Can't read markerdefs from %s:" % mapconfig.marker_locations
        print summary
        traceback.print_exc()
        return summary

    failed_on_open, failed_on_load = check_markers(markerdefs)
    summary = []
    if failed_on_open:
        summary.append("%d marker image file(s) can't be opened:"
                       % len(failed_on_open))
        for filename in failed_on_open:
            summary.append(filename)
    else:
        summary.append("all marker image files can be opened")
    if failed_on_load:
        summary.append("%d marker image file(s) can't be loaded:"
                       % len(failed_on_load))
        for filename in failed_on_load:
            summary.append(filename)
    else:
        summary.append("all marker image files can be loaded")
    return string.join(summary, "\n")

def check_syntax():
    """Read and compile the mapit frontend scripts and modules"""
    files = ['genmap', 'showmap', 'genmap.fcgi', 'showmap.fcgi']
    modules = ['showmap', 'modmapit', 'genmap', 'fcgi']

    problematic_files = []
    print "checking python files"
    for file in files:
        print "  checking", file
        try:
            contents = open(file).read()
            print "    reading succesful"
        except:
            print "    reading failed:"
            traceback.print_exc(0)
            problematic_files.append(file)
            continue
        try:
            compile(contents, file, "exec")
            print "    compilation succesful"
        except:
            print "    compilation failed:"
            traceback.print_exc(0)
            problematic_files.append(file)

    for module in modules:
        print "  checking module", module
        try:
            __import__(module)
            print "    import successful"
        except:
            print "    import failed:"
            traceback.print_exc(0)
            problematic_files.append(module)
            
    summary = []
    if problematic_files:
        summary.append("%d python file(s) can't be compiled/imported:"
                       % len(problematic_files))
        for filename in problematic_files:
            summary.append(filename)
    else:
        summary.append("all python files successfully compiled/imported:")
    return string.join(summary, "\n")
    

def main():
    if len(sys.argv) > 1 and sys.argv[1] in ('-h', '--help'):
        print __doc__
        return
    tile_summary = run_tile_check()
    marker_summary = run_marker_check()
    syntax_summary = check_syntax()

    print
    print "Summary:"
    print tile_summary
    print marker_summary
    print syntax_summary

if __name__ == '__main__':
    main()

