# Copyright (C) 2000 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the LGPL (>=v2)
# Read the file COPYING coming with MapIt! for details.

"""
This module contains classes that hold information about the tiles and
scales available

The code expects the tile data to be stored in the file system as
follows. The tiles are organized in tile sets. All tiles in a tile set
have the same scale and the tiles are described by a .dat file. The
format of the .dat file is not very well defined at the moment,
unfortunately. See the parse_datfile method in the code.

Each tile set is contained in a directory named after its scale, i.e.
a tile set with a scale of 1:50000 is stored in the directory 50000.
All tile sets are contained in the same directory whose name is the
input for this module. So the directory tree might look like this:

      Tiles/
              50000/
                      base.dat
                      base1x1.bmp
                      base1x2.bmp
                      .
                      .
                      .
              100000/
                      base.dat
                      base1x1.bmp
                      base1x2.bmp
                      .
                      .
                      .
              .
              .
              .

The basename of the tile image files and the file format extension are
also input parameters


The main class defined here is MapInfo. Instances of this class provide
access to all information about the tile sets available.
"""


import os
from string import strip
import re


class TileInfo:

    """Describe the indices and extents of one tile

    Public Instance Variables:

        x, y - The indices for the file name.

        llx, lly, urx, ury - The lower left and upper right corners of
                             the the tile in world coordinates
    """

    def __init__(self, x, y, llx, lly, urx, ury):
        self.x = x
        self.y = y
        self.llx = llx
        self.lly = lly
        self.urx = urx
        self.ury = ury

    def __repr__(self):
        # mostly useful for debugging purposes
        return "Tile [%d, %d], (%d, %d) -- (%d, %d) (%d, %d)" \
               % (self.x, self.y, self.llx, self.lly, self.urx, self.ury,
                  self.urx - self.llx, self.ury - self.lly)

class TileSet:

    """Hold the information about one tile set of a given scale

    Public Instance Variables:

        tiles - list of available tiles as TileInfo objects

        llx, lly, urx, ury - The lower left and upper right corners of
                             the the tile set in world coordinates

        tile_image_size - The pixel size of a tile as a (width, height) tuple

        tile_size - The size of a tile as a (width, height) tuple in
                    world coordinates

        tile_resolution - The resolution in world units/pixel as a tuple
                          (xres, yres)

    """

    def __init__(self, directory, scale, extension):
        """Initialize self

        Parameters:

        directory - the directory containing the tile set
        scale - the scale of the tile set
        extension - the file name extension of the tile image to use

        Making the extension configurable makes it easy to experiment
        with different file formats
        """
        self.directory = directory
        self.scale = scale
        self.extension  = extension
        self.llx = self.lly = self.urx = self.ury = 0
        self.tiles = []
        self.tile_image_size = (0, 0)
        self.tile_size = (0, 0)
        self.tile_resolution = (1, 1)
        self.infofile = "info"
        self.parse_info_file()

    def find_tile(self, x, y, clip = 0):
        """Return the tile coverin teh point (x, y) in world coordinates.

        If clip is true and the point is outside of the are covered by
        the tile set choose the nearest one.
        """
        if clip:
            if x < self.llx:
                x = self.llx
            elif x > self.urx:
                x = self.urx
            if y < self.lly:
                y = self.lly
            elif y > self.ury:
                y = self.ury
        for tile in self.tiles:
            if tile.llx <= x <= tile.urx and tile.lly <= y <= tile.ury:
                return tile

    def tile_filename(self, tile):
        """Return the filename of the image file associated with tile"""
        return os.path.join(self.directory,
                            "%dx%d%s" % (tile.x, tile.y, self.extension))

    def parse_info_file(self):
        """Read the info file and initialize self"""
        file = open(os.path.join(self.directory, self.infofile))
        lines = map(strip, file.readlines())
        self.tile_image_size = (int(lines[0]), int(lines[1]))
        del lines[:2]

        xs = []
        ys = []
        while lines and lines[0] and lines[0][0] == '#':
            match = re.match(r"# *\[(\d+), *(\d+)\]", lines[0])
            if match:
                x = int(match.group(1))
                y = int(match.group(2))
                tile = TileInfo(x, y,
                                float(lines[1]), float(lines[2]),
                                float(lines[3]), float(lines[4]))
                self.tiles.append(tile)
                xs.append(tile.llx)
                ys.append(tile.lly)
                xs.append(tile.urx)
                ys.append(tile.ury)
            del lines[:5]
        if self.tiles:
            self.llx = min(xs)
            self.lly = min(ys)
            self.urx = max(xs)
            self.ury = max(ys)

            tile = self.tiles[0]
            self.tile_size = (tile.urx - tile.llx, tile.ury - tile.lly)

            try:
                width, height = self.tile_image_size
                self.tile_resolution = (self.tile_size[0] / float(width),
                                        self.tile_size[1] / float(height))
            except ZeroDivisionError:
                pass



class MapInfo:

    """Hold the information about all tile sets

    Public Instance Variables:

        scales - list of all scales for which tile set are available
    """

    def __init__(self, directory, extension):
        """Initialize self

        Parameters:

        directory - The directory containing the available tile sets
        extension - the file name extension of the tile image to use

        Making the extension configurable makes it easy to experiment
        with different file formats
        """
        self.directory = directory
        self.extension = extension
        self.scales = []
        self.tile_sets = {}
        self.determine_available_scales()

    def determine_available_scales(self):
        """Scan self.directory to determine which scales are available"""
        subdirs = os.listdir(self.directory)
        for dir in subdirs:
            try:
                scale = int(dir)
                self.scales.append(scale)
            except ValueError:
                pass
        self.scales.sort()

    def get_tile_set(self, scale):
        """Return the tile set for the scale scale"""
        set = self.tile_sets.get(scale)
        if set is None:
            set = TileSet(os.path.join(self.directory, `scale`), scale,
                          self.extension)
            self.tile_sets[scale] = set
        return set
