# Copyright (C) 2000 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the LGPL (>=v2)
# Read the file COPYING coming with MapIt! for details.

"""Template Substitution

To make it easier to embed the map and the control elements into
web-sites, this module implements a simple mechanism that scans a given
template file for substitution instructions and replaces them with the
appropriate HTML code. This way the placing and look of the control
elements such as the navigation arrows can be defined by a web-site
designer in a very flexible manner.

The substitution instructions have the form

        <mapit:<instruction> [key1="value1" ...] >

Where <instruction> is an instruction name like arrow_up that describes
what information to substitute. An instruction can have parameters in
the form of key/value pairs much like an XML element. Values are strings
and must be enclosed in single or double quotes.

The beginning and end of the instruction ('<mapit:' and '>') are
defined in mapconfig by two regular expressions.

The following instructions are available:

    arrow_up

        If there's a neighboring map image available to the north of the
        currently shown region, insert a hyperlink for the neighboring
        map with an image defined by the on parameter. Otherwise, just
        insert an image reference to the image given by the off
        parameter.

        Parameters:

            on, off - The (relative) URL of the image to use when the
                      user can/can't scroll in that direction

            alt - The ALT text of the image reference for the on image
    
    arrow_down
    arrow_left
    arrow_right
    zoom_in
    zoom_out

        Just like arrow_up, only for another direction or for zooming in
        and out.


    mapimage

        Insert the reference to the mapimage

    marker_form

        Insert the form that lets the user select markers

        Parameters:

            button - The text for the submit button
            sameregion - The text for the 'same region' check box
            title - title text

    scale_list

        Insert the list of available scales.
        
        Parameters:

            title - title text

"""

        
from types import StringType

import re

import mapconfig, common


#
# The template parser/processor
#

rx_parameter = re.compile(r"\s*(?P<key>[a-zA-Z_0-9]+)\s*"
                          "(=\s*(?P<value>\"[^\"]+\"|'[^']+'))?")

class TemplateProcessor:

    def __init__(self, filename):
        self.template = self.parse_template(filename)
        self.ref_template = '<a href="%s"><img src="%s" border="0" alt="%s">'\
                            '</a>'
        self.no_icon = '<img src="%s">'


    def parse_template(self, filename):
        """Parse the template file filename and return the compiled template

        The result is a list of stings and tuples. Strings are the parts of
        the template between the instructions, while the tuples are
        string/dictionary pairs describing an instruction. The string is
        just the name of the instruction (e.g. "arrow_up") and the
        dictionary holds the key/value pairs of the parameters
        """
        rx_start = re.compile(mapconfig.template_item_start)
        rx_end = re.compile(mapconfig.template_item_end)
        text = open(filename).read()
        template = []
        pos = 0
        while 1:
            match = rx_start.search(text, pos)
            if not match:
                template.append(text[pos:])
                break
            template.append(text[pos:match.start()])
            pos = match.end()
            element = match.group("element")
            parameters = {}
            while 1:
                match = rx_end.match(text, pos)
                if match:
                    pos = match.end()
                    break
                match = rx_parameter.match(text, pos)
                if match:
                    pos = match.end()
                    key = match.group('key')
                    value = match.group('value')
                    if type(value) == StringType:
                        value = value[1:-1]
                    parameters[key] = value
                    continue
                # syntax error
                break
            template.append((element, parameters))
        return template

    def generate_html(self, cgi, out):
        """Process the templatefile filename and write the result to out.

        This method works by iterating through the items in a
        precompiled version of the template. Strings are passed directly
        through to out.write. For an instruction (NAME, PARAMETERS),
        call self.handle_NAME(PARAMETERS, out).
        """
        for item in self.template:
            if type(item) == StringType:
                out.write(item)
            else:
                element, parameters = item
                method = getattr(self, 'handle_' + element, None)
                if method is not None:
                    method(cgi, parameters, out)


    def handle_mapimage(self, cgi, parameters, out):
        write = out.write
        resx, resy = cgi.get_tile_set().tile_resolution
        xorig = cgi.world_x - cgi.width * resx / 2
        yorig = cgi.world_y + cgi.height  * resy / 2
        grid_width = mapconfig.navigation_grid_width
        grid_height = mapconfig.navigation_grid_height
        gw2 = grid_width / 2
        gh2 = grid_height / 2
        write('<map name="objects">\n')
        if cgi.marker:
            for name in cgi.marker.keys():
                for m in cgi.markerinfo.name_to_marker[name].markers():
                    image = cgi.markerinfo.marker_image(m)
                    imgwidth, imgheight = image.size
                    x = (m.x - cgi.world_x) / resx + cgi.width / 2
                    y = (cgi.world_y - m.y) / resy + cgi.height / 2
                    url = cgi.url_params(cgi.showmap, world_x = m.x,
                                          world_y = m.y, autocenter = 0)
                    write('<area shape="rect" coords="%d,%d,%d,%d"'
                          ' href="%s" alt="%s">\n' 
                           % (x - imgwidth / 2, y - imgheight / 2,
                              x + imgwidth / 2, y + imgheight / 2,
                              url, m.label))
        for y in range(0, cgi.height, grid_height):
            for x in range(0, cgi.width, grid_width):
                url = cgi.url_params(cgi.showmap,
                                      world_x = xorig + (x + gw2) * resx,
                                      world_y = yorig - (y + gh2) * resy,
                                      autocenter = 0)
                write('<area shape="rect" coords="%d,%d,%d,%d"'
                      ' href="%s" alt="">\n' 
                      % (x, y, x + grid_width, y + grid_height, url))
        
        write('</map>\n')
        usemap = ' usemap="#objects" '
        #else:
        #    usemap = ''
        imgref = cgi.url_params(cgi.genmap, autocenter = 0)
        out.write('<img src="%s" border="0" width="%d" height="%d" %s>'
                  % (imgref, cgi.width, cgi.height, usemap))

    def handle_arrow_up(self, cgi, parameters, out):
        resx, resy = cgi.get_tile_set().tile_resolution
        yscroll = (cgi.height - mapconfig.scroll_overlap) * resy
        if cgi.world_y < cgi.maxy - resy:
            amount = min(yscroll, cgi.maxy - cgi.world_y)
            url = cgi.url_params(cgi.showmap, autocenter = 0,
                                  world_y = cgi.world_y + amount)
            out.write(self.ref_template % (url, parameters['on'],
                                           parameters['alt']))
        else:
            out.write(self.no_icon % parameters['off'])

    def handle_arrow_down(self, cgi, parameters, out):
        resx, resy = cgi.get_tile_set().tile_resolution
        yscroll = (cgi.height - mapconfig.scroll_overlap) * resy
        if cgi.world_y > cgi.miny + resy:
            amount = min(yscroll, cgi.world_y - cgi.miny)
            url = cgi.url_params(cgi.showmap, autocenter = 0,
                                  world_y = cgi.world_y - amount)
            out.write(self.ref_template % (url, parameters['on'],
                                           parameters['alt']))
        else:
            out.write(self.no_icon % parameters['off'])

    def handle_arrow_right(self, cgi, parameters, out):
        resx, resy = cgi.get_tile_set().tile_resolution
        xscroll = (cgi.width - mapconfig.scroll_overlap) * resx
        if cgi.world_x < cgi.maxx - resx:
            amount = min(xscroll, cgi.maxx - cgi.world_x)
            url = cgi.url_params(cgi.showmap, autocenter = 0,
                                  world_x = cgi.world_x + amount)
            out.write(self.ref_template % (url, parameters['on'],
                                           parameters['alt']))
        else:
            out.write(self.no_icon % parameters['off'])


    def handle_arrow_left(self, cgi, parameters, out):
        resx, resy = cgi.get_tile_set().tile_resolution
        xscroll = (cgi.width - mapconfig.scroll_overlap) * resx
        if cgi.world_x > cgi.minx + resx:
            amount = min(xscroll, cgi.world_x - cgi.minx)
            url = cgi.url_params(cgi.showmap, autocenter = 0,
                                  world_x = cgi.world_x - amount)
            out.write(self.ref_template % (url, parameters['on'],
                                           parameters['alt']))
        else:
            out.write(self.no_icon % parameters['off'])
        
    def handle_zoom_in(self, cgi, parameters, out):
        available_scales = cgi.map.scales
        scaleindex = available_scales.index(cgi.scale)
        if scaleindex > 0:
            scale = available_scales[scaleindex - 1]
            url = cgi.url_params(cgi.showmap, scale = scale, autocenter = 0)
            out.write(self.ref_template % (url, parameters['on'],
                                           parameters['alt']))
        else:
            out.write(self.no_icon % parameters['off'])

    def handle_zoom_out(self, cgi, parameters, out):
        available_scales = cgi.map.scales
        scaleindex = available_scales.index(cgi.scale)
        if scaleindex < len(available_scales) - 1:
            scale = available_scales[scaleindex + 1]
            url = cgi.url_params(cgi.showmap, scale = scale, autocenter = 0)
            out.write(self.ref_template % (url, parameters['on'],
                                           parameters['alt']))
        else:
            out.write(self.no_icon % parameters['off'])


    def handle_scale_list(self, cgi, parameters, out):
        write = out.write
        write(parameters['title'])
        write('<br>')
        for s in cgi.map.scales:
            if s != cgi.scale:
                url = cgi.url_params(cgi.showmap, scale = s, autocenter = 0)
                write('\n<a href="%s">1:%d</a><br>\n' % (url, s))
            else:
                write('\n1:%d<br>' % s)

    def handle_marker_form(self, cgi, parameters, out):
        write = out.write
        write('<form action="%s" method=get>\n' % cgi.showmap)
        write('%s<br>\n' % parameters['title'])
        write('<select name="marker" size="10" multiple>\n')
        for m in cgi.markerinfo.marker:
            selected = cgi.marker.get(m.name) and "selected" or ""
            write('<option value="%s" %s>%s</option>\n' % (m.name, selected,
                                                           m.label))
        write('</select>\n')
        write('<br>\n')
        #write('<input type="checkbox" name="sameregion" value="1">%s<br>\n'
         #     % parameters['sameregion'])
        write('<input type="checkbox" name="sameregion" value="1" %s>%s<br>\n'
              % (cgi.sameregion and "checked" or "",
                 parameters['sameregion']))
        write('<input type=submit value="%s">\n' % parameters['button'])
        write('<input type=hidden name="x" value="%d">\n' % cgi.world_x)
        write('<input type=hidden name="y" value="%d">\n' % cgi.world_y)
        write('<input type=hidden name="width" value="%d">\n' % cgi.width)
        write('<input type=hidden name="height" value="%d">\n' % cgi.height)
        write('<input type=hidden name="scale" value="%d">\n' % cgi.scale)
        write('</form>')

