# Copyright (C) 2000 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the LGPL (>=v2)
# Read the file COPYING coming with MapIt! for details.

"""
Common code for shomap and genmap.

Currently there's only the class Parameters whose instances hold the
parameters passed to the CGI scripts.

"""

import cgi, string, urllib
import mapconfig, tileinfo, markerinfo

class Parameters:

    """Hold the CGI parameters of showmap and genmap

    Public attributes:

        world_x - x coordinate of the center of the requested map in
                  world coordinates
                  
        world_y - y coordinate of the center of the requested map in
                  world coordinates

        scale - The requested scale value

        width, height - The requested size of the map image

        marker - A dictionary whose keys are the names of the objects
                 that are to be marked. The values are always 1. It's a
                 dictionary to make testing of a particular object is
                 selected easier.

        minx, miny, maxx, maxy - Minimal and maximal x and y world
                                 coordinates that are useable as world_x
                                 and world_y in the current scale

        markerinfo - A MarkerInfo isntance describing the available
                     markers (markerinfo.py)

        map - MapInfo instance.

        fields - The cgi.FieldStorage instance

    Methods:

        marker_query()
    """

    def __init__(self, fields, showmap = "showmap", genmap = "genmap",
                 map = None, marker = None):
        """Read the cgi parameters and initialize the instance variables
        """
        self.showmap = showmap
        self.genmap = genmap
        self.fields = fields
        if map is None:
            self.map = tileinfo.MapInfo(mapconfig.tile_dir, mapconfig.tile_ext)
        else:
            self.map = map
        if marker is None:
            self.markerinfo = markerinfo.MarkerInfo(mapconfig.marker_locations,
                                                    mapconfig.marker_dir,
                                                    mapconfig.marker_ext)
        else:
            self.markerinfo = marker

        try:
            self.scale = int(fields['scale'].value)
        except KeyError:
            self.scale = mapconfig.scale
        try:
            self.world_x = int(fields['x'].value)
        except KeyError:
            self.world_x = mapconfig.default_x
        try:
            self.world_y = int(fields['y'].value)
        except KeyError:
            self.world_y = mapconfig.default_y
            
        try:
            self.width = int(fields['width'].value)
        except KeyError:
            self.width = mapconfig.image_width
        if self.width > mapconfig.max_image_width:
            self.width = mapconfig.max_image_width
        if self.width < mapconfig.min_image_width:
            self.width = mapconfig.min_image_width
        try:
            self.height = int(fields['height'].value)
        except KeyError:
            self.height = mapconfig.image_height
        if self.height > mapconfig.max_image_height:
            self.height = mapconfig.max_image_height
        if self.height < mapconfig.min_image_height:
            self.height = mapconfig.min_image_height
            
        try:
            self.sameregion = int(fields['sameregion'].value)
        except KeyError:
            self.sameregion = 0
        try:
            self.autocenter = int(fields['autocenter'].value)
        except KeyError:
            self.autocenter = 1
        if self.sameregion:
            # sameregion is user input, while autocenter is only used
            # internally, so sameregion overrides autocenter
            self.autocenter = 0
        self.marker = {}
        if fields.has_key('marker'):
            # There may be one or more marker values
            marker = fields['marker']
            if type(marker) != type([]):
                marker = (marker,)
            # Insert the markers into the marker dictionary but only if
            # they are known markers defined in markerinfo
            for m in marker:
                name = m.value
                if self.markerinfo.name_to_marker.has_key(name):
                    self.marker[name] = 1

        self.compute_region_and_scale()

    def compute_region_and_scale(self):
        """Compute the optimal region and scale for the given parameters"""
        # First, if sameregion is false and at least one marker is
        # selected, find the center and extents of the smallest region
        # covering all selected markers
        if self.autocenter and self.marker:
            xs = []
            ys = []
            for name in self.marker.keys():
                for m in self.markerinfo.name_to_marker[name].markers():
                    xs.append(m.x)
                    ys.append(m.y)
            if len(xs) > 1:
                llx = min(xs)
                lly = min(ys)
                urx = max(xs)
                ury = max(ys)
                width = urx - llx
                heigth = ury - lly
                for s in self.map.scales:
                    tileset = self.map.get_tile_set(s)
                    resx, resy = tileset.tile_resolution
                    if width/resx <= self.width and heigth/resy <= self.height:
                        break
                scale = s               
                world_x = (urx + llx) / 2
                world_y = (ury + lly) / 2
            else:
                world_x = xs[0]
                world_y = ys[0]
                scale = self.scale
        else:
            tileset = self.get_tile_set()
            world_x = self.world_x
            if world_x is None:
                world_x = (tileset.llx + tileset.urx) / 2
            world_y = self.world_y
            if world_y is None:
                world_y = (tileset.lly + tileset.ury) / 2
            scale = self.scale
        # At this point we should have world_x and world_y as the center
        # of the region in world coordinates and scale as the scale to
        # use. Set self.scale accordingly because it's definite now.
        self.scale = scale

        # Now that the scale and therefore the tileset is determined,
        # choose optimal world_x and world_y
        tileset = self.get_tile_set()
        resx, resy = tileset.tile_resolution
        border_x = int(resx * self.width / 2)
        border_y = int(resy * self.height / 2)
        minx = tileset.llx + border_x
        maxx = tileset.urx - border_x
        miny = tileset.lly + border_y
        maxy = tileset.ury - border_y
        if minx > maxx:
            world_x = (tileset.llx + tileset.urx) / 2
        elif world_x < minx:
            world_x = minx
        elif world_x > maxx:
            world_x = maxx
        if miny > maxy:
            world_y = (tileset.lly + tileset.ury) / 2
        elif world_y < miny:
            world_y = miny
        elif world_y > maxy:
            world_y = maxy

        # Now world_x and world_y as well as minx, ... are known
        self.world_x = world_x
        self.world_y = world_y
        self.minx = minx
        self.maxx = maxx
        self.miny = miny
        self.maxy = maxy

    def get_tile_set(self):
        return self.map.get_tile_set(self.scale)

    def marker_query(self):
        """Return the contents of self.marker as a string usable in a CGI query
        """
        result = []
        for name in self.marker.keys():
            result.append("marker=%s" % name)
        return string.join(result, "&")

    CGI_params = [('x=%d', 'world_x'), ('y=%d', 'world_y'),
                  ('width=%d', 'width'), ('height=%d', 'height'),
                  ('scale=%d', 'scale'),
                  ('sameregion=%d', 'sameregion'),
                  ('autocenter=%d', 'autocenter')]

    def url_params(self, prefix, **override):
        args = []
        append = args.append
        for conv, name in self.CGI_params:
            if override.has_key(name):
                append(conv % override[name])
            else:
                append(conv % getattr(self, name))
            
        for name in self.marker.keys():
            append("marker=%s" % urllib.quote(name))
        return prefix + '?' + string.join(args, '&')
            
    
