# Copyright (C) 2000 Intevation GmbH <intevation@intevation.de>
# Author: Bernhard Herzog <bh@intevation.de>

"""
Module to read and manage the marker information

The markers are described by a text file with a line oriented format
where every line is either a comment or describes a marker. Comments are
lines starting with '#'. A line that describes a marker contains five
comma separated values:

    x, y, icon, name, label

x and y are integers giving the position of the marker in world
coordinates.

icon is the basename of the image file to use as an icon. It shouldn't
contain an extension. The filename of the icon file is constructed by
concatenating the name of the icon directory with the value of this item
and the extension.

name is a unique name for the label that is used internally to identify
the marker. The value is used as is in URLs, so it shouldn't contain any
characters that have to be treated in a special way. It's save to use
only ASCII alpha numerics.

label is a human readable name for the marker. It's displayed in the
list of markers in the HTML form. It doesn't have to be unique but it
probably should because it's the only information the user has to
identify the entries in the list.

Parsing rules:

    1.  empty lines and comment lines are ignored

    2. a line describing a marker is first split at the commas, the
       first two items are then converted to integers and the remaining
       three items are stripped of leading and trailing whitespace and
       then used 'as is'.

Note that 2. means that quotes aren't treated in a special way.


The main contents of this module is the class MarkerInfo that reads the
marker file represents the information contained in it.

"""

import os, sys
from string import split, strip

import PIL.Image

class Marker:

    """Describe a single marker.

    Public instance variables:

        x, y - integers. The position of the marker in world coordinates.

        icon - basename of the image file to use as an icon without
               the extension

        name - unique name for the label. Used internally to identify
               the marker

        label - Human readable name for the marker.
    """

    def __init__(self, x, y, name, icon, label):
        self.x = x
        self.y = y
        self.icon = icon
        self.name = name
        self.label = label

    def markers(self):
        return (self,)

class MarkerGroup:

    """Describe a whole group of markers"""

    def __init__(self, name, icon, label):
        self.icon = icon
        self.name = name
        self.label = label
        self._markers = []

    def markers(self):
        return self._markers

    def add_marker(self, marker):
        if not marker.icon:
            marker.icon = self.icon
        self._markers.append(marker)


class MarkerInfo:

    """Read and manage the marker descriptions and icons

    Public Methods:

        __init__(filename, image_dir, image_ext)

                Read the marker definitions from the file given by
                filename. image_dir is the directory where the marker
                icons are found, image_ext is the file extension of the
                images.

        marker_image(marker)

                Return the image associated with the Marker object marker. 

    Public Instance Variables:

        marker - list of Marker instances. One for each marker defined
                 in the marker file. The order is the same as in the
                 marker file.

        name_to_marker - Dictionary mapping the names of the markers to
                         Marker objects.

    Private Instace Variables:

        marker_images - Dictionary to cache the icons
        image_dir - The value of the __init__ parameter image_dir
        image_ext - The value of the __init__ parameter image_ext
        
    """

    def __init__(self, filename, image_dir, image_ext):
        self.marker = []
        self.name_to_marker = {}
        self.marker_images = {}
        self.image_dir = image_dir
        self.image_ext = image_ext
        self.read_marker_data(filename)

    def read_marker_data(self, filename):
        groups = {}
        lines = open(filename).readlines()
        for line in lines:
            line = strip(line)
            if line and line[0] != '#':
                items = split(line, ',')
                if len(items) == 3:
                    # a group definition
                    icon, name, label = items
                    g = MarkerGroup(name, icon, label)
                    groups[name] = g
                    self.name_to_marker[name] = g
                    self.marker.append(g)
                else:
                    x, y, icon, name, label = items
                    x = float(x)
                    y = float(y)
                    icon = strip(icon)
                    name = strip(name)
                    label = strip(label)
                    marker = Marker(x, y, name, icon, label)
                    g = groups.get(name)
                    if g is not None:
                        g.add_marker(marker)
                    else:
                        self.name_to_marker[name] = marker
                        self.marker.append(marker)

    def marker_image(self, marker):
        image = self.marker_images.get(marker.icon)
        if image is None:
            filename = os.path.join(self.image_dir, marker.icon+self.image_ext)
            image = PIL.Image.open(filename)
            self.marker_images[marker.icon] = image
        return image

    def check_marker_images(self):
        for marker in self.marker:
            for m in marker.markers():
                try:
                    image = self.marker_image(m)
                except:
                    return "Unknown Markerimage %s:\n%s" \
                           % (m.icon, sys.exc_info()[1])
        return ""

                    

                
