# Copyright (C) 2002, 2003 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Unit test for parameter.py
"""

__version__ = "$Revision: 1.10 $"
# $Source: /greaterrepository/sciparam/test/test_parameter.py,v $
# $Id: test_parameter.py,v 1.10 2003/07/21 13:54:05 tkoester Exp $

import unittest

import support
support.init()

from SciParam import SciParam
from SciParam import FloatParam, IntParam, StringParam, ChoiceParam, DistParam
from SciParam import Distribution


class ImportingTest(unittest.TestCase):

    SciParam = SciParam

    def test_import1(self):
        """import SciParam"""
        import SciParam
        self.assert_(SciParam.SciParam is self.SciParam)

    def test_import2(self):
        """import SciParam.parameter"""
        import SciParam.parameter
        self.assert_(SciParam.parameter.SciParam is self.SciParam)

    def test_import3(self):
        """from SciParam import SciParam"""
        from SciParam import SciParam
        self.assert_(SciParam is self.SciParam)

    def test_import4(self):
        """from SciParam import *"""
        from SciParam import *
        self.assert_(SciParam is self.SciParam)


class SciParamTest(unittest.TestCase):

    def setUp(self):
        """create a parameter"""
        self.pclass = SciParam
        self.known = 'something'

    def test_unknown(self):
        """check unknown and notunknown"""
        parameter = self.pclass(self.pclass.__name__)
        self.assert_(parameter.isunknown())
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())

        parameter.notunknown = 1
        self.assert_(parameter.isunknown())
        self.failIf(parameter.isvalid())
        self.failIf(parameter.isusual())

        parameter.value = self.known
        self.failIf(parameter.isunknown())
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())

        parameter.notunknown = 0
        parameter.value = 'unknown'
        self.assert_(parameter.isunknown())
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())

    def test_common(self):
        """check common parameters"""
        self.assertRaises(ValueError, self.pclass)

        parameter = self.pclass(self.pclass.__name__)
        self.assertEqual(parameter.name, self.pclass.__name__)
        self.assertEqual(parameter.description, '')
        self.assertEqual(parameter.unit, '')
        self.assertEqual(parameter.default, None)
        self.assert_(parameter.isunknown())
        self.assertEqual(parameter.comment, '')
        self.assertEqual(parameter.required, 0)
        self.assertEqual(parameter.notunknown, 0)
        self.assertEqual(parameter.disabled, 0)

        parameter = self.pclass(self.pclass.__name__, 'description', 'unit',
                                self.known, self.known, 'comment', 1, 1, 1)
        self.assertEqual(parameter.name, self.pclass.__name__)
        self.assertEqual(parameter.description, 'description')
        self.assertEqual(parameter.unit, 'unit')
        self.assertEqual(parameter.default, str(self.known))
        self.failIf(parameter.isunknown())
        self.assertEqual(parameter.comment, 'comment')
        self.assertEqual(parameter.required, 1)
        self.assertEqual(parameter.notunknown, 1)
        self.assertEqual(parameter.disabled, 1)

    def test_bad_common(self):
        """check illegal common parameters"""
        self.assertRaises(TypeError, self.pclass, self.pclass.__name__,
                          'description', 'unit', self.known, self.known,
                          'comment', 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1)

        self.assertRaises(TypeError, self.pclass, self.pclass.__name__,
                          'description', 'unit', description='desc')

        self.assertRaises(TypeError, self.pclass, frog='crunchy')

    def test_hook_valid_usual(self):
        """check hook for additional valid and usual checks"""
        parameter = self.pclass(self.pclass.__name__)
        parameter.hook_isvalid = lambda parameter, value, error: 1
        parameter.hook_isusual = lambda parameter, value, error: 1
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())

        parameter.hook_isvalid = lambda parameter, value, error: 0
        self.failIf(parameter.isvalid())
        self.assert_(parameter.isusual())

        parameter.hook_isvalid = 0
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())

        parameter.hook_isusual = lambda parameter, value, error: 0
        self.assert_(parameter.isvalid())
        self.failIf(parameter.isusual())


class FloatParamTest(SciParamTest, support.FloatTestCase):

    def setUp(self):
        """create a parameter"""
        self.pclass = FloatParam
        self.known = 1.0


class DistParamTest(SciParamTest, support.FloatTestCase):

    def setUp(self):
        """create a parameter"""
        self.pclass = DistParam
        self.known = 1.0

    def test_dist_object(self):
        """check if value is always a Distribution object"""
        for val in [None, '', 'unknown', 0, 1, 1.0, '0', '1', '1.0',
                    '1/none', '0;1/normal', '1;2/uniform']:
            parameter = DistParam(self.pclass.__name__, value=val)
            self.assert_(isinstance(parameter.value, Distribution))

class IntParamTest(SciParamTest):

    def setUp(self):
        """create a parameter"""
        self.pclass = IntParam
        self.known = 1


class StringParamTest(SciParamTest):

    def setUp(self):
        """create a parameter"""
        self.pclass = StringParam
        self.known = 'something'

    def test_maxlength(self):
        """check parameter: maxlength"""
        parameter = self.pclass(self.pclass.__name__)
        self.assertRaises(ValueError, setattr, parameter, 'maxlength', -1)
        parameter.maxlength = 10
        self.assertEqual(parameter.maxlength, 10)
        parameter.value = 'short'
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())
        parameter.value = 'really too long'
        self.failIf(parameter.isvalid())
        self.failIf(parameter.isusual())
        parameter.maxlength = None
        self.assert_(parameter.isvalid())
        self.assert_(parameter.isusual())


class ChoiceParamTest(SciParamTest):

    def setUp(self):
        """create a parameter"""
        self.pclass = ChoiceParam
        self.known = 'Yes'

    def test_long(self):
        """check parameter: long"""
        parameter = self.pclass(self.pclass.__name__)
        self.assertEqual(parameter.long, 0)
        parameter.long = 1
        self.assertEqual(parameter.long, 1)
        parameter = self.pclass(self.pclass.__name__, long=0)
        self.assertEqual(parameter.long, 0)
        parameter = self.pclass(self.pclass.__name__, long=1)
        self.assertEqual(parameter.long, 1)


if __name__ == "__main__":
    unittest.main()
