# Copyright (C) 2002, 2003 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Unit test for distribution.py
"""

__version__ = "$Revision: 1.6 $"
# $Source: /greaterrepository/sciparam/test/test_distribution.py,v $
# $Id: test_distribution.py,v 1.6 2003/07/21 13:54:05 tkoester Exp $

import unittest

import support
support.init()

import types
import math

from SciParam import Distribution


class Importing(unittest.TestCase):

    Distribution = Distribution

    def test_import1(self):
        """import SciParam"""
        import SciParam
        self.assert_(SciParam.Distribution is self.Distribution)

    def test_import2(self):
        """import SciParam.distribution"""
        import SciParam.distribution
        self.assert_(SciParam.distribution.Distribution is self.Distribution)

    def test_import3(self):
        """from SciParam import Distribution"""
        from SciParam import Distribution
        self.assert_(Distribution is self.Distribution)

    def test_import4(self):
        """from SciParam import *"""
        from SciParam import *
        self.assert_(Distribution is self.Distribution)


class DistributionTest(support.FloatTestCase):

    def test_none(self):
        """check creating an emtpy Distribution instance"""
        dist = Distribution()
        self.assertEqual(str(dist), 'None/none')
        self.assertEqual(len(dist.descriptives), 1)
        self.assertEqual((dist.descriptives[0], dist.type),
                         (None, Distribution.none))

    def test_types(self):
        """check assigning of distribution types"""
        for i in range(len(Distribution.types)):
            type = Distribution.types[i]
            desc_len = len(Distribution.descriptives_map[type])
            desc = tuple(map(lambda x: x*0.5+1, range(desc_len)))
            dist = Distribution(desc, type)
            map(self.assertFloatEqual, dist.descriptives, desc)
            self.assertEqual(dist.type, type)
            dist.type = dist.type
            self.failIfEqual(dist.descriptives[0], None)
            newtype = Distribution.types[(i+1) % len(Distribution.types)]
            dist.type = newtype
            self.assertEqual(dist.type, newtype)
            self.assertEqual(dist.descriptives_len(), len(dist.descriptives))
            for j in range(dist.descriptives_len()):
                self.assertEqual(dist.descriptives[j], None)
        self.assertEqual(Distribution(2, None).type, Distribution.none)
        self.assertEqual(Distribution((1, 2), ' UniForm ').type, 'uniform')

    def test_bad_types(self):
        """check assigning of illegal distribution types"""
        for type in ['', 0, 1, 1.0, ['none'], ('none', 'normal'), '/none']:
            self.assertRaises(ValueError, Distribution, (1,), type)

    def test_values(self):
        """check assigning of values"""
        for desc, type, result in [
            (11.5, Distribution.none, '11.5/none'),
            (('-5',), Distribution.none, '-5.0/none'),
            ('+5; 1e2', ' Uniform ', '5.0;100.0/uniform'),
            (('0.8e3 ', 2), 'normal', '800.0;2.0/normal'),
            (' 3.2 ; -0.8e-3/normal', None, '3.2;-0.0008/normal'),
        ]:
            dist = Distribution(desc, type)
            self.assertEqual(str(dist), result)
            dist = Distribution()
            dist.type = type
            dist.descriptives = desc
            self.assertEqual(str(dist), result)

    def test_confidence_interval(self):
        """check confidence interval"""
        for dist, ref_ci in [
            ('1/none', [1]),
            ('1;0/normal', [1, 1, 1]),
            ('10;1/normal', [10, 6.71, 13.29]),
            ('-10;10/normal', [-10, -42.9, 22.9]),
            ('1.648721;2.161197/lognormal', [1.648721, 0.037254, 26.842858]),
            ('1;2/uniform', [1, 2]),
        ]:
            dist = Distribution(dist)
            ci = dist.confidence_interval()
            self.assertEqual(type(ci), types.ListType)
            self.assertEqual(len(ci), len(ref_ci))
            for ci_item, ref_ci_item in zip(ci, ref_ci):
                self.assertFloatEqual(ci_item[1], ref_ci_item)

    def test_valid_usual(self):
        """check valid or usual descriptives"""
        for dist, valid, usual, loglen in [
            ('1/none', 1, 1, 0),
            ('1;-0.1/normal', 0, 0, 2),
            ('1;0/normal', 1, 0, 1),
            ('1;0.1/normal', 1, 1, 0),
            ('0;-0.1/lognormal', 0, 0, 4),
            ('0;0/lognormal', 0, 0, 3),
            ('0;0.1/lognormal', 0, 0, 2),
            ('1;-0.1/lognormal', 0, 0, 2),
            ('1;0/lognormal', 1, 0, 1),
            ('1;0.1/lognormal', 1, 1, 0),
            ('1;0/uniform', 0, 0, 2),
            ('1;1/uniform', 1, 0, 1),
            ('1;2/uniform', 1, 1, 0),
        ]:
            dist = Distribution(dist)
            errors = []
            self.assertEqual(dist.isvalid(errors), valid,
                             "%s.isvalid() is %d (should be %d)"
                             % (dist, dist.isvalid(), valid))
            self.assertEqual(dist.isusual(errors), usual,
                             "%s.isusual() is %d (should be %d)"
                             % (dist, dist.isusual(), usual))
            self.assertEqual(len(errors), loglen)


if __name__ == "__main__":
    unittest.main()
