# Copyright (C) 2002, 2003 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Notebook full of Scientific Parameter Controls for wxPython
"""

__version__ = "$Revision: 1.5 $"
# $Source: /greaterrepository/sciparam/SciParam/UI/notebook.py,v $
# $Id: notebook.py,v 1.5 2003/07/21 13:54:05 tkoester Exp $

import types

from wxPython.wx import *

if __name__ == "__main__":
    import sciparampath

from dialog import ParameterDialog, ParameterDetailsBox, ParameterGrid
from dialog import _SPACE, _SMALLSPACE

wxNB_MULTILINE_enabled = true

# wxNB_MULTILINE isn't available in old wxPython versions
if not wxNB_MULTILINE_enabled or not globals().has_key('wxNB_MULTILINE'):
    wxNB_MULTILINE = 0

class ParameterNotebookPage(wxPanel):

    def __init__(self, parent, id, parameter):
        wxPanel.__init__(self, parent, id)

        # box with details for current parameter
        if parent.parent.show_details:
            details = ParameterDetailsBox(self, -1, parent.parent)
        else:
            details = None

        # grid of ParameterCtrl
        self.pargrid = ParameterGrid(self, details, parameter)

        # put all together in a vbox
        vbox = wxBoxSizer(wxVERTICAL)
        vbox.Add(self.pargrid, 1, wxALL | wxGROW, _SPACE)
        if parent.parent.show_details:
            vbox.Add(details.sizer, 0, wxALIGN_BOTTOM | wxALL | wxGROW, _SPACE)

        self.SetAutoLayout(true)
        self.SetSizer(vbox)
        vbox.Fit(self)
        vbox.SetSizeHints(self)
        self.InitDialog()


class ParameterNotebook(wxNotebook):

    def __init__(self, parent, id, parameter):
        wxNotebook.__init__(self, parent, id, style=wxNB_MULTILINE)
        self.sizer = wxNotebookSizer(self)
        self.pages = []
        self.parent = parent

        for pagename, pagepar in parameter:
            page = ParameterNotebookPage(self, -1, pagepar)
            self.AddPage(page, pagename)
            self.pages.append(page)
        self.parent.pargrid = self.pages[0].pargrid

        EVT_NOTEBOOK_PAGE_CHANGED(self,
                                  self.GetId(),
                                  self.OnNotebookPageChanged)

        EVT_NOTEBOOK_PAGE_CHANGING(self,
                                   self.GetId(),
                                   self.OnNotebookPageChanging)

    def OnNotebookPageChanged(self, event):
        self.parent.pargrid = self.pages[event.GetSelection()].pargrid
        self.parent.pargrid.refocus()
        event.Skip()

    def OnNotebookPageChanging(self, event):
        if self.GetPage(event.GetOldSelection()).Validate():
            event.Skip()
        else:
            event.Veto()


class ParameterNotebookDialog(ParameterDialog):

    def __init__(self, parent, id, title, parameter, columns=1,
                 pos=wxDefaultPosition, size=wxDefaultSize,
                 style=wxDEFAULT_DIALOG_STYLE
                       | wxSYSTEM_MENU
                       | wxRESIZE_BORDER,
                 show_details=true, show_comment=true, comment_length=None):
        ParameterDialog.__init__(self, parent, id, title, parameter,
                                 columns=columns, pos=pos, size=size,
                                 style=style, show_details=show_details,
                                 show_comment=show_comment,
                                 comment_length=comment_length)
        # workaround for wxNB_MULTILINE bug,
        # see https://intevation.de/rt/webrt?serial_num=1441
        if wxNB_MULTILINE:
            oldsize = self.GetSize()
            size = self.GetSize()
            size.SetWidth(size.GetWidth() + 1)
            self.SetSize(size)
            self.SetSize(oldsize)

    def add_content(self):
        # EVT_BUTTON is needed because recursive validation is currently broken
        EVT_BUTTON(self, wxID_OK, self.OnButtonOK)

        # notebook with grids of ParameterCtrl
        self.notebook = ParameterNotebook(self, -1, self.parameter)
        return self.notebook.sizer

    def parse_param(self, parameter, columns):
        """return parameters split into columns and parameter list."""
        if not (parameter and type(parameter) == types.ListType):
            raise TypeError, "%s used without parameter list: %r" % \
                             (self.__class__, parameter)

        parsed = []
        parameter_list = []
        for page in parameter:
            if type(page) == types.TupleType and len(page) == 2 and \
               type(page[0]) == types.StringType:
                par, par_list = ParameterDialog.parse_param(self, page[1],
                                                            columns)
                parsed.append((page[0], par))
                parameter_list.extend(par_list)
            else:
                raise TypeError, "%s used with wrong object: %r" % \
                                 (self.__class__, page)

        return parsed, parameter_list

    def OnButtonOK(self, event):
        validate=true
        for page in self.notebook.pages:
            if validate and not page.Validate():
                validate = false
        if validate:
            for page in self.notebook.pages:
                page.TransferDataFromWindow()
            event.Skip()


class _TestApp(wxApp):

    def OnInit(self):
        from test_parameter_list import parameter

        dialog = ParameterNotebookDialog(NULL, -1,
                                         'Scientific Parameter Notebook',
                                         parameter, comment_length=100,
                                         columns=2)
        if dialog.ShowModal() == wxID_OK:
            print "OK"
        else:
            print "Cancel"
        for par in dialog.get_parameters():
            print "%s = %s (%s)" % (par.name, par.value, par)
            print " comment = %s" % par.comment
        dialog.Destroy()
        return true


def _test():
    try:
        import locale
        locale.setlocale(locale.LC_ALL, "")
    except ImportError:
        # the locale module may not be available on some systems
        pass

    app = _TestApp()
    app.MainLoop()


if __name__ == "__main__":
    _test()
