# Copyright (C) 2002, 2003 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Dialog full of Scientific Parameter Controls for wxPython
"""

__version__ = "$Revision: 1.4 $"
# $Source: /greaterrepository/sciparam/SciParam/UI/dialog.py,v $
# $Id: dialog.py,v 1.4 2003/07/21 13:54:05 tkoester Exp $

import types

from wxPython.wx import *

if __name__ == "__main__":
    import sciparampath
from SciParam import SciParam

from control import _SPACE, _SMALLSPACE
from pargrid import ParameterGrid
from detailsbox import ParameterDetailsBox

class ParameterDialog(wxDialog):

    def __init__(self, parent, id, title, parameter, columns=1,
                 pos=wxDefaultPosition, size=wxDefaultSize,
                 style=wxDEFAULT_DIALOG_STYLE
                       | wxSYSTEM_MENU
                       | wxRESIZE_BORDER,
                 show_details=true, show_comment=true, comment_length=None):
        wxDialog.__init__(self, parent, id, title,
                          pos=pos, size=size, style=style)

        (self.parameter,
         self.parameter_list) = self.parse_param(parameter, columns)
        self.show_details = show_details
        if show_details:
            self.show_comment = show_comment
            self.comment_length = comment_length
        else:
            self.show_comment = false
            self.comment_length = None

        # box with buttons
        buttonbox = wxBoxSizer(wxHORIZONTAL)
        okButton = wxButton(self, wxID_OK, ' OK ')
        okButton.SetDefault()
        cancelButton = wxButton(self, wxID_CANCEL, ' Cancel ')
        buttonbox.Add(okButton, 0, wxLEFT | wxRIGHT, _SPACE)
        buttonbox.Add(cancelButton, 0, wxLEFT | wxRIGHT, _SPACE)

        vbox = wxBoxSizer(wxVERTICAL)
        vbox.Add(self.add_content(), 1, wxALL | wxGROW, _SPACE)
        vbox.Add(buttonbox, 0, wxBOTTOM | wxALIGN_RIGHT, _SPACE)

        EVT_SET_FOCUS(self, self.OnSetFocus)
        self.SetAutoLayout(true)
        self.SetSizer(vbox)
        #vbox.Fit(self)
        vbox.SetSizeHints(self)
        self.pargrid.refocus()

    def add_content(self):
        """return a sizer with everything besides OK and Cancel."""

        # box with details for current parameter
        if self.show_details:
            details = ParameterDetailsBox(self, -1, self)
        else:
            details = None

        # grid of ParameterCtrl
        self.pargrid = ParameterGrid(self, details, self.parameter)

        # put all together
        box = wxBoxSizer(wxVERTICAL)
        box.Add(self.pargrid, 1, wxBOTTOM | wxGROW, _SPACE)
        if self.show_details:
            box.Add(details.sizer, 0, wxALIGN_BOTTOM | wxTOP | wxGROW, _SPACE)
        return box

    def parse_param(self, parameter, columns):
        """return parameters split into columns and parameter list."""
        if not (parameter and type(parameter) == types.ListType):
            raise TypeError, "%s used without parameter list: %r" % \
                             (self.__class__, parameter)
        length_list = []
        parameter_list = []

        for par in parameter:
            if isinstance(par, SciParam):
                length_list.append(1)
                parameter_list.append(par)
            elif type(par) == types.TupleType and len(par) == 2 and \
                 (par[0] is None or type(par[0]) == types.StringType) and \
                 (type(par[1]) == types.ListType and len(par[1])) and \
                 reduce(lambda ok, x: ok and isinstance(x, SciParam), par[1]):
                length_list.append(len(par[1]))
                parameter_list.extend(par[1])
            else:
                raise TypeError, "%s used with wrong object: %r" % \
                                 (self.__class__, par)

        if columns == 1:
            parsed = [parameter]
        elif columns == 2:
            left, right = 0, len(parameter)-1
            llen, rlen = 0, 0
            parsed = [[],[]]
            while left <= right:
                if llen > rlen:
                    parsed[1].insert(0, parameter[right])
                    rlen += length_list[right]
                    right -= 1
                else:
                    parsed[0].append(parameter[left])
                    llen += length_list[left]
                    left += 1

        else:
            raise ValueError, "%s used with illegal number of columns: %r" % \
                               (self.__class__, columns)

        return parsed, parameter_list

    def get_parameters(self):
        """List of all SciParam objects"""
        return self.parameter_list

    def OnSetFocus(self, event):
        self.pargrid.refocus()
        event.Skip()


class _TestApp(wxApp):

    def OnInit(self):
        from test_parameter_list import parameter as parameter_list

        for parameter in parameter_list[:3]:
            dialog = ParameterDialog(NULL, -1, 'Scientific Parameter Dialog',
                                     parameter[1], comment_length=100,
                                     columns=2)
            if dialog.ShowModal() == wxID_OK:
                print "OK"
            else:
                print "Cancel"
            for par in dialog.get_parameters():
                print "%s = %s (%s)" % (par.name, par, par.value)
                print " comment = %s" % par.comment
            dialog.Destroy()
        return true


def _test():
    try:
        import locale
        locale.setlocale(locale.LC_ALL, "")
    except ImportError:
        # the locale module may not be available on some systems
        pass

    app = _TestApp()
    app.MainLoop()


if __name__ == "__main__":
    _test()
