#!/usr/bin/env python
#
# Copyright (C) 2002 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Notebook full of Scientific Parameter Controls for wxPython
"""

__version__ = "$Revision: 1.19 $"
# $Source: /greaterrepository/sciparam/SciParam/notebook.py,v $
# $Id: notebook.py,v 1.19 2002/07/31 09:02:34 tkoester Exp $

from wxPython.wx import *

from control import *

_SPACE = 10

class ParameterNotebookPage(wxPanel):
    def __init__(self, parent, id, parameter=[], columns=1):
        wxPanel.__init__(self, parent, id)

        # box with details for current parameter
        details = ParameterDetailsBox(self, -1, parent.parent)

        # grid of ParameterCtrl
        self.pargrid = ParameterGrid(self, details, parameter, columns)

        # put all together in a vbox
        vbox = wxBoxSizer(wxVERTICAL)
        vbox.Add(self.pargrid, 0, wxALL | wxGROW, _SPACE)
        vbox.Add(details.sizer, 1, wxALL | wxGROW, _SPACE)

        self.SetAutoLayout(true)
        self.SetSizer(vbox)
        vbox.Fit(self)
        vbox.SetSizeHints(self)
        self.InitDialog()


class ParameterNotebook(wxNotebook):
    def __init__(self, parent, id, parameter, columns=1):
        wxNotebook.__init__(self, parent, id)
        self.sizer = wxNotebookSizer(self)
        self.pages = []
        self.parent = parent

        for pagename, pagepar in parameter:
            page = ParameterNotebookPage(self, -1, pagepar, columns)
            self.AddPage(page, pagename)
            self.pages.append(page)
        self.pages[0].pargrid.refocus()

        EVT_NOTEBOOK_PAGE_CHANGED(self,
                                  self.GetId(),
                                  self.OnNotebookPageChanged)

        EVT_NOTEBOOK_PAGE_CHANGING(self,
                                   self.GetId(),
                                   self.OnNotebookPageChanging)

    def OnNotebookPageChanged(self, event):
        self.pages[event.GetSelection()].pargrid.refocus()
        event.Skip()

    def OnNotebookPageChanging(self, event):
        if self.GetPage(event.GetOldSelection()).Validate():
            event.Skip()
        else:
            event.Veto()


class ParameterNotebookDialog(wxDialog):
    def __init__(self, parent, id, title, parameter, columns=1,
                 pos=wxDefaultPosition, size=wxDefaultSize,
                 style=wxDEFAULT_DIALOG_STYLE
                       | wxSYSTEM_MENU
                       | wxRESIZE_BORDER):
        wxDialog.__init__(self, parent, id, title,
                          pos=pos, size=size, style=style)

        buttonbox = wxBoxSizer(wxHORIZONTAL)
        okButton = wxButton(self, wxID_OK, ' OK ')
        cancelButton = wxButton(self, wxID_CANCEL, ' Cancel ')
        buttonbox.Add(okButton, 0, wxLEFT | wxRIGHT, _SPACE)
        buttonbox.Add(cancelButton, 0, wxLEFT | wxRIGHT, _SPACE)
        # EVT_BUTTON is needed because recursive validation is currently broken
        EVT_BUTTON(self, wxID_OK, self.OnButtonOK)

        # notebook with grids of ParameterCtrl
        self.notebook = ParameterNotebook(self, -1, parameter, columns)

        # put all together in a vbox
        vbox = wxBoxSizer(wxVERTICAL)
        vbox.Add(self.notebook.sizer, 1, wxALL | wxGROW, _SPACE)
        vbox.Add(buttonbox, 0, wxBOTTOM | wxALIGN_RIGHT, _SPACE)

        EVT_SET_FOCUS(self, self.OnSetFocus)
        self.SetAutoLayout(true)
        self.SetSizer(vbox)
        #vbox.Fit(self)
        vbox.SetSizeHints(self)

    def OnSetFocus(self, event):
        page = self.notebook.GetSelection()
        if page >= 0:
            self.notebook.pages[page].pargrid.refocus()
        event.Skip()

    def OnButtonOK(self, event):
        validate=true
        for page in self.notebook.pages:
            if validate and not page.Validate():
                validate = false
        if validate:
            for page in self.notebook.pages:
                page.TransferDataFromWindow()
            event.Skip()


class _TestApp(wxApp):
    def OnInit(self):
        parameter1 = [
            StringParam('Name', 'a unique identifier', required=1,
                        value='Silicon',
                        comment='This is an example for a required value.\n'
                                'The user can change this comment to note the '
                                'source of this information.'),
            StringParam('Symbol', 'chemical symbol', required=1, value='Si'),
            IntParam('Atomic No',
                     'Number in the periodic table of the elements',
                     value=14, wrange='[1;260]', erange='[1;oo['),
            FloatParam('Atomic Mass', None, value=28.0855, erange='[0;oo['),
            FloatParam('Melting Point', 'under normal conditions', 'C',
                       value=1414,
                       wrange='[-273.15;4e3]', erange='[-273.15;oo['),
            FloatParam('Boiling Point', 'under normal conditions', 'C',
                       value=3265, wrange='[-270;6e3]', erange='[-273.15;oo['),
            ChoiceParam('Crystal', 'crystal structure',
                        choices=[None, 'simple cubic', 'face centered cubic',
                                 'body centered cubic', 'diamond cubic',
                                 'tetragonal', 'orthorombic', 'monoclinc'],
                        value='face centered cubic', long=1),
            DistParam('Temperature', 'temperature of samples', 'C',
                      default=20, value=17.5, erange='[-273.15;oo[',
                      dist=(3.7, 'normal')),
            DistParam('Volume', 'Volume of used samples', 'cm',
                      default='2;20/uniform', erange='[0;oo['),
            ChoiceParam('Verified', 'Have these values been checked?',
                        choices=ChoiceParam.yes_no, value=0),
        ]
        parameter2 = [
            StringParam('Name', 'a unique identifier'),
            IntParam('Things', 'Number of things', 'pcs',
                     erange='[0;oo['),
            FloatParam('Price', 'per thing', '', value=0.49, default=1.20,
                       wrange='[0.50;1e5[', erange='[0;oo[',
                       comment='This is too cheap'),
            ChoiceParam('Size', 'Select one of 3m, 5m, 7m', 'm',
                        default=5, choices=[3, 5, 7]),
            ChoiceParam('In Stock', 'Are there enough things?'),
        ]
        parameter = [("Element", parameter1),
                     ("Things", parameter2)]

        dialog = ParameterNotebookDialog(NULL, -1,
                                         'Scientific Parameter Notebook',
                                         parameter, columns=2)
        if dialog.ShowModal() == wxID_OK:
            print "OK"
        else:
            print "Cancel"
        for page in parameter:
            for par in page[1]:
                print par.name, "=", par.value
                print " comment =", par.comment
                if isinstance(par, DistParam):
                    print " dist =", par.dist
        dialog.Destroy()
        return true


def _test():
    try:
        import locale
        locale.setlocale(locale.LC_ALL, "")
    except ImportError:
        # the locale module may not be available on some systems
        pass

    app = _TestApp()
    app.MainLoop()


if __name__ == "__main__":
    _test()
