#!/usr/bin/env python
#
# Copyright (C) 2002 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Range class for Scientific Parameter
"""

__version__ = "$Revision: 1.3 $"
# $Source: /greaterrepository/sciparam/SciParam/range.py,v $
# $Id: range.py,v 1.3 2002/07/30 15:02:44 tkoester Exp $

import re

_inf = float('1e1000')   # FIXME: hack for infinite

class Range:
    number_re = '(?P<%s>-?(\d+\.?\d*([eE]-?\d+)?|oo))'
    brace_re = '(?P<%s>[][])'
    range_re = '^' + brace_re % 'left' + number_re % 'begin' + \
               ';' + number_re % 'end' + brace_re % 'right' + '$'
    parse_range = re.compile(range_re)

    def __init__(self, range=None):
        self.SetRange(range)

    def SetRange(self, range):
        if range is None:
            self.SetRange(']-oo;oo[')
        elif isinstance(range, Range):
            self.SetRange(str(range))
        else:
            self._range = range
            match = self.parse_range.match(self._range)
            if match:
                self._left = match.group('left')
                self._begin = self.float(match.group('begin'))
                self._end = self.float(match.group('end'))
                self._right = match.group('right')
            else:
                raise ValueError("can't parse range: %s" % range)

    def GetRange(self):
        """return internal representation of range
        
        4-tuple ('[' or ']', begin(float), end(float), '[' or ']')"""
        return (self._left, self._begin, self._end, self._right)

    def float(self, string):
        if string == 'oo':
            return _inf
        elif string == '-oo':
            return -_inf
        else:
            return float(string)

    def __contains__(self, value):
        if self._left == ']':
            contains = value > self._begin
        else:
            contains = value >= self._begin
        if self._right == '[':
            contains = contains and (value < self._end)
        else:
            contains = contains and (value <= self._end)
        return contains

    def __eq__(self, other):
        return (self.GetRange() == other.GetRange())

    def __ne__(self, other):
        return not self.__eq__(other)

    def __str__(self):
        return self._range


def _test():
    range1 = Range(']0;99]')
    print 'range1 =', range1, range1.GetRange()
    for i in [-0.1, 0, 0.1, 9.9, 99, 99.9]:
        print '%4.1f in range1 =' % i, i in range1
    range2 = Range(']-oo;10[')
    print 'range2 =', range2, range2.GetRange()
    for i in [-0.1, 0, 0.1, 9.9, 10, 10.1]:
        print '%4.1f not in range2 =' % i, i not in range2
    range3 = Range(']1e-1;1E2]')
    print 'range3 =', range3, range3.GetRange()
    for i in [0, 0.1, 0.11, 10, 100, 101]:
        print '%4.1f not in range3 =' % i, i not in range3
    print 'range3 != range2 =', range3 != range2
    print 'range3 != Range("]1e-1;1E2]") =', range3 != Range("]1e-1;1E2]")


if __name__ == "__main__":
    _test()
