#!/usr/bin/env python
#
# Copyright (C) 2002 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Scientific Parameter
"""

__version__ = "$Revision: 1.26 $"
# $Source: /greaterrepository/sciparam/SciParam/parameter.py,v $
# $Id: parameter.py,v 1.26 2002/07/29 14:24:06 tkoester Exp $

import types

from range import Range
from distribution import Distribution

class SciParam:
    _attr = ['name', 'description', 'unit', 'default', 'value', 'comment',
             'required']
    unknown = ['unknown', '', 'None', None]
    def __init__(self, name, description=None, unit=None,
                 default=None, value=None, comment=None,
                 required=0):
        # FIXME: better use **kwargs with list of attributes
        self.name = name
        self.description = description
        self.unit = unit

        self.default = default
        self.value = value
        self.comment = comment
        self.required = required

    def isvalid(self, *args):
        """are values valid (or unknown) for this parameter?"""
        if not args:
            value = self.value
        elif len(args) == 1:
            value = args[0]
        else:
            raise TypeError, "isvalid() takes no or one argument"
        return self.is1unknown(value) or self.is1valid(value)

    def is1valid(self, value):
        """is exactly one value valid for this parameter?"""
        return 1

    def isusual(self, *args):
        """are values usual (or unknown) for this parameter?"""
        if not args:
            value = self.value
        elif len(args) == 1:
            value = args[0]
        else:
            raise TypeError, "isusual() takes no or one argument"
        return self.is1unknown(value) or self.is1usual(value)

    def is1usual(self, value):
        """is exactly one value usual for this parameter?"""
        return 1

    def isunknown(self, *args):
        """is value unknown?"""
        if not args:
            value = self.value
        elif len(args) == 1:
            value = args[0]
        else:
            raise TypeError, "isunknown() takes no or one argument"
        return self.is1unknown(value)

    def is1unknown(self, value):
        """is exactly one value unknown?"""
        return value in self.unknown

    def range(self):
        """return a string telling what kind of value is expected"""
        return ""

    def convert(self, value):
        """convert string or number to internal representation

        internal representation is: value or None
        returns internal representation or raises ValueError
        """
        if self.is1unknown(value):
            return None
        else:
            return value

    def string(self, value):
        """convert internal representation to string"""
        if self.is1unknown(value):
            return self.unknown[0]
        else:
            return str(value)

    def normalize(self, value, default=None):
        """convert string to normalized string

        returns either a string that is valid for convert()
        or default
        """
        try:
            return self.string(self.convert(value))
        except ValueError, why:
            return default

    def __str__(self):
        return self.string(self.value)

    def __setattr__(self, name, value):
        if name == 'value':
            self.__dict__[name] = self.convert(value)
        elif name in ['description', 'unit', 'comment'] and value == None:
            self.__dict__[name] = ''
        elif name in self._attr:
            self.__dict__[name] = value
        else:
            raise AttributeError, "%s has no attribute '%s'" % \
                                  (self.__class__, name)


class FloatParam(SciParam):
    """float values with warning and error range"""
    def __init__(self, name, description=None, unit=None,
                 default=None, value=None, comment=None,
                 required=0,
                 wrange=None, erange=None):
        SciParam.__init__(self, name, description, unit,
                          default, value, comment,
                          required)
        self.wrange = wrange
        self.erange = erange

    def is1valid(self, value):
        """is exactly one value valid for this parameter?"""
        return self.is1inrange(value, self.erange)

    def is1usual(self, value):
        """is exactly one value usual for this parameter?"""
        return self.is1inrange(value, self.wrange)

    def is1inrange(self, value, range):
        """is exactly one value (can be string) in range?"""
        if type(value) == types.StringType:
            value = self.convert(value)
        return value in range

    def range(self):
        """return a string telling what kind of value is expected"""
        any = Range()
        if self.wrange == any:
            if self.erange == any:
                return "Float value"
            else:
                return "Float: valid is %s" % self.erange
        else:
            if self.erange == any:
                return "Float: usual is %s" % self.wrange
            else:
                return "Float: usual is %s  valid is %s" % \
                       (self.wrange, self.erange)

    def convert(self, value):
        """convert string or number to internal representation

        internal representation is: float or None
        returns internal representation or raises ValueError
        """
        if self.is1unknown(value):
            return None
        else:
            return float(value)

    def __setattr__(self, name, value):
        if name in ['wrange', 'erange']:
            if isinstance(value, Range):
                self.__dict__[name] = value
            else:
                self.__dict__[name] = Range(value)
        else:
            SciParam.__setattr__(self, name, value)


class DistParam(FloatParam):
    """float values with warning/error range and optional distribution"""
    def __init__(self, name, description=None, unit=None,
                 default=None, value=None, comment=None,
                 required=0,
                 wrange=None, erange=None,
                 dist=None):
        FloatParam.__init__(self, name, description, unit,
                            default, value, comment,
                            required,
                            wrange=wrange, erange=erange)
        self.dist=dist

    def is1valid(self, value):
        """is exactly one value valid for this parameter?"""
        return self.is1inrange(value, self.erange, checkdist=1)

    def is1usual(self, value):
        """is exactly one value usual for this parameter?"""
        return self.is1inrange(value, self.wrange, checkdist=2)

    def is1inrange(self, value, range, checkdist=0):
        """is exactly one value (can be string) in range?"""
        if type(value) == types.StringType:
            value = self.convert(value)
        elif type(value) != types.TupleType or len(value) != 2:
            raise ValueError, "is1inrange() doesn't accept this value"
        value, dist = value
        return (value in range and
                self.is1distinrange(value, dist, range, checkdist))

    def is1distinrange(self, value, dist, range, checkdist=0):
        """is exactly one value's distribution in range?

        checkdist=0: don't check distribution
        checkdist=1: check for valid range
        checkdist=2: check for usual range
        """
        if checkdist:
            valtext, dvaltext = dist.parameter[dist.type]
            if dvaltext == 'std. deviation':
                return ((checkdist == 1 and dist.value >= 0) or
                        (checkdist == 2 and dist.value > 0))
            elif dvaltext == 'max. value':
                return (dist.value in range and
                        ((checkdist == 1 and dist.value >= value) or
                         (checkdist == 2 and dist.value > value)))
        return 1

    def convert(self, value):
        """convert string or number to internal representation

        internal representation is: (float, dist) or None
        returns internal representation or raises ValueError
        """
        if self.is1unknown(value):
            return None

        if type(value) == types.StringType:
            parts = value.split(';',1)
        else:
            parts = value, None

        floatval = FloatParam.convert(self, parts[0])

        if floatval is None:
            return None

        if len(parts) == 2:
            dist = Distribution(parts[1])
        else:
            dist = Distribution()
        return (floatval, dist)

    def string(self, value):
        """convert internal representation to string"""
        if self.is1unknown(value) or self.is1unknown(value[0]):
            return self.unknown[0]

        floatval, dist = value
        if not dist or dist.value is None or dist.type == dist.none:
            return FloatParam.string(self, floatval)
        else:
            return "%s;%s" % (FloatParam.string(self, floatval), dist)

    def __str__(self):
        return self.string((self.value, self.dist))

    def __setattr__(self, name, value):
        if name == 'value':
            value = self.convert(value)
            if type(value) == types.TupleType:
                self.__dict__[name], self.dist = value
            else:
                self.__dict__[name] = value
        elif name == 'dist':
            if isinstance(value, Distribution):
                self.__dict__[name] = value
            else:
                self.__dict__[name] = Distribution(value)
        else:
            FloatParam.__setattr__(self, name, value)


class IntParam(FloatParam):
    """long integer values with warning and error range"""
    def range(self):
        """return a string telling what kind of value is expected"""
        return FloatParam.range(self).replace('Float', 'Integer')

    def convert(self, value):
        """convert string or number to internal representation

        internal representation is: long int or None
        returns internal representation or raises ValueError
        """
        if self.is1unknown(value):
            return None
        else:
            return int(value)


class StringParam(SciParam):
    """string values"""
    def convert(self, value):
        """convert string or number to internal representation

        internal representation is: string or None
        returns internal representation or raises ValueError
        """
        if self.is1unknown(value):
            return None
        else:
            return str(value)


class ChoiceParam(SciParam):
    """values from a list of choices
    
    long=1: state that this will be a long list of choices"""
    unknown_yes_no = [(SciParam.unknown[0], None), ('Yes', 1), ('No', 0)]
    yes_no = unknown_yes_no[1:]
    def __init__(self, name, description=None, unit=None,
                 default=None, value=None, comment=None,
                 required=0,
                 choices=unknown_yes_no, long=0):
        self.choices = choices
        self.long = long
        if value is None:
            value = self.choices[0]
        SciParam.__init__(self, name, description, unit,
                          default, value, comment,
                          required)

    def is1valid(self, value):
        """is exactly one value valid for this parameter?"""
        try:
            self.convert(value)
            return 1
        except ValueError, why:
            return 0

    def convert(self, value):
        """convert string or number to internal representation

        internal representation is: (choice string, choice value)
        returns internal representation or raises ValueError
        """
        if self.choicedict.has_key(value):
            return (value, self.choicedict[value])
        elif self.valuedict.has_key(value):
            return (self.valuedict[value], value)
        else:
            raise ValueError, \
                  "%s instance has no choice '%s'" % (self.__class__, value)

    def string(self, value):
        """convert internal representation to string"""
        return SciParam.string(self, value[0])

    def __str__(self):
        return self.string((self.currentchoice, self.value))

    def __setattr__(self, name, value):
        if name == 'value':
            self.__dict__['currentchoice'], \
            self.__dict__[name] = self.convert(value)
        elif name == 'choices':
            self.__dict__[name] = []
            self.__dict__['choicedict'] = {}
            self.__dict__['valuedict'] = {}
            for choice in value:
                if self.is1unknown(choice):
                    choice = self.unknown_yes_no[0]
                elif type(choice) == types.TupleType:
                    choice = str(choice[0]), choice[1]
                else:
                    choice = str(choice), choice
                self.__dict__['choices'].append(choice[0])
                self.__dict__['choicedict'][choice[0]] = choice[1]
                self.__dict__['valuedict'][choice[1]] = choice[0]
        elif name == 'long':
            self.__dict__[name] = value
        else:
            SciParam.__setattr__(self, name, value)


def _test_FloatParam():
    print "--- tests for FloatParam ---"
    range1 = Range(']0;99]')
    param1 = FloatParam('Temperature', 'This is parameter 1.', 'C',
                       default=20, wrange=range1, erange='[-273.5;oo[')

    print 'param1:', param1
    print param1.range()
    param1.value = 20
    print 'param1:', param1
    for i in [param1.value, -0.1, 0, 0.1]:
        print '%4.1f in wrange =' % i, param1.isusual(i)
    for i in [param1.value, -300, 0, 0.1]:
        print '%4.1f in erange =' % i, param1.isvalid(i)

    print "range1.SetRange('[0;99]')"
    range1.SetRange('[0;99]')
    print 'range1 (param1.wrange) =', range1
    for i in [param1.value, -0.1, 0, 0.1]:
        print '%4.1f in wrange =' % i, param1.isusual(i)

    # currently no exceptions will be thrown for values outside erange
    try:
        param1.value = -300
    except StandardError, why:
        print 'Exception:', why


def _test_DistParam():
    print "--- tests for DistParam ---"
    param1 = DistParam('Temperature', 'This is parameter 1.', 'C',
                        erange='[-273.5;oo[', required=1)
    print 'param1:', param1
    print 'param1.dist:', param1.dist
    print 'param1.dist.value = 10, \'normal\''
    param1.dist.value = 10, 'normal'
    print 'param1:', param1
    print 'param1.value = 100'
    param1.value = 100
    print 'param1:', param1
    print 'param1.value:', param1.value
    print 'param1.dist:', param1.dist
    print 'param1.dist.value:', param1.dist.value
    print 'param1.dist.type:', param1.dist.type
    print 'param1.required:', param1.required


def _test_IntParam():
    print "--- tests for IntParam ---"
    param = IntParam('piesec', 'Number of something.')

    print 'param:', param

    print 'param.value = 20.5'
    param.value = 20.5
    print 'param:', param

    print 'param.value = 14'
    param.value = 14
    print 'param:', param


def _test_StringParam():
    print "--- tests for StringParam ---"
    param = StringParam('Name', 'unique name.')
    print 'param:', param
    param.value = 'a string parameter'
    print 'param:', param


def _test_ChoiceParam():
    print "--- tests for ChoiceParam ---"

    choice1 = ChoiceParam('Simple choice', 'This is choice 1.')
    print 'choice1:', choice1
    choice1.value = 'No'
    print 'choice1 (set to "No"): %s, %s' % (choice1, choice1.value)
    choice1.value = 1
    print 'choice1 (set to 1): %s, %s' % (choice1, choice1.value)

    choice2 = ChoiceParam('Another choice', 'This is choice 2.',
                          choices=['foo', 'bar', 'bletch'])
    print 'choice2:', choice2
    choice2.value = 'bar'
    print 'choice2 (set to "bar"): %s, %s' % (choice2, choice2.value)

    choice3 = ChoiceParam('Number choice', 'This is choice 3.',
                          choices=[1, 2, 3])
    print 'choice3:', choice3
    choice3.value = 2
    print 'choice3 (set to 2): %s, %s' % (choice3, choice3.value)
    choice3.value = '3'
    print 'choice3 (set to "3"): %s, %s' % (choice3, choice3.value)



def _test():
    _test_FloatParam()
    _test_DistParam()
    _test_IntParam()
    _test_StringParam()
    _test_ChoiceParam()


if __name__ == "__main__":
    _test()
