#!/usr/bin/env python
#
# Copyright (C) 2002 by Intevation GmbH
# Authors:
# Thomas Koester <tkoester@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with the software for details.

"""
Scientific Parameter Control for wxPython
"""

__version__ = "$Revision: 1.48 $"
# $Source: /greaterrepository/sciparam/SciParam/control.py,v $
# $Id: control.py,v 1.48 2002/07/30 10:35:58 tkoester Exp $

import inspect
import os.path

from wxPython.wx import *

from parameter import *

_SMALLSPACE = 3
_SPACE = 10

class ParameterDistributionDialog(wxDialog):
    """input of distribution parameters like mean value and std. deviation"""
    def __init__(self, detailsbox, dist_type):
        parameter = detailsbox.parctrl.parameter
        wxDialog.__init__(self, detailsbox.dialog, -1,
                          "%s (%s)" % (parameter.name, dist_type))

        # behave like detailsbox class
        self.parctrl = None

        # create parameters for distribution
        value = detailsbox.parctrl.GetValue()
        try:
            floatval, dist = parameter.convert(detailsbox.parctrl.GetValue())
        except TypeError, why:
            floatval, dist = None, Distribution()
        except ValueError, why:
            floatval, dist = None, Distribution()

        valtext, dvaltext = parameter.dist.parameter[dist_type]
        self.val = FloatParam(valtext, None, parameter.unit, value=floatval,
                              wrange=parameter.wrange, erange=parameter.erange)
        if dvaltext == 'std. deviation':
            self.dval = FloatParam(dvaltext, None, value=dist.value,
                                   wrange=']0;oo[', erange='[0;oo]')
        elif dvaltext == 'max. value':
            self.dval = FloatParam(dvaltext, None, parameter.unit,
                                   value=dist.value,
                                   wrange=parameter.wrange,
                                   erange=parameter.erange)
        else:
            self.dval = FloatParam(dvaltext, None, value=dist.value)

        pargrid = ParameterGrid(self, self, [self.val, self.dval], columns=1)

        if dvaltext == 'max. value':
            validator = ParameterValidator(pargrid.controls[1],
                                           minctrl=pargrid.controls[0])
            pargrid.controls[1].SetValidator(validator)

        # box with buttons
        buttonbox = wxBoxSizer(wxHORIZONTAL)
        button = wxButton(self, wxID_OK, ' OK ')
        buttonbox.Add(button, 0, wxLEFT | wxRIGHT, _SPACE)
        button = wxButton(self, wxID_CANCEL, ' Cancel ')
        buttonbox.Add(button, 0, wxLEFT | wxRIGHT, _SPACE)

        # put all together in a vbox
        vbox = wxBoxSizer(wxVERTICAL)
        vbox.Add(pargrid, 0, wxALL | wxGROW, _SPACE)
        vbox.Add(buttonbox, 0, wxBOTTOM | wxALIGN_RIGHT, _SPACE)

        self.SetAutoLayout(true)
        self.SetSizer(vbox)
        vbox.Fit(self)
        vbox.SetSizeHints(self)

    def connect(self, parctrl):
        self.parctrl = parctrl


class ParameterDetailsBox(wxStaticBox):
    def __init__(self, parent, id, dialog):
        wxStaticBox.__init__(self, parent, id, 'Details')
        self.parent = parent
        self.dialog = dialog
        self.sizer = wxStaticBoxSizer(self, wxVERTICAL)

        self.parctrl = None
        self.name_desc = wxStaticText(parent, -1, 'Parameter:')
        font = wxFont(10, wxSWISS, wxNORMAL, wxBOLD, false)
        self.name_desc.SetFont(font)
        self.sizer.Add(self.name_desc, 0, wxBOTTOM, _SMALLSPACE)
        self.range = wxStaticText(parent, -1, 'Range:')
        self.sizer.Add(self.range, 0, wxBOTTOM, _SMALLSPACE)

        hbox = wxBoxSizer(wxHORIZONTAL)
        ## Distribution
        hbox.Add(wxStaticText(parent, -1, 'Distribution:'), 0,
                              wxALIGN_CENTER_VERTICAL)
        self.distChoice = wxChoice(parent, -1, choices=['not applicable']
                                                       + Distribution.types)
        self.distChoice.Enable(false)
        self.distChoice.SetSelection(0)
        hbox.Add(self.distChoice, 0, wxLEFT | wxRIGHT |
                                     wxALIGN_CENTER_VERTICAL, _SMALLSPACE)
        ## Default
        self.default = wxStaticText(parent, -1, 'Default:',
                                    style=wxALIGN_RIGHT | wxST_NO_AUTORESIZE)
        hbox.Add(self.default, 1, wxLEFT | wxRIGHT |
                                  wxALIGN_CENTER_VERTICAL, _SPACE)
        self.defaultButton = wxButton(parent, -1, 'Apply Default')
        self.defaultButton.Enable(false)
        hbox.Add(self.defaultButton, 0, wxLEFT | wxRIGHT |
                                        wxALIGN_CENTER_VERTICAL, _SMALLSPACE)
        # end of hbox
        self.sizer.Add(hbox, 0, wxBOTTOM | wxGROW, _SMALLSPACE)


        comment = wxStaticText(parent, -1, 'Comment on the value you entered:')
        self.sizer.Add(comment, 0, wxTOP, _SPACE)
        self.comment = wxTextCtrl(parent, -1,
                                  size=wxSize(-1, 75),
                                  style=wxTE_MULTILINE)
        self.sizer.Add(self.comment, 1, wxTOP | wxGROW, _SMALLSPACE)
        EVT_KILL_FOCUS(self.comment, self.OnKillFocus)
        EVT_BUTTON(parent, self.defaultButton.GetId(), self.OnDefaultButton)
        EVT_CHOICE(self.distChoice, self.distChoice.GetId(), self.OnDistChoice)

    def connect(self, parctrl):
        if parctrl is self.parctrl:
            return
        self.parctrl = parctrl
        par = parctrl.parameter
        if par.description:
            label = "%s: %s" % (par.name, par.description)
        else:
            label = par.name
        self.name_desc.SetLabel(label)
        size = self.name_desc.GetBestSize()
        self.sizer.SetItemMinSize(self.name_desc, size.width, size.height)

        self.range.SetLabel(par.range())
        size = self.range.GetBestSize()
        self.sizer.SetItemMinSize(self.range, size.width, size.height)

        self.connect_distribution(parctrl)

        ## Default
        if par.default is not None:
            label = "Default: %s" % par.default
            self.defaultButton.Enable(true)
        else:
            label = "Default: not available"
            self.defaultButton.Enable(false)
        self.default.SetLabel(label)
        size = self.default.GetBestSize()
        self.sizer.SetItemMinSize(self.default, size.width, size.height)

        self.comment.SetValue(parctrl._comment)

        #XXX: adjust Layout for long names and descriptions (ugly)
        size = self.parent.GetSize()
        bestsize = self.parent.GetBestSize()
        if size.width+_SPACE < bestsize.width or \
           size.height+_SPACE < bestsize.height:
            self.dialog.GetSizer().Fit(self.dialog)
            self.dialog.GetSizer().SetSizeHints(self.dialog)
        wxPostEvent(self.dialog, wxSizeEvent(bestsize))

    def connect_distribution(self, parctrl):
        if parctrl:
            if isinstance(parctrl.parameter, DistParam):
                self.distChoice.Clear()
                for choice in parctrl.parameter.dist.types:
                    self.distChoice.Append(choice)
                self.distChoice.SetStringSelection(parctrl.dist_type)
                self.distChoice.Enable(true)
            else:
                self.distChoice.Enable(false)
                self.distChoice.Clear()
                self.distChoice.Append('not applicable')
                self.distChoice.SetSelection(0)
            #XXX doesn't look good with wxPython 2.2.9 on Linux
            #self.sizer.SetItemMinSize(self.distChoice,
            #                          self.distChoice.GetBestSize().width,
            #                          self.distChoice.GetSize().height)

    def OnKillFocus(self, event):
        if self.parctrl:
            self.parctrl._comment = self.comment.GetValue()
        event.Skip()
        return true

    def OnDefaultButton(self, event):
        self.parctrl.SetValue(str(self.parctrl.parameter.default))
        self.parctrl.refocus = true
        event.Skip()

    def OnDistChoice(self, event):
        """call ParameterDistributionDialog for new distribution parameters."""
        if self.parctrl:
            param = self.parctrl.parameter
            new_type = self.distChoice.GetStringSelection()
            if new_type == param.dist.none:
                self.parctrl.SetValue(self.parctrl.GetValue().split(';')[0])
            elif (new_type == self.parctrl.dist_type or
                  # meaning of second value has changed?
                  param.dist.parameter[new_type] !=
                  param.dist.parameter[self.parctrl.dist_type]):
                dialog = ParameterDistributionDialog(self, new_type)
                if dialog.ShowModal() == wxID_OK:
                    self.parctrl.SetValue("%s;%s/%s" %
                        (dialog.val.value, dialog.dval.value, new_type))
                else:
                    new_type = self.parctrl.dist_type  # revert to old type
                    self.distChoice.SetStringSelection(self.parctrl.dist_type)
                dialog.Destroy()

            self.parctrl.dist_type = new_type
        event.Skip()
        return true


class ParameterValidator(wxPyValidator):
    def __init__(self, ctrl, minctrl=None):
        wxPyValidator.__init__(self)
        self.ctrl = ctrl
        # minctrl can only be used with int and float parameter controls
        self.minctrl = minctrl
        self.dialogexists = false
        EVT_CHAR(self, self.OnChar)

    def Clone(self):
        return self.__class__(self.ctrl, self.minctrl)

    def Validate(self, parent):
        parameter = self.ctrl.parameter
        val = self.ctrl.GetValue()
        if isinstance(parameter, DistParam):
            val = parameter.normalize(val+'/'+self.ctrl.dist_type, val)
        try:
            val = parameter.convert(val)
        except ValueError, why:
            if not self.dialogexists:
                self.dialogexists = true
                dialog = wxMessageDialog(parent,
                          "%s contains illegal characters."
                          % parameter.name,
                          "Value contains illegal characters",
                          wxOK | wxICON_ERROR)
                answer = dialog.ShowModal()
                dialog.Destroy()
                self.dialogexists = false
                self.ctrl.refocus = true
            return false
        if not parameter.isvalid(val):
            if not self.dialogexists:
                self.dialogexists = true
                dialog = wxMessageDialog(parent,
                          "%s is outside valid range."
                          % parameter.name,
                          "Value outside valid range",
                          wxOK | wxICON_ERROR)
                answer = dialog.ShowModal()
                dialog.Destroy()
                self.dialogexists = false
                self.ctrl.refocus = true
            return false
        elif self.minctrl:
            minval = self.minctrl.parameter.convert(self.minctrl.GetValue())
            maxval = parameter.convert(val)
            if maxval <= minval:
                if not self.dialogexists:
                    self.dialogexists = true
                    dialog = wxMessageDialog(parent,
                              "%s has to be greater than %s."
                              % (parameter.name, self.minctrl.parameter.name),
                              "Value outside valid range",
                              wxOK | wxICON_ERROR)
                    answer = dialog.ShowModal()
                    dialog.Destroy()
                    self.dialogexists = false
                    self.ctrl.refocus = true
                return false
        elif self.ctrl.IsModified() and not parameter.isusual(val):
            if not self.dialogexists:
                self.dialogexists = true
                dialog = wxMessageDialog(parent,
                          "%s is outside usual range.\n"
                          "Do you really want to use it?"
                          % parameter.name,
                          "Value outside usual range",
                          wxYES_NO | wxICON_WARNING)
                answer = dialog.ShowModal()
                dialog.Destroy()
                self.dialogexists = false
                if answer == wxID_YES:
                    # don't ask again unless the value has changed
                    self.ctrl.DiscardEdits()
                    return true
                else:
                    self.ctrl.refocus = true
            return false
        return true

    def TransferToWindow(self):
        self.ctrl.SetValue(str(self.ctrl.parameter))
        return true

    def TransferFromWindow(self):
        self.ctrl.parameter.value = self.ctrl.GetValue()
        self.ctrl.parameter.comment = self.ctrl._comment
        if isinstance(self.ctrl.parameter, DistParam):
            self.ctrl.parameter.dist.type = self.ctrl.dist_type
        return true

    def OnChar(self, event):
        key = event.GetKeyCode()
        event.Skip()
        return true


class ParameterCtrl(wxTextCtrl):
    def __init__(self, parent, id, parameter, details):
        self.validator = ParameterValidator(self)
        wxTextCtrl.__init__(self, parent, id, validator=self.validator)
        self.parent = parent
        self.parameter = parameter
        self.details = details
        self._comment = parameter.comment
        if isinstance(parameter, DistParam):
            self.dist_type = parameter.dist.type

        self.refocus = false
        EVT_SET_FOCUS(self, self.OnSetFocus)
        EVT_KILL_FOCUS(self, self.OnKillFocus)
        EVT_LEFT_DOWN(self, self.OnLeftDown)
        EVT_IDLE(self, self.OnIdle)

    def OnKillFocus(self, event):
        self.SetSelection(0, 0)
        self.SetInsertionPointEnd()
        if self.IsModified():
            # update "unknown" and warning/error colors
            self.SetValue(self.GetValue())
            # reset IsModified() status to true
            self.Replace(0, self.GetLastPosition(), self.GetValue())
        event.Skip()

    def OnSetFocus(self, event):
        event.Skip()
        self.SetSelection(0, self.GetLastPosition())
        self.SetInsertionPointEnd()
        self.details.connect(self)

    def OnLeftDown(self, event):
        # dirty hack to solve FIXME in OnIdle
        if self.details.parctrl is self:
            event.Skip()
        else:
            self.refocus = true

    def OnIdle(self, event):
        event.Skip()
        if self.refocus:
            #FIXME: set focus to current control doesn't work
            #       if other control selected via mouse click
# references for this bug:
#  http://lists.wxwindows.org/pipermail/wx-users/2000-May/002110.html
#  http://lists.wxwindows.org/pipermail/wx-users/2002-February/017738.html
            self.SetFocus()
            self.refocus = false

    def SetValue(self, value):
        if isinstance(self.parameter, DistParam):
            value = self.parameter.normalize(value+'/'+self.dist_type, value)

        try:
            val = self.parameter.convert(value)
            if not self.parameter.isvalid(val):
                self.SetBackgroundColour(wxNamedColour('orange red'))
            elif not self.parameter.isusual(val):
                self.SetBackgroundColour(wxNamedColour('yellow'))
            else:
                self.SetBackgroundColour(wxNamedColour('white'))
            if isinstance(self.parameter, DistParam):
                if val is None:
                    self.dist_type = Distribution.none
                else:
                    self.dist_type = val[1].type
                self.details.connect_distribution(self)
                value = self.parameter.string(val).split('/')[0]
            else:
                value = self.parameter.string(val)
            wxTextCtrl.SetValue(self, value)
        except ValueError, why:
            self.SetBackgroundColour(wxNamedColour('orange red'))
        self.Refresh()


class ParameterChoiceValidator(wxPyValidator):
    def __init__(self, ctrl):
        wxPyValidator.__init__(self)
        self.ctrl = ctrl
        self.dialogexists = false

    def Clone(self):
        return self.__class__(self.ctrl)

    def Validate(self, parent):
        val = self.ctrl.GetStringSelection()
        if not self.ctrl.parameter.isvalid(val):
            if not self.dialogexists:
                self.dialogexists = true
                dialog = wxMessageDialog(parent,
                          "%s has no valid choice."
                          % self.ctrl.parameter.name,
                          "Invalid choice",
                          wxOK | wxICON_ERROR)
                answer = dialog.ShowModal()
                dialog.Destroy()
                self.dialogexists = false
            return false
        return true

    def TransferToWindow(self):
        self.ctrl.SetStringSelection(str(self.ctrl.parameter))
        return true

    def TransferFromWindow(self):
        self.ctrl.parameter.value = self.ctrl.GetStringSelection()
        self.ctrl.parameter.comment = self.ctrl._comment
        return true


class ParameterLongChoice(wxChoice):
    def __init__(self, parent, id, parameter, details):
        self.validator = ParameterChoiceValidator(self)
        wxChoice.__init__(self, parent, id, choices=parameter.choices)
        if (wxMAJOR_VERSION, wxMINOR_VERSION) < (2, 3):
            print "ParameterChoiceValidator needs wxWindows 2.3.x"
        else:
            self.SetValidator(self.validator)
        self.parent = parent
        self.parameter = parameter
        self.details = details
        self._comment = parameter.comment

        EVT_SET_FOCUS(self, self.OnSetFocus)
        EVT_CHOICE(self, self.GetId(), self.OnChoice)

    def OnSetFocus(self, event):
        event.Skip()
        self.details.connect(self)

    def OnChoice(self, event):
        event.Skip()
        self.details.connect(self)


class ParameterRadioButtonValidator(wxPyValidator):
    def __init__(self, ctrl, radiobutton):
        wxPyValidator.__init__(self)
        self.ctrl = ctrl
        self.radiobutton = radiobutton

    def Clone(self):
        return self.__class__(self.ctrl, self.radiobutton)

    def Validate(self, parent):
        return true

    def TransferToWindow(self):
        if self.radiobutton.GetLabel() == str(self.ctrl.parameter):
            self.radiobutton.SetValue(true)
            self.ctrl.currentbuttonid = self.radiobutton.GetId()
        else:
            self.radiobutton.SetValue(false)
        return true

    def TransferFromWindow(self):
        if self.radiobutton.GetValue():
            self.ctrl.parameter.value = self.radiobutton.GetLabel()
            self.ctrl.parameter.comment = self.ctrl._comment
        return true


class ParameterChoice(wxBoxSizer):
    def __init__(self, parent, id, parameter, details):
        wxBoxSizer.__init__(self, wxHORIZONTAL)
        self.parent = parent
        self.parameter = parameter
        self.details = details
        self._comment = parameter.comment

        self.currentbuttonid = None
        self.buttonbyid = {}
        self.refocus = false
        style = wxRB_GROUP
        for choice in parameter.choices:
            rb_id = wxNewId()
            rb = wxRadioButton(parent, rb_id, choice, style=style)
            self.buttonbyid[rb_id] = rb
            if (wxMAJOR_VERSION, wxMINOR_VERSION) < (2, 3):
                print "ParameterRadioButtonValidator needs wxWindows 2.3.x"
            else:
                rb.SetValidator(ParameterRadioButtonValidator(self, rb))
            style = 0
            EVT_RADIOBUTTON(rb, rb_id, self.OnRadioButton)
            EVT_SET_FOCUS(rb, self.OnSetFocus)
            EVT_IDLE(rb, self.OnIdle)
            self.Add(rb, 0, wxLEFT | wxRIGHT, _SMALLSPACE)

    def OnSetFocus(self, event):
        event.Skip()
        self.details.connect(self)

    def OnIdle(self, event):
        event.Skip()
        if self.refocus and self.currentbuttonid is not None:
            self.buttonbyid[self.currentbuttonid].SetFocus()
            self.refocus = false

    def OnRadioButton(self, event):
        event.Skip()
        self.currentbuttonid = event.GetId()

    def SetValue(self, value):
        for id, button in self.buttonbyid.items():
            if button.GetLabel() == value:
                self.currentbuttonid = id
                button.SetValue(true)
            else:
                button.SetValue(false)


class ParameterGrid(wxFlexGridSizer):
    def __init__(self, parent, details, parameter=[], columns=1):
        wxFlexGridSizer.__init__(self,
                                 0, columns*4,      # rows, cols
                                 _SPACE, _SMALLSPACE) # hgap, vgap
        required_xpm = os.path.join(os.path.dirname(__file__), 'required.xpm')
        self.requiredIcon = wxBitmap(required_xpm, wxBITMAP_TYPE_XPM)
        self.parent = parent
        self.details = details

        # make TextCtrl columns growable
        for col in range(columns):
            self.AddGrowableCol(col*4+2)

        self.controls = []

        for par in parameter:
            if isinstance(par, SciParam):
                self.AddParameter(par)
            else:
                raise TypeError, "%s used with wrong object: %s" % \
                                 (self.__class__, repr(par))

    def AddParameter(self, parameter):
        if isinstance(parameter, ChoiceParam):
            if parameter.long:
                parameterCtrl = ParameterLongChoice(self.parent, -1,
                                                    parameter, self.details)
            else:
                parameterCtrl = ParameterChoice(self.parent, -1,
                                                parameter, self.details)
        else:
            parameterCtrl = ParameterCtrl(self.parent, -1,
                                          parameter, self.details)
        self.controls.append(parameterCtrl)
        self.Add(wxStaticText(self.parent, -1, parameter.name + ':'), 0,
                 wxALIGN_CENTER_VERTICAL | wxALIGN_RIGHT | wxLEFT, _SPACE)
        if parameter.required:
            icon = wxStaticBitmap(self.parent, -1, self.requiredIcon)
            icon.SetToolTip(wxToolTip('required'))
            self.Add(icon, 0, wxALIGN_CENTER_VERTICAL | wxALIGN_RIGHT |
                     wxLEFT, _SMALLSPACE)
        else:
            self.Add(wxStaticText(self.parent, -1, ''), 0)
        #FIXME: Don't want to grow in all directions -> use wxBoxSizers?
        self.Add(parameterCtrl, 0,
                 wxALIGN_CENTER_VERTICAL | wxALIGN_LEFT | wxGROW)
        self.Add(wxStaticText(self.parent, -1, parameter.unit), 0,
                 wxALIGN_CENTER_VERTICAL | wxALIGN_LEFT | wxRIGHT, _SPACE)

    def refocus(self, pos=None):
        if pos is None:
            if self.details.parctrl:
                self.details.parctrl.refocus = true
            elif self.controls:
                self.controls[0].refocus = true
        elif pos < len(self.controls):
            self.controls[pos].refocus = true


class ParameterDialog(wxDialog):
    def __init__(self, parent, id, title, parameter, columns=1,
                 pos=wxDefaultPosition, size=wxDefaultSize,
                 style=wxDEFAULT_DIALOG_STYLE
                       | wxSYSTEM_MENU
                       | wxRESIZE_BORDER):
        wxDialog.__init__(self, parent, id, title,
                          pos=pos, size=size, style=style)

        # box with buttons
        buttonbox = wxBoxSizer(wxHORIZONTAL)
        okButton = wxButton(self, wxID_OK, ' OK ')
        cancelButton = wxButton(self, wxID_CANCEL, ' Cancel ')
        buttonbox.Add(okButton, 0, wxLEFT | wxRIGHT, _SPACE)
        buttonbox.Add(cancelButton, 0, wxLEFT | wxRIGHT, _SPACE)

        # box with details for current parameter
        details = ParameterDetailsBox(self, -1, self)

        # grid of ParameterCtrl
        self.pargrid = ParameterGrid(self, details, parameter, columns)

        # put all together in a vbox
        vbox = wxBoxSizer(wxVERTICAL)
        vbox.Add(self.pargrid, 0, wxALL | wxGROW, _SPACE)
        vbox.Add(details.sizer, 1, wxALL | wxGROW, _SPACE)
        vbox.Add(buttonbox, 0, wxBOTTOM | wxALIGN_RIGHT, _SPACE)

        self.SetAutoLayout(true)
        self.SetSizer(vbox)
        #vbox.Fit(self)
        vbox.SetSizeHints(self)
        self.pargrid.refocus()

    def GetControls(self):
        return self.pargrid.controls

class _TestApp(wxApp):
    def OnInit(self):
        parameter = [
            StringParam('Name', 'a unique identifier', required=1,
                        value='Silicon',
                        comment='This is an example for a required value.\n'
                                'The user can change this comment to note the '
                                'source of this information.'),
            StringParam('Symbol', 'chemical symbol', required=1, value='Si'),
            IntParam('Atomic No',
                     'Number in the periodic table of the elements',
                     value=14, wrange='[1;260]', erange='[1;oo]'),
            FloatParam('Atomic Mass', None, value=28.0855, erange='[0;oo]'),
            FloatParam('Melting Point', 'under normal conditions', 'C',
                       value=1414,
                       wrange='[-273.15;4e3]', erange='[-273.15;oo['),
            FloatParam('Boiling Point', 'under normal conditions', 'C',
                       value=3265, wrange='[-270;6e3]', erange='[-273.15;oo['),
            ChoiceParam('Crystal', 'crystal structure',
                        choices=[None, 'simple cubic', 'face centered cubic',
                                 'body centered cubic', 'diamond cubic',
                                 'tetragonal', 'orthorombic', 'monoclinc'],
                        value='face centered cubic', long=1),
            DistParam('Temperature', 'temperature of samples', 'C',
                      default=20, value=17.5, erange='[-273.15;oo[',
                      dist=(3.7, 'normal')),
            DistParam('Volume', 'Volume of used samples', 'cm',
                      default='2;20/uniform', erange='[0;oo['),
            ChoiceParam('Verified', 'Have these values been checked?',
                        choices=ChoiceParam.yes_no, value=0),
        ]

        dialog = ParameterDialog(NULL, -1, 'Scientific Parameter Dialog',
                                 parameter, columns=2)
        if dialog.ShowModal() == wxID_OK:
            print "OK"
        else:
            print "Cancel"
        for par in parameter:
            print par.name, "=", par.value
            print " comment =", par.comment
            if isinstance(par, DistParam):
                print " dist =", par.dist
        dialog.Destroy()
        return true


def _test():
    try:
        import locale
        locale.setlocale(locale.LC_ALL, "")
    except ImportError:
        # the locale module may not be available on some systems
        pass

    app = _TestApp()
    app.MainLoop()


if __name__ == "__main__":
    import sys
    __file__ = sys.argv[0]
    _test()
