/**

Copyright (C) 1999 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms.tools;

import java.io.*;
import java.awt.geom.*;
import java.util.*;

import pms.*;
import pms.shape.*;
import pms.table.*;

public class ShapeCompactor {
  
  public static void main(String[] args) {
    
    try {

      ShapeCompactor compactor = new ShapeCompactor(args[0], args[1]);
      for (int i = 1; i < args.length; i++) 
        compactor.append(args[i]);
      compactor.close();

    } catch (Exception e) {
      e.printStackTrace();
    }
    System.exit(0);
  }

  protected ShapeCompactor() {}

  public ShapeCompactor(String outName, String templateName) 
    throws IOException {
   
    create(outName, templateName);
  }

  private short readFields(String outName) throws IOException {

    String fieldName = StringTools.setExtension(outName, "fields");
    File fieldFile = new File(fieldName);
    if (!fieldFile.exists()) return 0;

    BufferedReader br = new BufferedReader(new FileReader(fieldFile));

    ArrayList fieldList = new ArrayList();
    String line;
    while ((line = br.readLine()) != null) {
      if (! line.equals(""))
        fieldList.add(line.trim());
    }

    short recordLength = 1;
    columns = new int[fieldList.size()];
    DBaseFile dbFile = checkShape.getTable();
    for (int i = 0; i < columns.length; i++) {
      columns[i] = dbFile.getColumn((String) fieldList.get(i));
      recordLength += dbFile.getColumnLength(columns[i]);
    }

    return recordLength;
  }

  protected void create(String outName, String templateName)
    throws IOException {

    this.outName = outName;
    File outFile = new File(outName + ".shp");
    boolean exists = outFile.exists();

    outShape = new RandomAccessFile(outName + ".shp", "rw");
    outIndex = new RandomAccessFile(outName + ".shx", "rw");
    outTable = new RandomAccessFile(outName + ".dbf", "rw");
    
    checkShape = ShapeFile.create(templateName, false);
    short recordLength = readFields(outName);

    if (! exists) {

      shapeType = checkShape.getShapeType() + ShapeFile.NULL_SHAPE_FLOAT;
      bounds = new Extent(checkShape.getExtent(), "METERS");
      if (columns != null) {
        checkShape.getTable().writeHeader(outTable, columns, recordLength);
      } else {
        checkShape.getTable().writeHeader(outTable);
      }
      checkShape.close();

      writeHeader(outShape, 50);
      outShape.seek(100);
      writeHeader(outIndex, 50);
      outIndex.seek(100);

      checkShape = ShapeFile.create(outName, false);

    } else {

      checkShape = ShapeFile.create(outName, false);
      shapeType = checkShape.getShapeType();
      bounds = new Extent(checkShape.getExtent(), "METERS");
      fileLength = checkShape.getWordLength();
      recordCount = checkShape.getRecordCount();
      outShape.seek(outShape.length());
      outIndex.seek(outIndex.length());
    }

    outTable.seek(outTable.length());
  }

  protected void initAppend(String inName) throws IOException {

    inShape = ShapeFile.create(inName, false);
    inTable = inShape.getTable();
    
    addToBounds(inShape.getExtent());
  }

  public void append(String inName) throws IOException {

    initAppend(inName);
    System.err.println("Adding " + inShape.getRecordCount() + 
      " records from " + inName);

    for (int r = 1; r <= inShape.getRecordCount(); r++) {

      writeRecord(r); 
    }
  }

  protected void writeRecord(int r) throws IOException {

    int wordLength;

    // if (Debug.verbose)
    // System.err.println("Writing at " + outShape.getFilePointer() + " of " +
    // outShape.length());

    if (shapeType == ShapeFile.POINT_FLOAT)
      wordLength = writePoint(r, ++recordCount, outShape);
    else
      wordLength = writeMultiPart(r, ++recordCount, outShape);
    outIndex.writeInt(fileLength);
    outIndex.writeInt(wordLength);
    fileLength += wordLength;
      
    if (columns != null)
      inTable.writeRecord(r, outTable, columns);
    else
      inTable.writeRecord(r, outTable);
  }

  protected void addToBounds(Rectangle2D.Double addBounds) 
    throws IOException {

    if (bounds == null)
      bounds = new Extent(addBounds, "METERS");
    else
      bounds.union(addBounds);
  }
      
  public void close() throws IOException {

    System.err.println("Total records in output file: " + recordCount);
    writeHeader(outShape, fileLength);
    writeHeader(outIndex, 50 + (recordCount * 4));
    outShape.close();
    outIndex.close();

    DBaseFile.setRecordCount(outTable, recordCount);
    outTable.close();
  }

  private void writeHeader(RandomAccessFile file, int length) 
    throws IOException {

    file.seek(0);
    file.writeInt(9995);
    file.seek(24);
    file.writeInt(length);
    file.writeInt(LittleEndian.toInt(ShapeFile.VERSION));
    file.writeInt(LittleEndian.toInt(shapeType));

    file.writeFloat((float) bounds.x);
    double yMin = bounds.y - bounds.height;
    file.writeFloat((float) yMin);
    double xMax = bounds.x + bounds.width;
    file.writeFloat((float) xMax);
    file.writeFloat((float) bounds.y);
  }

  public int writePoint(int inRecord, int outRecord, DataOutput out)
    throws IOException {

    Point2D.Double point = (Point2D.Double) inShape.getShape(inRecord);

    // out.writeInt(outRecord);
    // out.writeInt(4);
    // out.writeInt(ShapeFile.POINT_FLOAT);
    out.writeFloat((float) point.x);
    out.writeFloat((float) point.y);

    return 4;
  }
  
  public int writeMultiPart(int inRecord, int outRecord, DataOutput out)
    throws IOException {
    
    DataInput recordStream = inShape.readShapeRecord(inRecord, 0);
    int shapeType = recordStream.readInt();
    int wordLength = 0;

    //  Do not write header or shape type to conserve space
    //  out.writeInt(outRecord);
    //  out.writeInt(0);
    //  out.writeInt(shapeType + ShapeFile.NULL_SHAPE_FLOAT);

    // For lines do not write bounds to conserve more space

    if (this.shapeType != ShapeFile.ARC_FLOAT) {
      for (int i = 0; i < 4; i++) {
        double d = LittleEndian.toDouble(recordStream.readLong());
        out.writeFloat((float) d);
      }
      wordLength += 8;
    } else {
      recordStream.skipBytes(32);
    }

    int numParts = LittleEndian.toInt(recordStream.readInt());
    // if (Debug.verbose)
    // System.err.println(outShape.getFilePointer()  + " < " + numParts );
    out.writeInt(numParts);
    int numPoints = LittleEndian.toInt(recordStream.readInt());
    out.writeInt(numPoints);
      
    for (int i = 0; i < numParts; i++) {
      int partOffset = LittleEndian.toInt(recordStream.readInt());
      out.writeInt(partOffset);
    }

    for (int i = 0; i < numPoints; i++) {

      float x = (float) LittleEndian.toDouble(recordStream.readLong());
      float y = (float) LittleEndian.toDouble(recordStream.readLong());
      out.writeFloat(x);
      out.writeFloat(y);
    }

    wordLength += 4 + numParts * 2 + numPoints * 4;
    return wordLength;
  }

  private Extent bounds;
  private int shapeType = ShapeFile.NULL_SHAPE;
  private int fileLength = 50;
  protected int recordCount = 0;
  protected int[] columns;
  protected String outName;

  protected ShapeFile inShape;
  protected DBaseFile inTable;
  protected RandomAccessFile outShape;
  protected RandomAccessFile outIndex;
  protected RandomAccessFile outTable;

  protected ShapeFile checkShape;
}
