/**

Copyright (C) 1999 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

/**

This class assembles exported chunks of StreetMap.
In addition to simply appending the shape files, it culls
out duplicate records that cross chunk boundaries and builds
the spatial index.

To build the spatial index, a MemoryTesselation using
the extent and cell size of the entire index area is 
constructed for each chunk.  MemoryTesselations are sparsely
populated so this should not present too great a demand on
memory.  Only the index cells in the chunk area are
then written to the assembled file.  (Shapes that extend
beyond the chunk will be covered in other chunks as well).
If the shape extends to the left or below the chunk, it is
assumed that a record for it has already been written to the shape
file and it is looked up rather than duplicating it.

*/

package pms.tools;

import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.util.*;

import pms.*;
import pms.shape.*;

class DirFilter implements FileFilter {

  public boolean accept(File file) {
    return file.isDirectory();
  }
}

class ExtensionFilter implements FileFilter {

  public ExtensionFilter(String extension) {
    this.extension = extension;
  }

  public boolean accept(File file) {
    return StringTools.getExtension(file.toString()).equals(extension);
  }

  String extension;
}

public class ShapeAssembler extends ShapeCompactor {

  public static void main(String[] args) {
    
    if (args.length != 4) {
      System.err.println(
        "Usage: [output file] [export directory] " +
        "[from metarow] [to metarow]");
      System.exit(0);
    }

    try {

      ShapeAssembler assembler = new ShapeAssembler(args[0], args[1], 
        Integer.parseInt(args[2]), Integer.parseInt(args[3]));
      assembler.assemble();
      assembler.close();

    } catch (Exception e) {
      e.printStackTrace();
    }
    System.exit(0);
  }

  public ShapeAssembler(String outFileName, String inDirName, 
    int from, int to) throws IOException {

    this.from = from;
    this.to = to;

    File outFile = new File(StringTools.setExtension(outFileName, "shp"));
    // if ((from > 1) && (! outFile.exists()))
      // throw new IOException("Existing shape file " + outFile + 
        // " not found.  Assembly must begin at metarow one.");

    inDir = new File(inDirName);
    if (! inDir.isDirectory())
      throw new IOException(inDir + " is not a directory");

    File rowDir = new File(inDir, Integer.toString(from));
    File[] cellFiles = rowDir.listFiles(shpFilter);
    String templateName = cellFiles[0].toString();

    create(outFileName, templateName);

    File outTessIndexFile = new File(outName + ".tsx");

    if (! outTessIndexFile.exists()) {
      createIndex(outTessIndexFile, inDir);
    }

    outTess = new FileTesselation(outName, "rw");
    cols = outTess.getColumns();

    outTessIndex = outTess.getIndexFile();
    outTessMain = outTess.getMainFile();
    offset = (int) outTessMain.length();
  }

  private void createIndex(File tessIndexFile, File inDir) 
    throws IOException {

    File paramFile = new File(inDir, "index.params");
    if (! paramFile.exists())
      throw new IOException("Parameters file " + paramFile + " not found");

    BufferedReader br = new BufferedReader(new FileReader(paramFile));

    double x = Double.parseDouble(br.readLine());
    double y = Double.parseDouble(br.readLine());
    double width = Double.parseDouble(br.readLine());
    double height = Double.parseDouble(br.readLine());
    double cellSize = Double.parseDouble(br.readLine());

    int rows = Integer.parseInt(br.readLine());
    if (to > rows)
      throw new IOException(
        "Last chunk row is beyond the number of chunk rows in the index");

    int cols = Integer.parseInt(br.readLine());

    RandomAccessFile outTessIndex = 
      new RandomAccessFile(tessIndexFile, "rw");
    outTessIndex.writeDouble(x);
    outTessIndex.writeDouble(y);
    outTessIndex.writeDouble(width);
    outTessIndex.writeDouble(height);
    outTessIndex.writeDouble(cellSize);
    outTessIndex.writeInt(rows);
    outTessIndex.writeInt(cols);

    outTessIndex.close();
  }

  public void assemble() throws IOException {

    for (int i = from; i <= to; i++) {
      for (int j = 1; j <= cols; j++)
        addChunk(j, i);
    }
  }

  private void addChunk(int col, int row) throws IOException {

    File rowDir = new File(inDir, Integer.toString(row));
    File shpFile = new File(rowDir, col + ".shp");
    if (! shpFile.exists()) return;

    if (Debug.verbose)
      System.err.println("Adding " + shpFile);

    initAppend(shpFile.toString());
    inTess = outTess.create();

    for (int r = 1; r <= inShape.getRecordCount(); r++)
      inTess.add(r, inShape.getBounds(r));

    for (int i = 0; i < 10; i++) {
      for (int j = 0; j < 10; j++) {

        int outCol = col * 10 + j;
        int outRow = row * 10 + i;

        int[] cell = inTess.getCell(outCol, outRow);
        writeOffset(outCol, outRow, cell.length);

        for (int k = 0; k < cell.length; k++)
          addRecord(cell[k], col, row);

        // if (Debug.verbose) {
        //  System.err.println("Wrote for " + col + "," + row);
        //  System.err.println("  At " + outTessMain.length());
        // }
      }
    }
    inShape.close();
  }

  private void addRecord(int inRecord, int col, int row) 
    throws IOException {

    int outRecord;

    Rectangle2D.Double shapeBounds = inShape.getBounds(inRecord);
    Point cell = outTess.pointToTesselation(
      shapeBounds.x, shapeBounds.y - shapeBounds.height);

    // if (Debug.verbose)
    // System.err.println("Record origin cell: " + cell.x + "," + cell.y);

    if ((cell.x >= col * 10) && (cell.y >= row * 10)) {

      writeRecord(inRecord);
      outRecord = recordCount;
      // if (Debug.verbose)
      // System.err.println("Wrote record " + inRecord + " as " + outRecord);

    } else {
      
      outRecord = matchRecord(cell, shapeBounds);

      if (outRecord == 0) {
        // throw new IOException("Failed to match record");
        // System.err.println("Failed to match record");
        // System.err.println(shapeBounds);
        writeRecord(inRecord);
        outRecord = recordCount;
      } 
      // else if (Debug.verbose)
      // System.err.println("Matched record " + inRecord + " to " + outRecord);
    }

    outTessMain.seek(outTessMain.length());
    outTessMain.writeInt(outRecord);
  }

  private int matchRecord(Point cell, Rectangle2D.Double matchBounds) 
    throws IOException {

    int[] cellRecords = outTess.getCell(cell.x, cell.y);

    // if (Debug.verbose)
    //  System.err.println("Matching " + matchBounds + " to...");
    for (int i = 0; i < cellRecords.length; i++) {

      Rectangle2D.Double shapeBounds = checkShape.getBounds(
        cellRecords[i]);
      // if (Debug.verbose)
      //   System.err.println(shapeBounds);

      if (extentsEqual(matchBounds, shapeBounds))
        return cellRecords[i];
    }

    return 0;
  }

  private boolean extentsEqual(Rectangle2D.Double matchBounds, 
    Rectangle2D.Double shapeBounds) {

    return (Math.abs(matchBounds.x - shapeBounds.x) < 1) &&
           (Math.abs(matchBounds.y - shapeBounds.y) < 1) &&
           (Math.abs(matchBounds.width - shapeBounds.width) < 1) &&
           (Math.abs(matchBounds.height - shapeBounds.height) < 1);
  }

  private void writeOffset(int col, int row, int cellLength) 
    throws IOException {
    
    int pos = FileTesselation.INDEX_HEADER_LENGTH + 
      ((row * cols) + col) * 8;
    outTessIndex.seek(pos);
    outTessIndex.writeInt((int) outTessMain.length());
    outTessIndex.writeInt(cellLength);

    // if (Debug.verbose) {
    // System.err.println("Wrote " + cellLength + " for " + col + "," + row);
    // System.err.println("  At " + outTessMain.length());
    // }
  }

  private int cols;

  private int from;
  private int to;

  private int offset;

  private File inDir;
  private MemoryTesselation inTess;
  private FileTesselation outTess;
  private RandomAccessFile outTessIndex;
  private RandomAccessFile outTessMain;

  protected static ExtensionFilter shpFilter = new ExtensionFilter("shp");
}
