/**

Copyright (C) 1999 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms.tools;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.net.*;

public class ImageLoader implements ImageObserver {

  private int flags = 0;
  private static Toolkit toolkit = Toolkit.getDefaultToolkit();

  public Image createImage(URL url) throws IOException {

    Debug.assertNotNull("ImageLoader.createImage", "url", url);
    
    File file = new File(url.getFile());
    if (! file.exists()) {
      throw new IOException("Image file " + file + " not found.");
    }
    
    Image image = toolkit.createImage(url);

    if (image != null) 
      waitForCompletion(image, -1, -1);

    return image;
  }

  public Image createImage(String path) throws IOException {

    Debug.assertNotNull("ImageLoader.createImage", "path", path);
    
    File file = new File(path);
    if (! file.exists()) {
      throw new IOException("Image file " + path + " not found.");
    }

    Image image = toolkit.createImage(path);

    if (image != null) 
      waitForCompletion(image, -1, -1);

    return image;
  }

  public synchronized boolean waitForCompletion(
    Image image, int width, int height) {

    toolkit.prepareImage(image, width, height, this);

    // check for images prepared synchronously before blocking
    if (checkForCompletion(image, width, height))
      return ((flags & ImageObserver.ALLBITS) != 0) ? true : false;
 
    try {
      wait();
    } catch (InterruptedException e) {
      System.err.println(e);
    }

    return ((flags & ImageObserver.ALLBITS) != 0) ? true : false;
  }

  private boolean checkForCompletion(
    Image image, int width, int height) {

    flags = toolkit.checkImage(image, -1, -1, this);
    boolean ALLBITS = ((flags & ImageObserver.ALLBITS) != 0);
    boolean ERROR = ((flags & ImageObserver.ERROR) != 0);
    return (ALLBITS || ERROR) ? true : false;
  }

  public synchronized boolean imageUpdate(
    Image image, int flags, int x, int y, int width, int height) {

    this.flags = flags;

    if ((flags & ImageObserver.ALLBITS) != 0) 
      notifyAll();
    if ((flags & ImageObserver.ERROR) != 0) 
      notifyAll();

    return true;
  }
}
