package pms.table;

import java.io.*;
import java.util.*;

import pms.tools.*;

public class DBaseFile implements AttributeSource {

  public void open(String path, RandomAccessFile file, long beginOffset) 
    throws IOException {

    this.path = StringTools.setExtension(path, "dbf");
    this.file = file;
    this.beginOffset = beginOffset;
    readHeader();
  }

  public void open(String path) throws IOException {

    this.path = StringTools.setExtension(path, "dbf");
    file = new RandomAccessFile(this.path, "r");
    beginOffset = 0;
    readHeader();
  }

  private void readHeader() throws IOException {

    file.seek(beginOffset);
    version = file.read();
    file.read(lastUpdate);
    recordCount = LittleEndian.toInt(file.readInt());
    headerLength = LittleEndian.toShort(file.readShort());

    recordLength = LittleEndian.toShort(file.readShort());

    int fieldCount = (headerLength - 33) / 32;
    int offset = 1;
    fields = new DBaseField[fieldCount]; 
    for (int i = 0; i < fieldCount; i++) {

      fields[i] = new DBaseField();
      fields[i].readHeader(file, beginOffset, i + 1);
      fields[i].offset = offset;
      offset += fields[i].length;
    }
  }

  public void writeHeader(DataOutput out) throws IOException {

    file.seek(beginOffset);
    byte[] header = new byte[headerLength];
    file.readFully(header);
    out.write(header);
  }

  public void writeHeader(RandomAccessFile out, int[] columns, 
    short recLength) throws IOException {

    file.seek(beginOffset);
    byte[] header = new byte[32];
    file.readFully(header);

    out.seek(0);
    out.write(header);

    short headerLength = (short) (33 + (columns.length * 32));
    out.seek(8);
    out.writeShort(LittleEndian.toShort(headerLength));
    out.writeShort(LittleEndian.toShort(recLength));
    out.seek(32);
    
    for (int i = 0; i < columns.length; i++) {
      int column = columns[i];
      file.seek(32 + (column - 1) * 32);
      file.readFully(header);
      out.write(header);
    }

    out.write(0x0D);
  }

  public int getRecordCount() {

    return recordCount;
  }

  public static void setRecordCount(RandomAccessFile file, int recordCount) 
    throws IOException {

    file.seek(4);
    file.writeInt(LittleEndian.toInt(recordCount));
  }

  public void writeRecord(int record, DataOutput out) throws IOException {

    long pos = headerLength + recordLength * (record - 1);
    byte[] bytes = new byte[recordLength];
    file.seek(beginOffset + pos);
    file.read(bytes);
    out.write(bytes);
  }

  public void writeRecord(int record, DataOutput out, int[] columns) 
    throws IOException {

    long pos = headerLength + recordLength * (record - 1);
    byte[] bytes = new byte[recordLength];
    file.seek(beginOffset + pos);
    file.read(bytes);

    out.write(' ');
    for (int i = 0; i < columns.length; i++) {
      int column = columns[i];
      out.write(bytes, fields[column - 1].offset, fields[column - 1].length);
    }
  }

  public int getColumnType(int column) {

    return fields[column - 1].type;
  }

  public int getColumnLength(int column) {

    return fields[column - 1].length;
  }
 
  public int getColumnCount() {
    return fields.length;
  }

  public void checkColumn(int column) throws IOException {

    if ((column < 1) || (column > fields.length))
      throw new IOException(
        "Invalid column " + column + " (1 - " + fields.length + ")"); 
  }

  public String getColumnName(int i) throws IOException {

    checkColumn(i);
    return fields[i - 1].name;
  }

  public int getColumn(String fieldName) throws IOException {

    int column = 0;

    if (fieldName == null) return 0;
    fieldName = fieldName.toUpperCase();
    for (int i = 0; i < fields.length; i++) {
      if (fields[i].name.equals(fieldName)) {
        column = i + 1;
        break;
      }
    }

    if (column == 0)
      throw new IOException("Field " + fieldName + " not found in " + path);

    return column;
  }

  public Attribute getAttribute(String field) throws IOException {

    int column = getColumn(field);
    return new Attribute(this, column);
  }

  public void setRecords(ArrayList selection) {}

  public void setColumns(ArrayList selection) throws IOException {

    selectedColumns = new int[selection.size()];
    for (int i = 0; i < selectedColumns.length; i++) {

      selectedColumns[i] = getColumn((String) selection.get(i));
    }
  }

  public String[] getRecord(int row) throws IOException {

    if (selectedColumns == null)
      throw new IOException("Get record: No columns selected");

    long pos = 
      headerLength + recordLength * (row - 1);
    byte[] bytes = new byte[recordLength];
    file.seek(beginOffset + pos);
    file.read(bytes);
    
    String[] record = new String[selectedColumns.length];

    for (int i = 0; i < record.length; i++) {
      int c = selectedColumns[i] - 1;
      record[i] = 
        new String(bytes, fields[c].offset, fields[c].length).trim();
    }

    return record;
  }

  public String getValue(int record, int column) throws IOException {

    checkColumn(column);
    column--;
    long pos = headerLength + 
      ((long) recordLength) * (record - 1) + fields[column].offset;

    byte[] bytes = new byte[fields[column].length];
    file.seek(beginOffset + pos);
    file.read(bytes);
    
    return (new String(bytes, 0, fields[column].length)).trim();
  }

  public void close() throws IOException {

    if (file != null)
      file.close();
  }

  public static void main(String[] args) {

    try {
      for (int i = 0; i < args.length; i++) {
	DBaseFile file = new DBaseFile();
	file.open(args[i]);
	int lastRec = file.getRecordCount();
	int lastCol = file.getColumnCount() - 1;
	String value = file.getValue(lastRec, lastCol);
	System.err.println("Read last value successfully.");
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  private String path;
  private RandomAccessFile file;
  private long beginOffset;
  private int version;
  private byte[] lastUpdate = new byte[3];
  private int recordCount;
  private short headerLength;
  private short recordLength;

  private int[] selectedColumns = null;

  private DBaseField[] fields;
}

class DBaseField {

  public void readHeader(RandomAccessFile file, long beginOffset, int column) 
    throws IOException {

    file.seek(beginOffset + (32 * column));
    byte[] nameBytes = new byte[11];
    file.read(nameBytes);
    name = (new String(nameBytes, 0, 11)).toUpperCase().trim();

    int typeInt = file.readUnsignedByte(); 
    type = (char) typeInt;

    file.skipBytes(4);
    length = file.readUnsignedByte();
    decimal = file.readUnsignedByte();
  }

  public String name;
  public char type;
  public int length;
  public int decimal;
  public int offset;
}
